package net.mehvahdjukaar.moonlight.api.resources.pack;

import ;
import com.google.common.base.Preconditions;
import com.google.common.base.Suppliers;
import com.google.gson.JsonElement;
import dev.architectury.injectables.annotations.PlatformOnly;
import net.mehvahdjukaar.moonlight.api.integration.ModernFixCompat;
import net.mehvahdjukaar.moonlight.api.misc.PathTrie;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.api.resources.RPUtils;
import net.mehvahdjukaar.moonlight.api.resources.ResType;
import net.mehvahdjukaar.moonlight.api.resources.SimpleTagBuilder;
import net.mehvahdjukaar.moonlight.api.resources.StaticResource;
import net.mehvahdjukaar.moonlight.api.resources.assets.LangBuilder;
import net.mehvahdjukaar.moonlight.core.CommonConfigs;
import net.mehvahdjukaar.moonlight.core.CompatHandler;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.SharedConstants;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.packs.PackResources;
import net.minecraft.server.packs.PackType;
import net.minecraft.server.packs.metadata.MetadataSectionSerializer;
import net.minecraft.server.packs.metadata.pack.PackMetadataSection;
import net.minecraft.server.packs.metadata.pack.PackMetadataSectionSerializer;
import net.minecraft.server.packs.repository.Pack;
import net.minecraft.server.packs.repository.PackSource;
import net.minecraft.server.packs.resources.IoSupplier;
import net.minecraft.world.flag.FeatureFlagSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

public abstract class DynamicResourcePack implements PackResources {

    protected static final Logger LOGGER = LogManager.getLogger();

    protected final boolean hidden;
    protected final boolean fixed;
    protected final Pack.Position position;
    protected final PackType packType;
    protected final Supplier<PackMetadataSection> metadata;
    protected final Component title;
    protected final ResourceLocation resourcePackName;
    protected final Set<String> namespaces = new HashSet<>();
    protected final Map<ResourceLocation, byte[]> resources = new ConcurrentHashMap<>();
    protected final ResourceLocPathTrie searchTrie = new ResourceLocPathTrie();
    protected final Map<String, byte[]> rootResources = new ConcurrentHashMap<>();
    protected final String mainNamespace;

    protected Set<ResourceLocation> staticResources = new HashSet<>();

    //for debug or to generate assets
    protected boolean generateDebugResources;
    private boolean needsClearingNonStatic = false;
    boolean addToStatic = false;
    private boolean wasRegistered = false;

    protected DynamicResourcePack(ResourceLocation name, PackType type) {
        this(name, type, Pack.Position.TOP, false, false);
    }

    protected DynamicResourcePack(ResourceLocation name, PackType type, Pack.Position position, boolean fixed, boolean hidden) {
        this.packType = type;
        this.resourcePackName = name;
        this.mainNamespace = name.m_135827_();
        this.namespaces.add(mainNamespace);
        this.title = Component.m_237115_(LangBuilder.getReadableName(name.toString()));

        this.position = position;
        this.fixed = fixed;
        this.hidden = hidden; //UNUSED. TODO: re add (forge)
        this.metadata = Suppliers.memoize(() -> new PackMetadataSection(this.makeDescription(),
                SharedConstants.m_183709_().m_264084_(type)));
        this.generateDebugResources = PlatHelper.isDev();
    }

    public Component makeDescription() {
        return Component.m_237115_(LangBuilder.getReadableName(mainNamespace + "_dynamic_resources"));
    }

    @Deprecated(forRemoval = true)
    public void setClearOnReload(boolean canBeCleared) {
    }

    @Deprecated(forRemoval = true)
    public void clearOnReload(boolean canBeCleared) {
    }

    public void markNotClearable(ResourceLocation staticResources) {
        this.staticResources.add(staticResources);
    }

    public void unMarkNotClearable(ResourceLocation staticResources) {
        this.staticResources.remove(staticResources);
    }

    public void setGenerateDebugResources(boolean generateDebugResources) {
        this.generateDebugResources = generateDebugResources;
    }

    /**
     * Dynamic textures are loaded after getNamespaces is called, so unfortunately we need to know those in advance
     * Call this if you are adding stuff for another mod namespace
     **/
    public void addNamespaces(String... namespaces) {
        this.namespaces.addAll(Arrays.asList(namespaces));
    }

    public Component getTitle() {
        return this.title;
    }

    @Override
    public String m_5542_() {
        return title.getString();
    }

    public ResourceLocation id() {
        return resourcePackName;
    }

    @Override
    public String toString() {
        return m_5542_();
    }

    /**
     * Registers this pack. Call on mod init
     */
    protected void registerPack() {
        if (wasRegistered) {
            return;
        } else {
            wasRegistered = true;
        }
        PlatHelper.registerResourcePack(this.packType, () ->
                Pack.m_245512_(
                        this.m_5542_(),    // id
                        this.getTitle(), // title
                        true,    // required -- this MAY need to be true for the pack to be enabled by default
                        (s) -> this, // pack supplier
                        new Pack.Info(metadata.get().m_10373_(), metadata.get().m_10374_(), FeatureFlagSet.m_246902_()), // description
                        this.packType,
                        Pack.Position.TOP,
                        this.fixed, // fixed position? no
                        PackSource.f_10528_));

    }

    //@Override
    @PlatformOnly(PlatformOnly.FORGE)
    public boolean isHidden() {
        return this.hidden;
    }

    @Override
    public Set<String> m_5698_(PackType packType) {
        return this.namespaces;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T m_5550_(MetadataSectionSerializer<T> serializer) {
        return serializer instanceof PackMetadataSectionSerializer ? (T) this.metadata : null;
    }

    public void addRootResource(String name, byte[] resource) {
        this.rootResources.put(name, resource);
    }

    @Nullable
    @Override
    public IoSupplier<InputStream> m_8017_(String... strings) {
        String fileName = String.join("/", strings);
        byte[] resource = this.rootResources.get(fileName);
        return resource == null ? null : () -> new ByteArrayInputStream(resource);
    }


    @Override
    public void m_8031_(PackType packType, String namespace, String id, ResourceOutput output) {

        if (PlatHelper.isDev()) {
            //validate
            for (var r : this.searchTrie.search("")) {
                if (!resources.containsKey(r)) {
                    Moonlight.LOGGER.error("Resource {} not ", r);
                    throw new IllegalStateException("Resource " + r + " not found in resources");
                }
            }
        }
        synchronized (this) {
            //why are we only using server resources here?
            if (packType == this.packType) {
                //idk why but somebody had an issue with concurrency here during world load

                this.searchTrie.search(namespace + "/" + id)
                        .forEach(r -> {
                            byte[] buf = resources.get(r);
                            if (buf == null) {
                                throw new IllegalStateException("Somehow search trie returned a resource not in resources " + r);
                            }
                            output.accept(r, () -> new ByteArrayInputStream(buf));
                        });
            }
        }
    }

    @Override
    public IoSupplier<InputStream> m_214146_(PackType type, ResourceLocation id) {

        var res = this.resources.get(id);
        if (res != null) {
            return () -> {
                if (type != this.packType) {
                    throw new IOException(String.format("Tried to access wrong type of resource on %s.", this.resourcePackName));
                }
                return new ByteArrayInputStream(res);
            };
        }
        return null;
    }

    @Override
    public void close() {
        // do not clear after reloading texture packs. should always be on
    }

    public FileNotFoundException makeFileNotFoundException(String path) {
        return new FileNotFoundException(String.format("'%s' in ResourcePack '%s'", path, this.resourcePackName));
    }

    protected void addBytes(ResourceLocation id, byte[] bytes) {
        this.namespaces.add(id.m_135827_());
        this.resources.put(id, Preconditions.checkNotNull(bytes));
        this.searchTrie.insert(id);
        if (addToStatic) markNotClearable(id);
        //debug
        if (generateDebugResources) {
            saveBytes(id, bytes);
        }
    }

    public static void saveJson(ResourceLocation id, JsonElement json) {
        try {
            saveBytes(id, RPUtils.serializeJson(json).getBytes());
        } catch (IOException e) {
            LOGGER.error("Failed to deserialize JSON {}", json, e);
        }
    }

    public static void saveBytes(ResourceLocation id, byte[] bytes) {
        try {
            Path p = Paths.get("debug", "generated_resource_pack").resolve(id.m_135827_() + "/" + id.m_135815_());
            Files.createDirectories(p.getParent());
            Files.write(p, bytes, StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING);
        } catch (IOException ignored) {
        }
    }

    @Deprecated(forRemoval = true)
    public void removeResource(ResourceLocation res) {
        synchronized (this) {
            this.searchTrie.remove(res);
            this.resources.remove(res);
            this.staticResources.remove(res);
        }
    }

    @Deprecated(forRemoval = true)
    public void addResource(StaticResource resource) {
        this.addBytes(resource.location, resource.data);
    }

    @Deprecated(forRemoval = true)
    private void addJson(ResourceLocation path, JsonElement json) {
        try {
            this.addBytes(path, RPUtils.serializeJson(json).getBytes());
        } catch (IOException e) {
            LOGGER.error("Failed to write JSON {} to resource pack {}.", path, this.resourcePackName, e);
        }
    }

    @Deprecated(forRemoval = true)
    public void addJson(ResourceLocation location, JsonElement json, ResType resType) {
        this.addJson(resType.getPath(location), json);
    }

    @Deprecated(forRemoval = true)
    public void addBytes(ResourceLocation location, byte[] bytes, ResType resType) {
        this.addBytes(resType.getPath(location), bytes);
    }


    public PackType getPackType() {
        return packType;
    }

    // Called after texture have been stitched. Only keeps needed stuff
    @ApiStatus.Internal
    protected void clearNonStatic() {
        if (!CommonConfigs.CLEAR_RESOURCES.get()) return;
        if (this.needsClearingNonStatic) {
            this.needsClearingNonStatic = false;
            boolean mf = MODERN_FIX && getPackType() == PackType.CLIENT_RESOURCES;
            // clear trie entirely and re populate as we always expect to have way less staitc resources than others
            if (!mf) this.searchTrie.clear();

            for (var r : this.resources.keySet()) {
                if (mf && modernFixHack(r.m_135815_())) {
                    continue;
                }
                if (!this.staticResources.contains(r)) {
                    this.resources.remove(r);
                }
            }


            if (mf) {
                List<String> toRemove = new ArrayList<>();
                for (String namespace : this.searchTrie.listFolders("")) {
                    for (String f : this.searchTrie.listFolders(namespace)) {
                        if (!modernFixHack(f)) {
                            toRemove.add(namespace + "/" + f);
                        }
                    }
                }
                toRemove.forEach(this.searchTrie::remove);
            }
            // rebuild search trie with just static
            for (var s : staticResources) {
                this.searchTrie.insert(s);
            }
        }
    }

    // Called after each reload
    @ApiStatus.Internal
    protected void clearAllContent() {
        synchronized (this) {
            this.searchTrie.clear();
            this.resources.clear();
            this.needsClearingNonStatic = true; //clear non static after dynamic textures have been stitched
        }
    }


    private static final boolean MODERN_FIX = CompatHandler.MODERNFIX && ModernFixCompat.areLazyResourcesOn();

    private boolean modernFixHack(String s) {
        return s.startsWith("model") || s.startsWith("blockstate");
    }

    public void addTag(SimpleTagBuilder tag, ResourceKey<?> key) {
    }

    protected static class ResourceLocPathTrie extends PathTrie<ResourceLocation> {

        public boolean remove(ResourceLocation object) {
            //remove last bit as that's the object
            String path = object.m_135827_() + '/' + object.m_135815_().substring
                    (0, object.m_135815_().lastIndexOf('/'));
            return super.remove(path);
        }

        public void insert(ResourceLocation object) {
            String path = object.m_135827_() + '/' + object.m_135815_().substring
                    (0, object.m_135815_().lastIndexOf('/'));
            super.insert(path, object);
        }

    }
}