package net.mehvahdjukaar.moonlight.api.set.wood;

import com.google.common.base.Preconditions;
import com.google.common.base.Suppliers;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.api.set.BlockType;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.mehvahdjukaar.moonlight.core.CompatHandler;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.core.Registry;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.item.AxeItem;
import net.minecraft.world.item.Item;
import net.minecraft.world.level.ItemLike;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.CeilingHangingSignBlock;
import net.minecraft.world.level.block.SignBlock;
import net.minecraft.world.level.block.state.BlockBehaviour;
import net.minecraft.world.level.block.state.BlockBehaviour.Properties;
import net.minecraft.world.level.material.MapColor;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Stream;

import static net.mehvahdjukaar.moonlight.api.set.wood.VanillaWoodChildKeys.*;

/**
 * CHILD AVAILABLITY:
 * <p>BLOCK:</p>
 * <ul>
 * planks, log, stripped_log, wood, stripped_wood, leaves,
 * slab, stairs, fence, fence_gate, door, trapdoor,
 * button, pressure_plate, hanging_sign, wall_hanging_sign, sign, wall_sign
 * </ul>
 * <p>ITEM:</p>
 * <ul>
 * boat, chest_boat, sapling
 * </ul>
 */
public class WoodType extends BlockType {

    public static final Codec<WoodType> CODEC = ResourceLocation.f_135803_.flatXmap(r -> {
                WoodType w = WoodTypeRegistry.INSTANCE.get(r);
                if (w == null) return DataResult.error(() -> "No such wood type: " + r);
                return DataResult.success(w);
            },
            t -> DataResult.success(t.id));

    public final Block planks;
    public final Block log;

    // like this so it can be called early. not too early tho as children might not be initialized
    //mega ugly. i cant initialize it immediately as mods might have not run setup yet
    private final Supplier<net.minecraft.world.level.block.state.properties.WoodType> vanillaType = Suppliers.memoize(this::detectVanillaWood);

    @Nullable
    private net.minecraft.world.level.block.state.properties.WoodType detectVanillaWood() {
        if (getChild("hanging_sign") instanceof CeilingHangingSignBlock c) {
            return c.m_56297_();
        }
        if (getChild("sign") instanceof SignBlock f) {
            return f.m_56297_();
        }
        String i = id.m_135827_().equals("minecraft") ? id.m_135815_() : id.toString();
        var values = net.minecraft.world.level.block.state.properties.WoodType.m_61843_();
        var o = values.filter(v -> v.f_61839_().equals(i)).findAny();
        return o.orElse(null);
    }

    public WoodType(ResourceLocation id, Block baseBlock, Block logBlock) {
        super(id);
        this.planks = baseBlock;
        this.log = logBlock;
    }

    @Override
    public ItemLike mainChild() {
        return planks;
    }

    @Nullable
    public net.minecraft.world.level.block.state.properties.WoodType toVanilla() {
        return this.vanillaType.get();
    }

    @NotNull
    public net.minecraft.world.level.block.state.properties.WoodType toVanillaOrOak() {
        var v = toVanilla();
        if (v != null) return v;
        return net.minecraft.world.level.block.state.properties.WoodType.f_61830_;
    }

    /**
     * Use this to get the texture path of a wood type
     *
     * @return something like minecraft/oak
     */
    public String getTexturePath() {
        String namespace = this.getNamespace();
        if (namespace.equals("minecraft")) return this.getTypeName();
        return this.getNamespace() + "/" + this.getTypeName();
    }

    public boolean canBurn() {
        return this.planks.m_49966_().m_278200_();
    }

    public MapColor getColor() {
        return this.planks.m_284356_();
    }

    @Override
    public String getTranslationKey() {
        return "wood_type." + this.getNamespace() + "." + this.getTypeName();
    }

    @Override
    public void initializeChildrenBlocks() {
        this.addChild(PLANKS, this.planks);
        this.addChild(LOG, this.log);
        this.addChild(LEAVES, this.findRelatedEntry("leaves", BuiltInRegistries.f_256975_));
        this.addChild(WOOD, this.findLogRelatedBlock("", "wood", "hyphae", "bark"));
        this.addChild(STRIPPED_LOG, this.findStrippedLog(LOG, "stem", "stalk"));
        this.addChild(STRIPPED_WOOD, this.findStrippedLog(WOOD, "hyphae", "bark"));
        this.addChild(SLAB, this.findRelatedEntry("slab", BuiltInRegistries.f_256975_));
        this.addChild(STAIRS, this.findRelatedEntry("stairs", BuiltInRegistries.f_256975_));
        Block fence = this.findRelatedEntry("fence", BuiltInRegistries.f_256975_);
        this.addChild(FENCE, fence);
        this.addChild(FENCE_GATE, this.findRelatedEntry("fence_gate", BuiltInRegistries.f_256975_));
        this.addChild(DOOR, this.findRelatedEntry("door", BuiltInRegistries.f_256975_));
        this.addChild(TRAPDOOR, this.findRelatedEntry("trapdoor", BuiltInRegistries.f_256975_));
        this.addChild(BUTTON, this.findRelatedEntry("button", BuiltInRegistries.f_256975_));
        this.addChild(PRESSURE_PLATE, this.findRelatedEntry("pressure_plate", BuiltInRegistries.f_256975_));
        this.addChild(HANGING_SIGN, this.findRelatedEntry("hanging_sign", BuiltInRegistries.f_256975_));
        this.addChild(WALL_HANGING_SIGN, this.findRelatedEntry("wall_hanging_sign", BuiltInRegistries.f_256975_));
        this.addChild(SIGN, this.findRelatedEntry("sign", BuiltInRegistries.f_256975_));
        this.addChild(WALL_SIGN, this.findRelatedEntry("wall_sign", BuiltInRegistries.f_256975_));

        if (this.id.m_135827_().matches("tfc|afc")) { // Including unidue blocks' path
            this.addChild(SIGN, this.findRelatedEntry("sign", "", BuiltInRegistries.f_256975_));
            this.addChild(HANGING_SIGN, this.findRelatedEntry("hanging_sign/wrought_sign", "", BuiltInRegistries.f_256975_));
        }

        if (fence != null && CompatHandler.DIAGONALFENCES) {
            var diagonalFence = BuiltInRegistries.f_256975_.m_6612_(
                    new ResourceLocation("diagonalfences", Utils.getID(fence)
                            .toString().replace(":", "/")));
            diagonalFence.ifPresent(block -> this.addChild("diagonalfences:fence", block));
        }
    }

    @Override
    public void initializeChildrenItems() {
        this.addChild(BOAT, this.findRelatedItem("boat", "raft"));
        this.addChild(CHEST_BOAT, this.findRelatedItem("chest_boat", "chest_raft"));
        this.addChild(SAPLING, this.findRelatedEntry("sapling", BuiltInRegistries.f_257033_));
        if (this.id.m_135827_().matches("tfc|afc")) { // Including unidue blocks' path
            this.addChild(STICK, this.findRelatedEntry("twig", BuiltInRegistries.f_256975_));
            this.addChild(BOAT, this.findRelatedEntry("boat", "", BuiltInRegistries.f_256975_));
        }
    }


    @Nullable
    protected <V> V findRelatedEntry(String prefixOrInfix, String suffix, Registry<V> reg) {
        if (!suffix.isEmpty()) suffix = "_" + suffix;
        ResourceLocation[] targets = {
                new ResourceLocation(id.m_135827_(), id.m_135815_() + "_" + prefixOrInfix + suffix),
                new ResourceLocation(id.m_135827_(), prefixOrInfix + "_" + id.m_135815_() + suffix),
                //weird conventions here
                new ResourceLocation(id.m_135827_(), id.m_135815_() + "_planks_" + prefixOrInfix + suffix),
                // TFC & AFC: Include children of wood_type: stairs, slab...
                new ResourceLocation(id.m_135827_(), "wood/planks/" + id.m_135815_() + "_" + prefixOrInfix),
                // TFC & AFC: Include twig (sticks), leaves, planks, sign
                new ResourceLocation(id.m_135827_(), "wood/" + prefixOrInfix + suffix + "/" + id.m_135815_())
        };
        return Utils.findFirstInRegistry(reg, targets);
    }


    @Nullable
    protected Block findStrippedLog(String... possibleNames) {
        for (var v : possibleNames) {
            Block b = this.getBlockOfThis(v);
            if (v != null) {
                Block stripped = AxeItem.f_150683_.get(b);
                if (stripped != null && stripped != b) {
                    return stripped;
                }
            }
        }
        return findLogRelatedBlock("stripped", possibleNames);
    }

    @Nullable
    protected Item findRelatedItem(String... names) {
        for (var n : names) {
            var b = findRelatedEntry(n, BuiltInRegistries.f_257033_);
            if (b != null) return b;
        }
        return null;
    }

    @Nullable
    protected Block findLogRelatedBlock(String prefix, String... possibleSuffix) {
        for (var n : possibleSuffix) {
            Block b = findLogWithAffix(prefix, n);
            if (b != null) return b;
        }
        return null;
    }

    @Nullable
    protected Block findLogWithAffix(String prefix, String suffix) {
        // ugly, shouldnt be here
        // SUPPORT: TFC & AFC
        if (this.id.m_135827_().matches("tfc|afc")) {
            String prefix_ = prefix.isEmpty() ? "" : prefix + "_";
            var o = BuiltInRegistries.f_256975_.m_6612_(
                    new ResourceLocation(this.getNamespace(),
                            "wood/" + prefix_ + suffix + "/" + id.m_135815_()));
            if (o.isPresent()) return o.get();
        }

        List<ResourceLocation> targets = makeKnownIDConventionsAffix(
                this.id.m_135827_(), this.id.m_135815_(),
                prefix, suffix, Utils.getID(this.log).m_135815_());
        return Utils.findFirstInRegistry(BuiltInRegistries.f_256975_, targets);
    }

    private static @NotNull List<ResourceLocation> makeKnownIDConventionsAffix(
            String myNamespace, String myPath,
            String prefixOrInfix, String suffix,
            @Nullable String alternateNamespace) {

        boolean noneEmpty = !prefixOrInfix.isEmpty() && !suffix.isEmpty();
        String prefix_ = prefixOrInfix.isEmpty() ? "" : prefixOrInfix + "_";
        String _infix = prefixOrInfix.isEmpty() ? "" : "_" + prefixOrInfix;
        String _suffix = suffix.isEmpty() ? "" : "_" + suffix;

        List<ResourceLocation> targets = new ArrayList<>();
        targets.add(new ResourceLocation(myNamespace, myPath + _infix + _suffix));
        targets.add(new ResourceLocation(myNamespace, prefix_ + myPath + _suffix));
        if (alternateNamespace != null)
            targets.add(new ResourceLocation(myNamespace, alternateNamespace + _infix + _suffix));
        if (noneEmpty && alternateNamespace != null)
            targets.add(new ResourceLocation(myNamespace, prefix_ + alternateNamespace + _suffix));

        //For things like grimwood_wood -> grimwood
        if (myPath.endsWith(suffix)) {
            targets.add(new ResourceLocation(myNamespace, prefix_ + myPath));
        }
        return targets;
    }

    static List<ResourceLocation> makeKnownIDConventions(ResourceLocation id, String... affixKeyword) {
        String myPath = id.m_135815_();
        String myNamespace = id.m_135827_();
        List<ResourceLocation> possibleTargets = new ArrayList<>();
        for (String affix : affixKeyword) {
            possibleTargets.addAll(makeKnownIDConventionsAffix(myNamespace, myPath, "", affix, null));
            possibleTargets.addAll(makeKnownIDConventionsAffix(myNamespace, myPath, affix, "", null));
        }
        return possibleTargets;
    }

    @Nullable
    static Block findLog(ResourceLocation id) {
        var tests = makeKnownIDConventions(id, "log", "stem", "stalk", "hyphae");
        return Utils.findFirstInRegistry(BuiltInRegistries.f_256975_, tests);
    }

    @Nullable
    static Block findPlanks(ResourceLocation id) {
        var tests = makeKnownIDConventions(id, "planks", "plank");
        return Utils.findFirstInRegistry(BuiltInRegistries.f_256975_, tests);
    }


    //just copies base properties without calling copy
    public BlockBehaviour.Properties copyProperties() {
        var p = BlockBehaviour.Properties.m_284310_();
        p.m_284180_(this.getColor());
        if (this.canBurn()) p.m_278183_();
        p.m_60918_(this.getSound());
        return p;
    }


    //TODO:move in wood registry class
    public static class Finder extends SetFinderBuilder<WoodType> {
        private Supplier<Block> planksFinder;
        private Supplier<Block> logFinder;

        public Finder(ResourceLocation id) {
            super(id, WoodTypeRegistry.INSTANCE);
            this.log(() -> findLog(id)); // defaults
            this.planks(() -> findPlanks(id)); // defaults
        }

        /// PLANKS \\\
        public Finder planks(Supplier<Block> planksFinder) {
            this.planksFinder = planksFinder;
            return this;
        }

        public Finder planks(ResourceLocation id) {
            return this.planks(() -> BuiltInRegistries.f_256975_.m_6612_(id).orElseThrow(
                    () -> new IllegalStateException("Failed to find planks block: " + id)
            ));
        }

        public Finder planks(String planksName) {
            return this.planks(Utils.idWithOptionalNamespace(planksName, id.m_135827_()));
        }

        /**
         * @param prefix include the underscore, "_" if the blockId has one
         * @param suffix include the underscore, "_" if the blockId has one
         */
        public Finder planksAffix(String prefix, String suffix) {
            return planks(prefix + id.m_135815_() + suffix);
        }

        /**
         * @param suffix include the underscore, "_" if the blockId has one
         */
        @SuppressWarnings("UnusedReturnValue")
        public Finder planksSuffix(String suffix) {
            return planks(id.m_135815_() + suffix);
        }

        /// LOG \\\
        public Finder log(Supplier<Block> logFinder) {
            this.logFinder = logFinder;
            return this;
        }

        /// @param id Full Id of MudType as ResourceLocation
        public Finder log(ResourceLocation id) {
            return this.log(() -> BuiltInRegistries.f_256975_.m_6612_(id).orElseThrow(
                    () -> new IllegalStateException("Failed to find log block: " + id)));
        }

        /// @param nameLog name of Log without modId or namespace
        public Finder log(String nameLog) {
            return this.log(Utils.idWithOptionalNamespace(nameLog, id.m_135827_()));
        }

        /**
         * @param prefix include the underscore, "_" if the blockId has one
         * @param suffix include the underscore, "_" if the blockId has one
         */
        public Finder logAffix(String prefix, String suffix) {
            return log(prefix + id.m_135815_() + suffix);
        }

        /**
         * @param suffix include the underscore, "_" if the blockId has one
         */
        public Finder logSuffix(String suffix) {
            return log(id.m_135815_() + suffix);
        }


        @Override
        @ApiStatus.Internal
        public Optional<WoodType> get() {
            if (PlatHelper.isModLoaded(id.m_135827_())) {
                try {
                    Block plank = Preconditions.checkNotNull(planksFinder.get(), "Manual Finder - failed to find a plank block for {}", id);
                    Block log = Preconditions.checkNotNull(logFinder.get(), "Manual Finder - failed to find a log block for {}", id);
                    var woodType = new WoodType(id, plank, log);
                    childNames.forEach((key, value) -> {
                        try {
                            ItemLike obj = Preconditions.checkNotNull(value.get());
                            woodType.addChild(key, obj);
                        } catch (Exception e) {
                            Moonlight.LOGGER.warn("Failed to find child for WoodType: {} - {}. Ignored! ERROR: {}", id, key, e.getMessage());
                        }
                    });
                    return Optional.of(woodType);
                } catch (Exception e) {
                    Moonlight.LOGGER.warn("Failed to find custom WoodType:  {} - ", id, e);
                }
            }
            return Optional.empty();
        }

// ─────────────────────────────────────────── Marked For Removal ────────────────────────────────────────────

        /// USE {@link WoodTypeRegistry#addSimpleFinder(String, String)}
        @Deprecated(forRemoval = true)
        public Finder(ResourceLocation id, Supplier<Block> planks, Supplier<Block> log) {
            super(id, WoodTypeRegistry.INSTANCE);
            this.planksFinder = planks;
            this.logFinder = log;
        }

        /// USE {@link WoodTypeRegistry#addSimpleFinder(String, String)}
        @Deprecated(forRemoval = true)
        public static Finder simple(String modId, String woodTypeName, String planksName, String logName) {
            return simple(new ResourceLocation(modId, woodTypeName), new ResourceLocation(modId, planksName), new ResourceLocation(modId, logName));
        }

        /// USE {@link WoodTypeRegistry#addSimpleFinder(String, String)}
        @Deprecated(forRemoval = true)
        public static Finder simple(ResourceLocation woodTypeName, ResourceLocation planksName, ResourceLocation logName) {
            return new Finder(woodTypeName,
                    () -> BuiltInRegistries.f_256975_.m_7745_(planksName),
                    () -> BuiltInRegistries.f_256975_.m_7745_(logName));
        }

        /**
         * USE {@link WoodTypeRegistry#addSimpleFinder(String, String)}
         * <br>add {@link SetFinderBuilder#childBlockAffix(String, String, String)}
         * <br>OR
         * <br>add {@link SetFinderBuilder#childBlockSuffix(String, String)}
         */
        @Deprecated(forRemoval = true)
        public void addChild(String childType, String childName) {
            this.childBlock(childType, childName);
        }

        /**
         * USE {@link WoodTypeRegistry#addSimpleFinder(String, String)}
         * <br>add {@link SetFinderBuilder#childBlockAffix(String, String, String)}
         * <br>OR
         * <br>add {@link SetFinderBuilder#childBlockSuffix(String, String)}
         */
        @Deprecated(forRemoval = true)
        public void addChild(String childType, ResourceLocation childName) {
            this.childBlock(childType, childName);
        }

    }

}
