package net.mehvahdjukaar.moonlight.api.util.math;

import net.mehvahdjukaar.moonlight.api.util.math.colors.BaseColor;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.Vec3i;
import net.minecraft.util.Mth;
import net.minecraft.util.RandomSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.HitResult;
import net.minecraft.world.phys.Vec3;
import net.minecraft.world.phys.shapes.Shapes;
import net.minecraft.world.phys.shapes.VoxelShape;

import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

public class MthUtils {

    public static float[] polarToCartesian(float a, float r) {
        float x = r * Mth.m_14089_(a);
        float y = r * Mth.m_14031_(a);
        return new float[]{x, y};
    }

    public static float signedAngleDiff(double to, double from) {
        float x1 = Mth.m_14089_((float) to);
        float y1 = Mth.m_14031_((float) to);
        float x2 = Mth.m_14089_((float) from);
        float y2 = Mth.m_14031_((float) from);
        return (float) Mth.m_14136_(x1 * y1 - y1 * x2, x1 * x2 + y1 * y2);
    }

    //vector relative to a new basis
    public static Vec3 changeBasisN(Vec3 newBasisYVector, Vec3 rot) {
        Vec3 y = newBasisYVector.m_82541_();
        Vec3 x = new Vec3(y.f_82480_, y.f_82481_, y.f_82479_).m_82541_();
        Vec3 z = y.m_82537_(x).m_82541_();
        return changeBasis(x, y, z, rot);
    }

    public static Vec3 changeBasis(Vec3 newX, Vec3 newY, Vec3 newZ, Vec3 rot) {
        return newX.m_82490_(rot.f_82479_).m_82549_(newY.m_82490_(rot.f_82480_)).m_82549_(newZ.m_82490_(rot.f_82481_));
    }

    public static Vec3 getNormalFrom3DData(int direction) {
        return V3itoV3(Direction.m_122376_(direction).m_122436_());
    }

    public static Vec3 V3itoV3(Vec3i v) {
        return new Vec3(v.m_123341_(), v.m_123342_(), v.m_123343_());
    }

    private static double isClockWise(UnaryOperator<Vec3> rot, Direction dir) {
        Vec3 v = MthUtils.V3itoV3(dir.m_122436_());
        Vec3 v2 = rot.apply(v);
        return v2.m_82526_(new Vec3(0, 1, 0));
    }

    /**
     * Gives a vector that is equal to the one given rotated on the Y axis by a given direction
     *
     * @param dir horizontal direction
     */
    public static Vec3 rotateVec3(Vec3 vec, Direction dir) {
        double cos = 1;
        double sin = 0;
        switch (dir) {
            case SOUTH -> {
                cos = -1;
                sin = 0;
            }
            case WEST -> {
                cos = 0;
                sin = 1;
            }
            case EAST -> {
                cos = 0;
                sin = -1;
            }
            case UP -> {
                return new Vec3(vec.f_82479_, -vec.f_82481_, vec.f_82480_);
            }
            case DOWN -> {
                return new Vec3(vec.f_82479_, vec.f_82481_, vec.f_82480_);
            }
        }
        double dx = vec.f_82479_ * cos + vec.f_82481_ * sin;
        double dy = vec.f_82480_;
        double dz = vec.f_82481_ * cos - vec.f_82479_ * sin;
        return new Vec3(dx, dy, dz);
    }

    /**
     * Takes angles from 0 to 1
     *
     * @return mean angle
     */
    public static float averageAngles(Float... angles) {
        float x = 0, y = 0;
        for (float a : angles) {
            x += Mth.m_14089_((float) (a * Math.PI * 2));
            y += Mth.m_14031_((float) (a * Math.PI * 2));
        }
        return (float) (Mth.m_14136_(y, x) / (Math.PI * 2));
    }

    // in degrees. Opposite of Vec3.fromRotation
    public static double getPitch(Vec3 vec3) {
        return -Math.toDegrees(Math.asin(vec3.f_82480_));
    }

    // in degrees
    public static double getYaw(Vec3 vec3) {
        return Math.toDegrees(Math.atan2(-vec3.f_82479_, vec3.f_82481_));
    }

    // not sure about this one tbh
    public static double getRoll(Vec3 vec3) {
        return Math.toDegrees(Math.atan2(vec3.f_82480_, vec3.f_82479_));
    }

    public static double wrapRad(double pValue) {
        double p = Math.PI * 2;
        double d0 = pValue % p;
        if (d0 >= Math.PI) {
            d0 -= p;
        }

        if (d0 < -Math.PI) {
            d0 += p;
        }

        return d0;
    }

    public static float wrapRad(float pValue) {
        float p = (float) (Math.PI * 2);
        float d0 = pValue % p;
        if (d0 >= Math.PI) {
            d0 -= p;
        }

        if (d0 < -Math.PI) {
            d0 += p;
        }

        return d0;
    }

    /**
     * @param rand a rng
     * @param max  maximum value. Has to be >0
     * @param bias when a positive number, skew the average towards 0 (has to be from 0 to infinity).
     *             negative toward max (has to be from 0 to negative infinity). Values <= -1 are invalid.
     *             Setting it to 0 is equivalent to rand.nextFloat()*max.
     *             bias = 1 is slightly skewed towards 0 with average 0.38*max
     * @return a number between 0 and max
     * The bias parameters control how much the average is skewed toward 0 or max
     */
    public static float nextWeighted(RandomSource rand, float max, float bias) {
        float r = rand.m_188501_();
        if (bias <= 0) {
            if (bias == 0) return r * max;
            //mapping 0 -1 to 0 -inf
            bias = -bias / (bias - 1);
        }
        return (max * (1 - r)) / ((bias * max * r) + 1);
    }

    /**
     * Same as above but value is included between max and min
     */
    public static float nextWeighted(RandomSource rand, float max, float bias, float min) {
        return nextWeighted(rand, max - min, bias) + min;
    }

    public static float nextWeighted(RandomSource rand, float max) {
        return nextWeighted(rand, max, 1);
    }

    /**
     * Golden ratio
     */
    public static final float PHI = (float) (1 + (Math.sqrt(5d) - 1) / 2f);

    public static <T extends BaseColor<T>> T lerpColorScale(List<T> palette, float phase) {
        if (phase >= 1) phase = phase % 1;

        int n = palette.size();
        float g = n * phase;
        int ind = (int) Math.floor(g);

        float delta = g % 1;
        T start = palette.get(ind);
        T end = palette.get((ind + 1) % n);

        return start.mixWith(end, delta);
    }

    public static boolean isWithinRectangle(int x, int y, int width, int height, int mouseX, int mouseY) {
        return mouseX >= x && mouseX < x + width && mouseY >= y && mouseY < y + height;
    }

    public static VoxelShape rotateVoxelShape(VoxelShape source, Direction direction) {
        if (direction == Direction.NORTH) return source;
        AtomicReference<VoxelShape> newShape = new AtomicReference<>(Shapes.m_83040_());
        source.m_83286_((minX, minY, minZ, maxX, maxY, maxZ) -> {
            Vec3 min = new Vec3(minX - 0.5, minY - 0.5, minZ - 0.5);
            Vec3 max = new Vec3(maxX - 0.5, maxY - 0.5, maxZ - 0.5);
            Vec3 v1 = MthUtils.rotateVec3(min, direction);
            Vec3 v2 = MthUtils.rotateVec3(max, direction);
            VoxelShape s = Shapes.m_166049_(0.5 + Math.min(v1.f_82479_, v2.f_82479_), 0.5 + Math.min(v1.f_82480_, v2.f_82480_), 0.5 + Math.min(v1.f_82481_, v2.f_82481_),
                    0.5 + Math.max(v1.f_82479_, v2.f_82479_), 0.5 + Math.max(v1.f_82480_, v2.f_82480_), 0.5 + Math.max(v1.f_82481_, v2.f_82481_));
            newShape.set(Shapes.m_83110_(newShape.get(), s));
        });
        return newShape.get();
    }

    public static VoxelShape moveVoxelShape(VoxelShape source, Vec3 v) {
        AtomicReference<VoxelShape> newShape = new AtomicReference<>(Shapes.m_83040_());
        source.m_83286_((minX, minY, minZ, maxX, maxY, maxZ) -> {
            VoxelShape s = Shapes.m_166049_(minX + v.f_82479_, minY + v.f_82480_, minZ + v.f_82481_,
                    maxX + v.f_82479_, maxY + v.f_82480_, maxZ + v.f_82481_);
            newShape.set(Shapes.m_83110_(newShape.get(), s));
        });
        return newShape.get();
    }


    public static double lambertW0(double x) {
        double maxError = 1e-6;
        if (x == -1 / Math.E) {
            return -1;
        } else if (x >= -1 / Math.E) {
            double nLog = Math.log(x);
            double nLog0 = 1;
            while (Math.abs(nLog0 - nLog) > maxError) {
                nLog0 = (x * Math.exp(-nLog)) / (1 + nLog);
                nLog = (x * Math.exp(-nLog0)) / (1 + nLog0);
            }
            // precision of the return value
            return (Math.round(1000000 * nLog) / 1000000);
        } else {
            throw new IllegalArgumentException("Not in valid range for lambertW function. x has to be greater than or equal to -1/e.");
        }
    }

    // just brute forces it with newton approximation method
    // this only uses the secondary branch of the W function
    public static double lambertW1(double x) {
        double maxError = 1e-6;
        if (x == -1 / Math.E) {
            return -1;
        } else if (x < 0 && x > -1 / Math.E) {
            double nLog = Math.log(-x);
            double nLog0 = 1;
            while (Math.abs(nLog0 - nLog) > maxError) {
                nLog0 = (nLog * nLog + x / Math.exp(nLog)) / (nLog + 1);
                nLog = (nLog0 * nLog0 + x / Math.exp(nLog0)) / (nLog0 + 1);
            }
            // precision of the return value
            return (Math.round(1000000 * nLog) / 1000000);
        } else if (x == 0) {
            return 0;
        } else {
            throw new IllegalArgumentException("Not in valid range for lambertW function. x has to be in [-1/e,0]");
        }
    }

    /**
     * Exponent function that passed by 0,0 and 1,1
     */
    private static float exp01(float t, float base) {
        return (float) (base * Math.pow(1 / base + 1, t) - base);
    }

    /**
     * An exponent function that passes by 0,0 and 1,1
     *
     * @param t     time
     * @param curve determines the "curve" of the exponent graph.
     *              0 will be a line
     *              from 0 to 1 will curve with increasing severity (edge cases with vertical line at 1, which is not a valid input)
     *              from 0 to -1 will curve downwards in the same manner
     *              This parameter essentially controls the base of the exponent
     *              0.55 happens to map to a base close to Euler's number
     */
    public static float normalizedExponent(float t, float curve) {
        if (curve == 0) return t;
        float base;
        if (curve > 0) {
            base = (float) -Math.log(curve);
        } else {
            base = (float) (Math.log(-curve) - 1);
        }
        return exp01(t, base);
    }



    // collision code


    public static BlockHitResult collideWithSweptAABB(Entity entity, Vec3 movement, double maxStep) {
        AABB aabb = entity.m_20191_();
        return collideWithSweptAABB(entity.m_20182_(), aabb, movement, entity.m_9236_(), maxStep);
    }

    /**
     * Unlike vanilla .collide method this will have no tunnelling whatsoever and will stop the entity exactly when the first collision happens
     * It's somehow also more efficient than the vanilla method, around 2 times.
     */
    public static BlockHitResult collideWithSweptAABB(Vec3 myPos, AABB myBox, Vec3 movement, Level level, double maxStep) {
        double len = movement.m_82553_();
        if (maxStep >= len) return MthUtils.collideWithSweptAABB(myPos, myBox, movement, level);

        // Divide movement into smaller steps
        Vec3 stepMovement = movement.m_82541_().m_82490_(maxStep);
        Vec3 currentPos = myPos;
        BlockHitResult result;

        for (double moved = 0; moved < len; moved += maxStep) {
            if (moved + maxStep > len) {
                stepMovement = movement.m_82490_((len - moved) / len);
            }

            result = MthUtils.collideWithSweptAABB(currentPos, myBox, stepMovement, level);
            if (result.m_6662_() != HitResult.Type.MISS) {
                return result;
            }

            currentPos = currentPos.m_82549_(stepMovement);
            myBox = myBox.m_82383_(stepMovement);
        }

        Vec3 missPos = myPos.m_82549_(movement);
        return BlockHitResult.m_82426_(missPos, Direction.UP, BlockPos.m_274446_(missPos));
    }

    public static BlockHitResult collideWithSweptAABB(Vec3 myPos, AABB myBox, Vec3 movement, Level level) {
        AABB encompassing = myBox.m_82369_(movement);
        Set<BlockPos> positions = BlockPos.m_121921_(encompassing)
                .map(BlockPos::m_7949_).collect(Collectors.toSet());

        CollisionResult earliestCollision = null;
        BlockPos hitPos = null;

        for (BlockPos pos : positions) {
            BlockState state = level.m_8055_(pos);
            if (state.m_60795_()) continue;
            List<AABB> boxes = state.m_60812_(level, pos).m_83299_();
            for (AABB box : boxes) {
                box = box.m_82338_(pos);
                CollisionResult result = sweptAABB(myBox, box, movement);
                if (result == null || result.entryTime < 0) continue;
                if (earliestCollision == null) {
                    earliestCollision = result;
                    hitPos = pos;
                } else if (result.entryTime == earliestCollision.entryTime) {
                    Vec3 collidedPos = myPos.m_82549_(movement.m_82490_(result.entryTime));
                    if (pos.m_203193_(collidedPos) < hitPos.m_203193_(collidedPos)) {
                        earliestCollision = result;
                        hitPos = pos;
                    }
                } else if (result.entryTime < earliestCollision.entryTime) {
                    earliestCollision = result;
                    hitPos = pos;
                }
            }
        }


        if (earliestCollision != null && earliestCollision.entryTime < 1.0) {
            double entryTime = earliestCollision.entryTime - 0.00001f;
            movement = movement.m_82490_(entryTime);
            Vec3 finalPos = myPos.m_82549_(movement);

            return new BlockHitResult(finalPos, earliestCollision.direction.m_122424_(), hitPos, false);
        }

        Vec3 missPos = myPos.m_82549_(movement);
        return BlockHitResult.m_82426_(missPos, Direction.UP, BlockPos.m_274446_(missPos));
    }

    private static CollisionResult sweptAABB(AABB movingBox, AABB staticBox, Vec3 movement) {
        double entryX, entryY, entryZ;
        double exitX, exitY, exitZ;
        Direction collisionDirection;

        if (movement.f_82479_ > 0.0) {
            entryX = (staticBox.f_82288_ - movingBox.f_82291_) / movement.f_82479_;
            exitX = (staticBox.f_82291_ - movingBox.f_82288_) / movement.f_82479_;
        } else if (movement.f_82479_ < 0.0) {
            entryX = (staticBox.f_82291_ - movingBox.f_82288_) / movement.f_82479_;
            exitX = (staticBox.f_82288_ - movingBox.f_82291_) / movement.f_82479_;
        } else {
            entryX = Double.NEGATIVE_INFINITY;
            exitX = Double.POSITIVE_INFINITY;
        }

        if (movement.f_82480_ > 0.0) {
            entryY = (staticBox.f_82289_ - movingBox.f_82292_) / movement.f_82480_;
            exitY = (staticBox.f_82292_ - movingBox.f_82289_) / movement.f_82480_;
        } else if (movement.f_82480_ < 0.0) {
            entryY = (staticBox.f_82292_ - movingBox.f_82289_) / movement.f_82480_;
            exitY = (staticBox.f_82289_ - movingBox.f_82292_) / movement.f_82480_;
        } else {
            entryY = Double.NEGATIVE_INFINITY;
            exitY = Double.POSITIVE_INFINITY;
        }

        if (movement.f_82481_ > 0.0) {
            entryZ = (staticBox.f_82290_ - movingBox.f_82293_) / movement.f_82481_;
            exitZ = (staticBox.f_82293_ - movingBox.f_82290_) / movement.f_82481_;
        } else if (movement.f_82481_ < 0.0) {
            entryZ = (staticBox.f_82293_ - movingBox.f_82290_) / movement.f_82481_;
            exitZ = (staticBox.f_82290_ - movingBox.f_82293_) / movement.f_82481_;
        } else {
            entryZ = Double.NEGATIVE_INFINITY;
            exitZ = Double.POSITIVE_INFINITY;
        }

        double entryTime = Math.max(Math.max(entryX, entryY), entryZ);
        double exitTime = Math.min(Math.min(exitX, exitY), exitZ);

        if (entryTime > exitTime || (entryX < 0.0 && entryY < 0.0 && entryZ < 0.0) || entryX > 1.0 || entryY > 1.0 || entryZ > 1.0) {
            return null;
        }

        if (entryX > entryY && entryX > entryZ) {
            if (movement.f_82479_ > 0.0) {
                collisionDirection = Direction.EAST;
            } else {
                collisionDirection = Direction.WEST;
            }
        } else if (entryY > entryZ) {
            if (movement.f_82480_ > 0.0) {
                collisionDirection = Direction.UP;
            } else {
                collisionDirection = Direction.DOWN;
            }
        } else {
            if (movement.f_82481_ > 0.0) {
                collisionDirection = Direction.SOUTH;
            } else {
                collisionDirection = Direction.NORTH;
            }
        }

        return new CollisionResult(entryTime, collisionDirection);
    }

    private record CollisionResult(double entryTime, Direction direction) {
    }


    //ease functions

}
