package net.mehvahdjukaar.moonlight.core.set;

import com.google.common.base.Stopwatch;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import net.mehvahdjukaar.moonlight.api.MoonlightRegistry;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.core.HolderSet;
import net.minecraft.core.HolderSet.Named;
import net.minecraft.core.Registry;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.tags.TagKey;
import net.minecraft.world.item.DyeColor;
import net.minecraft.world.item.Item;
import net.minecraft.world.level.block.Block;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.Map.Entry;

public class BlocksColorInternal {
    public static final List<DyeColor> VANILLA_COLORS = List.of(DyeColor.WHITE,
            DyeColor.ORANGE, DyeColor.MAGENTA, DyeColor.LIGHT_BLUE, DyeColor.YELLOW, DyeColor.LIME, DyeColor.PINK, DyeColor.GRAY,
            DyeColor.LIGHT_GRAY, DyeColor.CYAN, DyeColor.PURPLE, DyeColor.BLUE, DyeColor.BROWN, DyeColor.GREEN, DyeColor.RED, DyeColor.BLACK);
    public static final List<DyeColor> MODDED_COLORS = List.of(Arrays.stream(DyeColor.values()).filter(v -> !VANILLA_COLORS.contains(v)).toArray(DyeColor[]::new));

    private static final Map<String, ColoredSet<Block>> BLOCK_COLOR_SETS = new HashMap<>();
    private static final Map<String, ColoredSet<Item>> ITEM_COLOR_SETS = new HashMap<>();

    private static final Object2ObjectOpenHashMap<Object, DyeColor> OBJ_TO_COLORS = new Object2ObjectOpenHashMap<>();
    private static final Object2ObjectOpenHashMap<Object, String> OBJ_TO_TYPE = new Object2ObjectOpenHashMap<>();


    public static void setup() {
        Stopwatch sw = Stopwatch.createStarted();

        Map<String, DyeColor> colors = new HashMap<>();
        VANILLA_COLORS.forEach(d -> colors.put(d.m_41065_(), d));
        List<String> colorPriority = new ArrayList<>(colors.keySet().stream().toList());

        addColoredFromRegistry(colors, colorPriority, BuiltInRegistries.f_256975_, BLOCK_COLOR_SETS);
        addColoredFromRegistry(colors, colorPriority, BuiltInRegistries.f_257033_, ITEM_COLOR_SETS);

        Moonlight.LOGGER.info("Initialized color sets in {}ms", sw.elapsed().toMillis());
    }

    public static void registerBlockColorSet(ResourceLocation key, EnumMap<DyeColor, Block> blocks, @Nullable Block defaultBlock) {
        BLOCK_COLOR_SETS.put(key.toString(), new ColoredSet<>(key, blocks, BuiltInRegistries.f_256975_, defaultBlock));
    }

    public static void registerItemColorSet(ResourceLocation key, EnumMap<DyeColor, Item> items, @Nullable Item defaultItem) {
        ITEM_COLOR_SETS.put(key.toString(), new ColoredSet<>(key, items, BuiltInRegistries.f_257033_, defaultItem));
    }

    private static <T> void addColoredFromRegistry(Map<String, DyeColor> colors, List<String> colorPriority,
                                                   Registry<T> registry, Map<String, ColoredSet<T>> colorSetMap) {
        Map<ResourceLocation, EnumMap<DyeColor, T>> groupedByType = new HashMap<>();
        colorPriority.sort(Comparator.comparingInt(String::length));
        Collections.reverse(colorPriority);
        //group by color
        loop1:
        for (var e : registry.m_6579_()) {
            ResourceLocation id = e.getKey().m_135782_();
            String name = id.m_135815_();
            if (!name.contains("_")) continue;

            for (var c : colorPriority) {
                ResourceLocation newId = null;
                if (name.startsWith(c + "_")) {
                    newId = new ResourceLocation(id.m_135827_(), name.substring((c + "_").length()));
                }
                if (name.endsWith("_" + c)) {
                    newId = new ResourceLocation(id.m_135827_(), name.substring(0, name.length() - ("_" + c).length()));
                }
                if (newId != null) {
                    DyeColor dyeColor = colors.get(c);
                    groupedByType.computeIfAbsent(newId, a -> new EnumMap<>(DyeColor.class)).put(dyeColor, e.getValue());
                    continue loop1;
                }
            }
        }

        //to qualify all vanilla colors must be found
        for (var j : groupedByType.entrySet()) {
            var map = j.getValue();
            ResourceLocation id = j.getKey();
            if (isBlacklisted(id)) continue;
            if (map.keySet().containsAll(VANILLA_COLORS)) {
                var set = new ColoredSet<>(id, map, registry);
                colorSetMap.put(id.toString(), set);

                for (var v : set.colorsToObj.entrySet()) {
                    OBJ_TO_COLORS.put(v.getValue(), v.getKey());
                    OBJ_TO_TYPE.put(v.getValue(), id.toString());
                }
                OBJ_TO_TYPE.put(set.defaultObj, id.toString());
            }
        }
    }

    private static boolean isBlacklisted(ResourceLocation id) {
        String modId = id.m_135827_();
        return modId.equals("energeticsheep") || modId.equals("xycraft_world") || modId.equals("botania") || modId.equals("spectrum");
    }

    @Nullable
    public static DyeColor getColor(Block block) {
        return OBJ_TO_COLORS.get(block);
    }

    @Nullable
    public static DyeColor getColor(Item item) {
        return OBJ_TO_COLORS.get(item);
    }

    @Nullable
    public static Item getColoredItem(String key, @Nullable DyeColor color) {
        var set = getItemSet(key);
        if (set != null) {
            return set.with(color);
        }
        return null;
    }

    @Nullable
    public static Block getColoredBlock(String key, @Nullable DyeColor color) {
        var set = getBlockSet(key);
        if (set != null) {
            return set.with(color);
        }
        return null;
    }

    public static Set<String> getBlockKeys() {
        return BLOCK_COLOR_SETS.keySet();
    }

    public static Set<String> getItemKeys() {
        return ITEM_COLOR_SETS.keySet();
    }

    /**
     * Changes this block color
     * If the given color is null it will yield the default colored block, usually uncolored or white
     * Will return null if no block can be found using that color
     */
    @Nullable
    public static Block changeColor(Block old, @Nullable DyeColor newColor) {
        String key = getKey(old);
        if (key != null) {
            var set = getBlockSet(key);
            if (set != null) {
                var b = set.with(newColor);
                if (b != old) return b;
            }
        }
        return null;
    }

    /**
     * Changes this item color
     * If the given color is null it will yield the default colored item, usually uncolored or white
     * Will return null if no item can be found using that color
     */
    @Nullable
    public static Item changeColor(Item old, @Nullable DyeColor newColor) {
        String key = getKey(old);
        if (key != null) {
            var set = getItemSet(key);
            if (set != null) {
                var i = set.with(newColor);
                if (i != old) return i;
            }
        }
        return null;
    }

    @Nullable
    public static String getKey(Block block) {
        return OBJ_TO_TYPE.get(block);
    }

    @Nullable
    public static String getKey(Item item) {
        return OBJ_TO_TYPE.get(item);
    }

    @Nullable
    private static ColoredSet<Block> getBlockSet(String key) {
        key = new ResourceLocation(key).toString();
        return BLOCK_COLOR_SETS.get(key);
    }

    @Nullable
    private static ColoredSet<Item> getItemSet(String key) {
        key = new ResourceLocation(key).toString();
        return ITEM_COLOR_SETS.get(key);
    }

    @Nullable
    public static HolderSet<Block> getBlockHolderSet(String key) {
        var set = getBlockSet(key);
        if (set != null) {
            return set.makeHolderSet(BuiltInRegistries.f_256975_);
        }
        return null;
    }

    @Nullable
    public static HolderSet<Item> getItemHolderSet(String key) {
        var set = getItemSet(key);
        if (set != null) {
            return set.makeHolderSet(BuiltInRegistries.f_257033_);
        }
        return null;
    }

    /**
     * A collection of blocks or items that come in all colors
     */
    private static class ColoredSet<T> {

        private final ResourceLocation id;
        private final Map<DyeColor, T> colorsToObj;
        private final T defaultObj;

        private ColoredSet(ResourceLocation id, EnumMap<DyeColor, T> map, Registry<T> registry) {
            this(id, map, registry, null);
        }

        private ColoredSet(ResourceLocation id, EnumMap<DyeColor, T> map, Registry<T> registry, @Nullable T defBlock) {
            this.colorsToObj = map;
            this.id = id;

            //fill optional
            List<String> newColorMods = List.of("tinted", "dye_depot", "dyenamics");
            colors:
            for (var c : MODDED_COLORS) {
                String namespace = id.m_135827_();
                String path = id.m_135815_();

                for (var mod : newColorMods) {
                    for (var s : new String[]{namespace + ":" + path + "_%s", namespace + ":%s_" + path, mod + ":" + path + "_%s", mod + ":%s_" + path}) {
                        var o = registry.m_6612_(new ResourceLocation(String.format(s, c.m_41065_())));
                        if (o.isPresent()) {
                            colorsToObj.put(c, o.get());
                            continue colors;
                        }
                    }
                }
            }

            //fill default
            this.defaultObj = defBlock == null ? computeDefault(id, registry) : defBlock;
        }

        private T computeDefault(ResourceLocation id, Registry<T> registry) {
            if (id.m_135827_().equals("minecraft") && id.m_135815_().contains("stained_glass")) {
                id = new ResourceLocation(id.m_135815_().replace("stained_", ""));
            } else if (id.m_135827_().equals("quark")) {
                if (id.m_135815_().equals("rune")) {
                    id = new ResourceLocation("quark", "blank_rune");
                } else if (id.m_135815_().equals("shard")) {
                    id = new ResourceLocation("quark", "clear_shard");
                }
            } else if (id.equals(new ResourceLocation("suppsquared:sack"))) {
                id = new ResourceLocation("supplementaries:sack");
            }
            ResourceLocation finalId = id;
            var o = registry.m_6612_(id);
            if (o.isEmpty()) {
                return registry.m_6612_(new ResourceLocation(finalId.m_135815_()))
                        .orElseGet(() -> colorsToObj.get(DyeColor.WHITE));
            } else {
                return o.get();
            }
        }

        /**
         * Kind of expensive. don't call too often
         */
        private HolderSet<T> makeHolderSet(Registry<T> registry) {
            //standard tag location
            var v = registry.m_203431_(TagKey.m_203882_(registry.m_123023_(),
                    new ResourceLocation(id.m_135827_(), id.m_135815_() + "s")));
            if (v.isEmpty()) {
                v = registry.m_203431_(TagKey.m_203882_(registry.m_123023_(),
                        new ResourceLocation(PlatHelper.getPlatform().isForge() ? "forge" : "c", id.m_135815_() + "s")));
            }
            if (v.isPresent()) {
                var tag = v.get();
                boolean success = true;
                for (var t : colorsToObj.values()) {
                    if (!tag.m_203333_(registry.m_246971_(registry.m_7854_(t).get()))) {
                        success = false;
                        break;
                    }
                }
                if (success) return tag;
            }
            return HolderSet.m_205803_(t -> registry.m_246971_(registry.m_7854_(t).get()),
                    new ArrayList<>(colorsToObj.values()));
        }

        /**
         * Null if no color is available.
         * If null dye is provided will give the default color
         */
        @Nullable
        private T with(@Nullable DyeColor newColor) {
            if (newColor != null && !colorsToObj.containsKey(newColor)) return null;
            return colorsToObj.getOrDefault(newColor, defaultObj);
        }

    }
}