package gollorum.signpost.utils;

import com.mojang.serialization.Codec;
import io.netty.buffer.ByteBuf;
import net.minecraft.network.codec.ByteBufCodecs;
import net.minecraft.network.codec.StreamCodec;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

/// Not a monad
public abstract class Either<Left, Right> {

    public static <Left, Right> Either<Left, Right> left(Left left) { return new LeftImpl<>(left); }
    public static <Left, Right> Either<Left, Right> right(Right right) { return new RightImpl<>(right); }

    public static <Left, Right> Either<Left, Right> leftIfPresent(Optional<Left> left, Supplier<Right> right) {
        return left.map(Either::<Left, Right>left).orElseGet(() -> Either.<Left, Right>right(right.get()));
    }
    public static <Left, Right> Either<Left, Right> rightIfPresent(Optional<Right> right, Supplier<Left> left) {
        return right.map(Either::<Left, Right>right).orElseGet(() -> Either.<Left, Right>left(left.get()));
    }

    public static <Left, Right> Either<Left, Right> fromMojangEither(com.mojang.datafixers.util.Either<Left, Right> mojangEither) {
        return mojangEither.map(Either::left, Either::right);
    }

    public com.mojang.datafixers.util.Either<Left, Right> toMojangEither() {
        return match(
            com.mojang.datafixers.util.Either::left,
            com.mojang.datafixers.util.Either::right
        );
    }

    private Either() { }

    public abstract boolean isLeft();
    public boolean isRight() { return !isLeft(); }

    public Either<Right, Left> flip() {
        return match(Either::right, Either::left);
    }

    public abstract Right rightOr(Function<Left, Right> func);
    public abstract Left leftOr(Function<Right, Left> func);

    public abstract <NewRight> Either<Left, NewRight> flatMapRight(Function<Right, Either<Left, NewRight>> mapping);
    public abstract <NewLeft> Either<NewLeft, Right> flatMapLeft(Function<Left, Either<NewLeft, Right>> mapping);

    public final <NewRight> Either<Left, NewRight> mapRight(Function<Right, NewRight> mapping) {
        return flatMapRight(r -> Either.right(mapping.apply(r)));
    }
    public final <NewLeft> Either<NewLeft, Right> mapLeft(Function<Left, NewLeft> mapping) {
        return flatMapLeft(l -> Either.left(mapping.apply(l)));
    }

    public abstract <Out> Out match(Function<Left, Out> leftMapping, Function<Right, Out> rightMapping);

    public abstract void consume(Consumer<Left> leftMapping, Consumer<Right> rightMapping);

    public Right rightOrThrow() { return rightOr(l -> { throw new RuntimeException("Right value was not present."); }); }

    public Left leftOrThrow() { return leftOr(l -> { throw new RuntimeException("Left value was not present."); }); }

    private static class LeftImpl<Left, Right> extends Either<Left, Right> {
        private final Left left;
        private LeftImpl(Left left) { this.left = left; }

        @Override
        public boolean isLeft() { return true; }

        @Override
        public Right rightOr(Function<Left, Right> func) { return func.apply(left); }

        @Override
        public Left leftOr(Function<Right, Left> func) { return left; }

        @Override
        public <NewRight> Either<Left, NewRight> flatMapRight(Function<Right, Either<Left, NewRight>> mapping) {
            return Either.left(left);
        }

        @Override
        public <NewLeft> Either<NewLeft, Right> flatMapLeft(Function<Left, Either<NewLeft, Right>> mapping) {
            return mapping.apply(left);
        }

        @Override
        public <Out> Out match(Function<Left, Out> leftMapping, Function<Right, Out> rightMapping) {
            return leftMapping.apply(left);
        }

        @Override
        public void consume(Consumer<Left> leftMapping, Consumer<Right> rightMapping) {
            leftMapping.accept(left);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            LeftImpl<?, ?> left1 = (LeftImpl<?, ?>) o;
            return Objects.equals(left, left1.left);
        }

        @Override
        public int hashCode() {
            return Objects.hash(left);
        }

        @Override
        public String toString() {
            return "Left{" + left + '}';
        }
    }

    private static class RightImpl<Left, Right> extends Either<Left, Right> {
        private final Right right;
        private RightImpl(Right right) { this.right = right; }

        @Override
        public boolean isLeft() { return false; }

        @Override
        public Right rightOr(Function<Left, Right> func) { return right; }

        @Override
        public Left leftOr(Function<Right, Left> func) { return func.apply(right); }

        @Override
        public <NewRight> Either<Left, NewRight> flatMapRight(Function<Right, Either<Left, NewRight>> mapping) {
            return mapping.apply(right);
        }

        @Override
        public <NewLeft> Either<NewLeft, Right> flatMapLeft(Function<Left, Either<NewLeft, Right>> mapping) {
            return Either.right(right);
        }

        @Override
        public <Out> Out match(Function<Left, Out> leftMapping, Function<Right, Out> rightMapping) {
            return rightMapping.apply(right);
        }

        @Override
        public void consume(Consumer<Left> leftMapping, Consumer<Right> rightMapping) {
            rightMapping.accept(right);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            RightImpl<?, ?> right1 = (RightImpl<?, ?>) o;
            return Objects.equals(right, right1.right);
        }

        @Override
        public int hashCode() {
            return Objects.hash(right);
        }

        @Override
        public String toString() {
            return "Right{" + right + '}';
        }
    }

    public static <Left, Right, TBuf extends ByteBuf> StreamCodec<TBuf, Either<Left, Right>> streamCodec(StreamCodec<? super TBuf, Left> leftCodec, StreamCodec<? super TBuf, Right> rightCodec) {
        return ByteBufCodecs.<TBuf, Left, Right>either(leftCodec, rightCodec).map(
            Either::fromMojangEither,
            Either::toMojangEither
        );
    }

    public static <Left, Right> Codec<Either<Left, Right>> codec(Codec<Left> leftCodec, Codec<Right> rightCodec) {
        return Codec.BOOL.fieldOf("IsLeft").codec().<Either<Left, Right>>dispatch(
            Either::isLeft,
            isLeft -> (isLeft
                ? leftCodec.<Either<Left, Right>>xmap(Either::left, Either::leftOrThrow)
                : rightCodec.<Either<Left, Right>>xmap(Either::right, Either::rightOrThrow)
            ).fieldOf("Data")
        );
    }
}
