package me.alexdevs.solstice.modules.afk;

import eu.pb4.placeholders.api.PlaceholderContext;
import eu.pb4.placeholders.api.PlaceholderResult;
import eu.pb4.placeholders.api.Placeholders;
import me.alexdevs.solstice.Solstice;
import me.alexdevs.solstice.api.ServerLocation;
import me.alexdevs.solstice.api.events.CommandEvents;
import me.alexdevs.solstice.api.events.PlayerActivityEvents;
import me.alexdevs.solstice.api.events.SolsticeEvents;
import me.alexdevs.solstice.api.module.ModuleBase;
import me.alexdevs.solstice.api.text.Format;
import me.alexdevs.solstice.modules.afk.commands.ActiveTimeCommand;
import me.alexdevs.solstice.modules.afk.commands.AfkCommand;
import me.alexdevs.solstice.modules.afk.data.*;
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerTickEvents;
import net.fabricmc.fabric.api.event.player.*;
import net.fabricmc.fabric.api.message.v1.ServerMessageEvents;
import net.fabricmc.fabric.api.networking.v1.ServerPlayConnectionEvents;
import net.minecraft.class_1269;
import net.minecraft.class_1271;
import net.minecraft.class_1297;
import net.minecraft.class_243;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.server.MinecraftServer;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

public class AfkModule extends ModuleBase.Toggleable {
    public static final double sprintSpeed = 0.280617;
    public static final double walkSpeed = 0.215859;
    public static final double sneakSpeed = 0.0841;

    public static final int LEADERBOARD_SIZE = 10;

    public AfkModule(class_2960 id) {
        super(id);
    }

    public enum AfkTriggerReason {
        MANUAL,
        MOVEMENT,
        LOOK_CHANGE,
        CHAT_MESSAGE,
        COMMAND,
        BLOCK_ATTACK,
        BLOCK_INTERACT,
        ENTITY_ATTACK,
        ENTITY_INTERACT,
        ITEM_USE,
    }

    private final Map<UUID, PlayerActivityState> activities = new ConcurrentHashMap<>();


    @Override
    public void init() {
        registerConfig(AfkConfig.class, AfkConfig::new);
        registerLocale(AfkLocale.MODULE);
        registerPlayerData(AfkPlayerData.class, AfkPlayerData::new);
        registerServerData(AfkServerData.class, AfkServerData::new);

        this.commands.add(new AfkCommand(this));
        this.commands.add(new ActiveTimeCommand(this));

        Placeholders.register(class_2960.method_60655(Solstice.MOD_ID, "afk"), (context, arg) -> {
            if (!context.hasPlayer())
                return PlaceholderResult.invalid("No player!");

            var player = context.player();

            if (isPlayerAfk(player))
                return PlaceholderResult.value(Format.parse(getConfig().tag));
            else
                return PlaceholderResult.value("");
        });

        SolsticeEvents.READY.register((instance, server) -> {
            Solstice.scheduler.scheduleAtFixedRate(this::updateActiveTime, 0, 1, TimeUnit.SECONDS);
            calculateLeaderboard();
        });

        ServerPlayConnectionEvents.JOIN.register((handler, sender, server) -> {
            activities.put(handler.method_32311().method_5667(), new PlayerActivityState(handler.method_32311(), server.method_3780()));
        });

        ServerPlayConnectionEvents.DISCONNECT.register((handler, server) -> {
            activities.remove(handler.method_32311().method_5667());
        });

        ServerTickEvents.END_SERVER_TICK.register(this::tick);

        PlayerActivityEvents.AFK.register((player) -> {
            var config = getConfig();

            if (player.method_51469().method_33144()) {
                player.method_51469().method_8448();
            }

            Solstice.LOGGER.info("{} is AFK. Active time: {} seconds.", player.method_7334().getName(), getActiveTime(player.method_5667()));
            if (!config.announce)
                return;

            var playerContext = PlaceholderContext.of(player);

            Solstice.getInstance().broadcast(locale().get("goneAfk", playerContext));
        });

        PlayerActivityEvents.AFK_RETURN.register((player, reason) -> {
            var config = getConfig();

            if (player.method_51469().method_33144()) {
                player.method_51469().method_8448();
            }

            Solstice.LOGGER.info("{} is no longer AFK due to {}. Active time: {} seconds.", player.method_7334().getName(), reason.name(), getActiveTime(player.method_5667()));
            if (!config.announce)
                return;

            var playerContext = PlaceholderContext.of(player);

            Solstice.getInstance().broadcast(locale().get("returnAfk", playerContext));
        });

        registerTriggers();
    }

    public AfkServerData getServerData() {
        return Solstice.serverData.getData(AfkServerData.class);
    }

    private void updateActiveTime() {
        var activePlayers = Solstice.server.method_3760().method_14571()
                .stream().filter(player -> !isPlayerAfk(player));

        activePlayers.forEach(player -> {
            var activity = activities.computeIfAbsent(player.method_5667(), uuid -> new PlayerActivityState(player, player.method_5682().method_3780()));
            if (!activity.activeTimeEnabled)
                return;

            var playerData = getPlayerData(player.method_5667());
            playerData.activeTime++;
            tryInsertLeaderboard(player, playerData.activeTime);
        });
    }

    private void tryInsertLeaderboard(class_3222 player, int activeTime) {
        var serverData = getServerData();
        var leaderboard = serverData.leaderboard;

        // if in list, update
        var entry = leaderboard.stream().filter(e -> e.uuid().equals(player.method_5667())).findFirst();
        if (entry.isPresent()) {
            entry.get().activeTime(activeTime);
            entry.get().name(player.method_7334().getName());
            leaderboard.sort((o1, o2) -> Integer.compare(o2.activeTime(), o1.activeTime()));
            return;
        }

        // if not in list, insert
        var smallest = leaderboard.stream().min(Comparator.comparingInt(LeaderboardEntry::activeTime));
        if (smallest.isPresent()) {
            if (smallest.get().activeTime() < activeTime) {
                leaderboard.remove(smallest.get());
                leaderboard.add(new LeaderboardEntry(player.method_7334().getName(), player.method_5667(), activeTime));
                leaderboard.sort((o1, o2) -> Integer.compare(o2.activeTime(), o1.activeTime()));
            }
        } else {
            leaderboard.add(new LeaderboardEntry(player.method_7334().getName(), player.method_5667(), activeTime));
        }
    }

    private void tick(MinecraftServer server) {
        var config = getConfig();
        if (!config.enable)
            return;

        server.method_3760().method_14571().forEach(player -> {
            var activity = activities.computeIfAbsent(player.method_5667(), uuid -> new PlayerActivityState(player, server.method_3780()));

            var curLocation = new ServerLocation(player);
            var oldLocation = activity.location;
            activity.location = curLocation;

            var delta = curLocation.getDelta(oldLocation);
            var horizontalDelta = new class_243(delta.method_10216(), 0, delta.method_10215());

            var speed = horizontalDelta.method_1033();

            // Suppose the player in a vehicle will look around, so we only check for movement when not in a vehicle.
            if (player.method_5854() == null) {
                // Defeats some anti-afk stuff, like pools. Works best when no lag.
                if ((player.method_5715() && speed >= sneakSpeed) || (player.method_5624() && speed >= sprintSpeed) || (speed >= walkSpeed)) {
                    if (getConfig().triggers.onMovement) {
                        clearAfk(player, AfkTriggerReason.MOVEMENT);
                    }
                }
            }

            // Looking around requires player input
            if (curLocation.getPitch() != oldLocation.getPitch() || curLocation.getYaw() != oldLocation.getYaw()) {
                if (getConfig().triggers.onLookChange) {
                    clearAfk(player, AfkTriggerReason.LOOK_CHANGE);
                }
            }

            var ticks = server.method_3780();
            if (activity.lastUpdate < ticks - config.timeTrigger * 20) {
                if (!activity.isAfk && activity.afkEnabled) {
                    activity.isAfk = true;
                    PlayerActivityEvents.AFK.invoker().onAfk(player);
                }
            }
        });
    }

    public AfkConfig getConfig() {
        return Solstice.configManager.getData(AfkConfig.class);
    }

    public AfkPlayerData getPlayerData(UUID playerUuid) {
        return Solstice.playerData.get(playerUuid).getData(AfkPlayerData.class);
    }

    public boolean isPlayerAfk(class_3222 player) {
        return activities.containsKey(player.method_5667()) && activities.get(player.method_5667()).isAfk;
    }

    public void setPlayerAfk(class_3222 player, boolean isAfk) {
        if (!activities.containsKey(player.method_5667()))
            return;

        var config = getConfig();
        var activity = activities.get(player.method_5667());
        if (isAfk) {
            activity.lastUpdate = activity.lastUpdate - (config.timeTrigger * 20);
        } else {
            clearAfk(player, AfkTriggerReason.MANUAL);
        }
    }

    public int getActiveTime(UUID playerUuid) {
        return getPlayerData(playerUuid).activeTime;
    }

    public void forceRecalculateLeaderboard() {
        getServerData().forceCalculateLeaderboard = true;
        calculateLeaderboard();
    }

    private void calculateLeaderboard() {
        var serverData = getServerData();
        if (!serverData.forceCalculateLeaderboard)
            return;

        serverData.forceCalculateLeaderboard = false;

        var userCache = Solstice.getUserCache();
        var temp = new ArrayList<LeaderboardEntry>();
        for (var name : userCache.getAllNames()) {
            var profile = userCache.getByName(name);
            if (profile.isEmpty())
                continue;

            var playerData = Solstice.playerData.get(profile.get().getId()).getData(AfkPlayerData.class);
            if (playerData.activeTime > 0) {
                var entry = new LeaderboardEntry(profile.get().getName(), profile.get().getId(), playerData.activeTime);
                temp.add(entry);
            }
        }

        temp.sort((o1, o2) -> Integer.compare(o2.activeTime(), o1.activeTime()));

        serverData.leaderboard.clear();

        for (var i = 0; i < Math.min(temp.size(), LEADERBOARD_SIZE); i++) {
            serverData.leaderboard.add(temp.get(i));
        }

        var onlinePlayers = Solstice.server.method_3760().method_14571().stream().map(class_1297::method_5667).toList();
        Solstice.playerData.disposeMissing(onlinePlayers);
    }

    public List<LeaderboardEntry> getActiveTimeLeaderboard() {
        var serverData = getServerData();
        return Collections.unmodifiableList(serverData.leaderboard);
    }

    public List<class_3222> getCurrentActivePlayers() {
        return Solstice.server.method_3760().method_14571().stream().filter(player -> !isPlayerAfk(player)).toList();
    }

    private void clearAfk(class_3222 player, AfkTriggerReason reason) {
        if (!activities.containsKey(player.method_5667()))
            return;

        var activity = activities.get(player.method_5667());
        activity.lastUpdate = Solstice.server.method_3780();

        if (!activity.afkEnabled)
            return;

        if (activity.isAfk) {
            activity.isAfk = false;
            PlayerActivityEvents.AFK_RETURN.invoker().onAfkReturn(player, reason);
        }

    }

    private void registerTriggers() {
        AttackBlockCallback.EVENT.register((player, world, hand, pos, direction) -> {
            if (getConfig().triggers.onBlockAttack) {
                clearAfk((class_3222) player, AfkTriggerReason.BLOCK_ATTACK);
            }
            return class_1269.field_5811;
        });

        AttackEntityCallback.EVENT.register((player, world, hand, entity, hitResult) -> {
            if (getConfig().triggers.onEntityAttack) {
                clearAfk((class_3222) player, AfkTriggerReason.ENTITY_ATTACK);
            }
            return class_1269.field_5811;
        });

        UseBlockCallback.EVENT.register((player, world, hand, hitResult) -> {
            if (getConfig().triggers.onBlockInteract) {
                clearAfk((class_3222) player, AfkTriggerReason.BLOCK_INTERACT);
            }
            return class_1269.field_5811;
        });

        UseEntityCallback.EVENT.register((player, world, hand, entity, hitResult) -> {
            if (getConfig().triggers.onEntityInteract) {
                clearAfk((class_3222) player, AfkTriggerReason.ENTITY_INTERACT);
            }
            return class_1269.field_5811;
        });

        UseItemCallback.EVENT.register((player, world, hand) -> {
            if (getConfig().triggers.onItemUse) {
                clearAfk((class_3222) player, AfkTriggerReason.ITEM_USE);
            }
            return class_1271.method_22430(player.method_5998(hand));
        });

        ServerMessageEvents.ALLOW_CHAT_MESSAGE.register((message, sender, params) -> {
            if (getConfig().triggers.onChat) {
                clearAfk(sender, AfkTriggerReason.CHAT_MESSAGE);
            }
            return true;
        });

        CommandEvents.ALLOW_COMMAND.register((source, command) -> {
            if (!source.method_43737())
                return true;

            if (getConfig().triggers.onCommand) {
                clearAfk(source.method_44023(), AfkTriggerReason.COMMAND);
            }
            return true;
        });
    }
}
