package me.alexdevs.solstice.modules.afk;

import eu.pb4.placeholders.api.PlaceholderContext;
import eu.pb4.placeholders.api.PlaceholderResult;
import eu.pb4.placeholders.api.Placeholders;
import me.alexdevs.solstice.Solstice;
import me.alexdevs.solstice.api.ServerPosition;
import me.alexdevs.solstice.api.events.PlayerActivityEvents;
import me.alexdevs.solstice.api.events.SolsticeEvents;
import me.alexdevs.solstice.api.module.ModuleBase;
import me.alexdevs.solstice.locale.Locale;
import me.alexdevs.solstice.modules.afk.commands.AfkCommand;
import me.alexdevs.solstice.modules.afk.data.AfkConfig;
import me.alexdevs.solstice.modules.afk.data.AfkLocale;
import me.alexdevs.solstice.modules.afk.data.AfkPlayerData;
import me.alexdevs.solstice.api.text.Format;
import me.lucko.fabric.api.permissions.v0.Permissions;
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerTickEvents;
import net.fabricmc.fabric.api.event.player.*;
import net.fabricmc.fabric.api.message.v1.ServerMessageEvents;
import net.fabricmc.fabric.api.networking.v1.ServerPlayConnectionEvents;
import net.minecraft.class_1269;
import net.minecraft.class_1271;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.server.MinecraftServer;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

public class AfkModule extends ModuleBase {
    public static final String ID = "afk";
    public static class_2561 afkTag;
    private final ConcurrentHashMap<UUID, PlayerActivityState> playerActivityStates = new ConcurrentHashMap<>();
    private int absentTimeTrigger;

    private Locale locale;
    private AfkConfig config;

    public AfkModule() {
        super(ID);
        Solstice.configManager.registerData(ID, AfkConfig.class, AfkConfig::new);
        Solstice.playerData.registerData(ID, AfkPlayerData.class, AfkPlayerData::new);
        Solstice.localeManager.registerModule(ID, AfkLocale.MODULE);

        this.commands.add(new AfkCommand(this));

        SolsticeEvents.READY.register((instance, server) -> register());
    }

    private void load() {
        loadConfig();
    }

    private void register() {
        locale = Solstice.localeManager.getLocale(ID);
        config = Solstice.configManager.getData(AfkConfig.class);

        load();

        Placeholders.register(class_2960.method_60655(Solstice.MOD_ID, "afk"), (context, argument) -> {
            if (!context.hasPlayer())
                return PlaceholderResult.invalid("No player!");
            var player = context.player();
            if (isPlayerAfk(player.method_5667())) {
                return PlaceholderResult.value(afkTag);
            } else {
                return PlaceholderResult.value("");
            }
        });

        if (!config.enable)
            return;

        PlayerActivityEvents.AFK.register((player, server) -> {
            Solstice.LOGGER.info("{} is AFK. Active time: {} seconds.", player.method_7334().getName(), getActiveTime(player));
            if (!config.announce)
                return;

            var playerContext = PlaceholderContext.of(player);

            Solstice.getInstance().broadcast(locale.get("goneAfk", playerContext));
        });

        PlayerActivityEvents.AFK_RETURN.register((player, server) -> {
            Solstice.LOGGER.info("{} is no longer AFK. Active time: {} seconds.", player.method_7334().getName(), getActiveTime(player));
            if (!config.announce)
                return;

            var playerContext = PlaceholderContext.of(player);

            Solstice.getInstance().broadcast(locale.get("returnAfk", playerContext));
        });

        SolsticeEvents.RELOAD.register(inst -> load());

        ServerTickEvents.END_SERVER_TICK.register(this::updatePlayers);

        ServerPlayConnectionEvents.JOIN.register((handler, sender, server) -> {
            final var player = handler.method_32311();
            playerActivityStates.put(player.method_5667(), new PlayerActivityState(player, server.method_3780()));
        });

        ServerPlayConnectionEvents.DISCONNECT.register((handler, server) -> {
            updatePlayerActiveTime(handler.method_32311(), server.method_3780());
            playerActivityStates.remove(handler.method_32311().method_5667());
        });

        AttackBlockCallback.EVENT.register((player, world, hand, pos, direction) -> {
            resetAfkState((class_3222) player, world.method_8503());
            return class_1269.field_5811;
        });

        AttackEntityCallback.EVENT.register((player, world, hand, entity, hitResult) -> {
            resetAfkState((class_3222) player, world.method_8503());
            return class_1269.field_5811;
        });

        UseBlockCallback.EVENT.register((player, world, hand, hitResult) -> {
            resetAfkState((class_3222) player, world.method_8503());
            return class_1269.field_5811;
        });

        UseEntityCallback.EVENT.register((player, world, hand, entity, hitResult) -> {
            resetAfkState((class_3222) player, world.method_8503());
            return class_1269.field_5811;
        });

        UseItemCallback.EVENT.register((player, world, hand) -> {
            resetAfkState((class_3222) player, world.method_8503());
            return class_1271.method_22430(player.method_5998(hand));
        });

        ServerMessageEvents.ALLOW_CHAT_MESSAGE.register((message, sender, params) -> {
            resetAfkState(sender, sender.method_5682());
            return true;
        });

        ServerMessageEvents.ALLOW_COMMAND_MESSAGE.register((message, source, params) -> {
            if (!source.method_43737())
                return true;
            resetAfkState(source.method_44023(), source.method_9211());
            return true;
        });
    }

    private void loadConfig() {
        afkTag = Format.parse(config.tag);
        absentTimeTrigger = config.timeTrigger * 20;
    }

    private void updatePlayer(class_3222 player, MinecraftServer server) {
        var currentTick = server.method_3780();
        var playerState = playerActivityStates.computeIfAbsent(player.method_5667(), uuid -> new PlayerActivityState(player, currentTick));

        var oldPosition = playerState.position;
        var newPosition = new ServerPosition(player);
        if (!oldPosition.equals(newPosition)) {
            playerState.position = newPosition;
            resetAfkState(player, server);
            return;
        }

        if (playerState.isAfk)
            return;

        if(!Permissions.check(player, getPermissionNode("base"), true)) {
            return;
        }

        if ((playerState.lastUpdate + absentTimeTrigger) <= currentTick) {
            // player is afk after 5 mins
            updatePlayerActiveTime(player, currentTick);
            playerState.isAfk = true;
            PlayerActivityEvents.AFK.invoker().onAfk(player, server);
        }
    }

    private void updatePlayerActiveTime(class_3222 player, int currentTick) {
        var playerActivityState = playerActivityStates.get(player.method_5667());
        if (!playerActivityState.isAfk) {
            var data = Solstice.playerData.get(player).getData(AfkPlayerData.class);
            var interval = currentTick - playerActivityState.activeStart;
            data.activeTime += interval / 20;
        }
    }

    private void updatePlayers(MinecraftServer server) {
        var players = server.method_3760().method_14571();
        players.forEach(player -> updatePlayer(player, server));
    }

    private void resetAfkState(class_3222 player, MinecraftServer server) {
        if(!Permissions.check(player, getPermissionNode("base"), true)) {
            return;
        }

        if (!playerActivityStates.containsKey(player.method_5667()))
            return;

        var playerState = playerActivityStates.get(player.method_5667());
        playerState.lastUpdate = server.method_3780();
        if (playerState.isAfk) {
            playerState.isAfk = false;
            playerState.activeStart = server.method_3780();
            PlayerActivityEvents.AFK_RETURN.invoker().onAfkReturn(player, server);
        }
    }

    public boolean isPlayerAfk(UUID playerUuid) {
        if (!playerActivityStates.containsKey(playerUuid)) {
            return false;
        }
        return playerActivityStates.get(playerUuid).isAfk;
    }

    public void setPlayerAfk(class_3222 player, boolean afk) {
        if (!playerActivityStates.containsKey(player.method_5667())) {
            return;
        }

        if (afk) {
            playerActivityStates.get(player.method_5667()).lastUpdate = -absentTimeTrigger - 20; // just to be sure
        } else {
            resetAfkState(player, Solstice.server);
        }

        updatePlayer(player, Solstice.server);
    }

    public int getActiveTime(class_3222 player) {
        var data = Solstice.playerData.get(player).getData(AfkPlayerData.class);
        return data.activeTime;
    }
}
