package me.alexdevs.solstice.modules.mail.commands;

import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import eu.pb4.placeholders.api.PlaceholderContext;
import me.alexdevs.solstice.modules.mail.data.PlayerMail;
import net.minecraft.class_2168;
import net.minecraft.class_2561;
import me.alexdevs.solstice.api.command.LocalGameProfile;
import me.alexdevs.solstice.api.module.ModCommand;
import me.alexdevs.solstice.api.text.Components;
import me.alexdevs.solstice.api.text.parser.MarkdownParser;
import me.alexdevs.solstice.core.coreModule.CoreModule;
import me.alexdevs.solstice.modules.ModuleProvider;
import me.alexdevs.solstice.modules.mail.MailModule;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Map;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

public class MailCommand extends ModCommand<MailModule> {
    public MailCommand(MailModule module) {
        super(module);
    }

    @Override
    public List<String> getNames() {
        return List.of("mail");
    }

    @Override
    public LiteralArgumentBuilder<class_2168> command(String name) {
        return method_9247(name)
                .requires(require(true))
                .executes(this::listMails)
                .then(method_9247("send")
                        .then(method_9244("recipient", StringArgumentType.word())
                                .suggests(LocalGameProfile::suggest)
                                .then(method_9244("message", StringArgumentType.greedyString())
                                        .executes(this::sendMail)
                                )
                        )
                )
                .then(method_9247("read")
                        .then(method_9244("index", IntegerArgumentType.integer(0))
                                .executes(this::readMail)))
                .then(method_9247("delete")
                        .then(method_9244("index", IntegerArgumentType.integer(0))
                                .executes(this::deleteMail)));
    }

    private int listMails(CommandContext<class_2168> context) throws CommandSyntaxException {
        var player = context.getSource().method_9207();
        var playerContext = PlaceholderContext.of(player);
        var mails = module.getMailList(player.method_5667());

        if (mails.isEmpty()) {
            context.getSource().method_9226(() -> module.locale().get("emptyMailbox", playerContext), false);
            return 0;
        }

        var output = class_2561.method_43473()
                .method_10852(module.locale().get("mailListHeader", playerContext))
                .method_10852(class_2561.method_30163("\n"));

        for (var i = 0; i < mails.size(); i++) {
            if (i > 0)
                output = output.method_10852(class_2561.method_30163("\n"));

            var mail = mails.get(i);
            var index = i + 1;

            var readButton = Components.button(
                    module.locale().raw("readButton"),
                    module.locale().raw("hoverRead"),
                    "/mail read " + index
            );

            var senderName = CoreModule.getUsername(mail.sender);
            var dateFormatter = new SimpleDateFormat(CoreModule.getConfig().dateTimeFormat);
            var placeholders = Map.of(
                    "index", class_2561.method_30163(String.valueOf(index)),
                    "sender", class_2561.method_30163(senderName),
                    "date", class_2561.method_30163(dateFormatter.format(mail.date)),
                    "readButton", readButton
            );
            output = output.method_10852(module.locale().get("mailListEntry", playerContext, placeholders));
        }

        final var finalOutput = output;

        context.getSource().method_9226(() -> finalOutput, false);

        return 1;
    }

    private int readMail(CommandContext<class_2168> context) throws CommandSyntaxException {
        var player = context.getSource().method_9207();
        var playerContext = PlaceholderContext.of(player);
        var mails = module.getMailList(player.method_5667());
        var index = IntegerArgumentType.getInteger(context, "index") - 1;

        if (index < 0 || index >= mails.size()) {
            context.getSource().method_9226(() -> module.locale().get("notFound"), false);
            return 0;
        }

        var mail = mails.get(index);

        var username = CoreModule.getUsername(mail.sender);

        var replyButton = Components.buttonSuggest(
                module.locale().raw("replyButton"),
                module.locale().raw("hoverReply"),
                "/mail send " + username + " "
        );
        var deleteButton = Components.button(
                module.locale().raw("deleteButton"),
                module.locale().raw("hoverDelete"),
                "/mail delete " + index + 1
        );

        var senderName = CoreModule.getUsername(mail.sender);
        var dateFormatter = new SimpleDateFormat(CoreModule.getConfig().dateTimeFormat);
        var message = MarkdownParser.defaultParser.parseNode(mail.message);
        var placeholders = Map.of(
                "sender", class_2561.method_30163(senderName),
                "date", class_2561.method_30163(dateFormatter.format(mail.date)),
                "message", message.toText(),
                "replyButton", replyButton,
                "deleteButton", deleteButton
        );

        context.getSource().method_9226(() -> module.locale().get("mailDetails", playerContext, placeholders), false);

        return 1;
    }

    private int deleteMail(CommandContext<class_2168> context) throws CommandSyntaxException {
        var player = context.getSource().method_9207();
        var playerContext = PlaceholderContext.of(player);
        var index = IntegerArgumentType.getInteger(context, "index") - 1;

        if (module.deleteMail(player.method_5667(), index)) {
            context.getSource().method_9226(() -> module.locale().get("mailDeleted", playerContext), false);
        } else {
            context.getSource().method_9226(() -> module.locale().get("notFound"), false);
        }

        return 1;
    }

    private int sendMail(CommandContext<class_2168> context) throws CommandSyntaxException {
        var sender = context.getSource().method_9207();
        var recipient = LocalGameProfile.getProfile(context, "recipient");

        var message = StringArgumentType.getString(context, "message");
        var server = context.getSource().method_9211();

        var mail = new PlayerMail(message, sender.method_5667());
        var actuallySent = module.sendMail(recipient.getId(), mail);

        var senderContext = PlaceholderContext.of(sender);

        context.getSource().method_9226(() -> module.locale().get("mailSent", senderContext), false);

        if (actuallySent) {
            var recPlayer = server.method_3760().method_14602(recipient.getId());
            if (recPlayer == null) {
                return 1;
            }

            if (ModuleProvider.IGNORE.isIgnoring(recPlayer, sender))
                return 0;

            var recContext = PlaceholderContext.of(recPlayer);
            recPlayer.method_43496(module.locale().get("mailReceived", recContext));
        }

        return 1;
    }
}
