package me.alexdevs.solstice.modules.note.commands;

import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import me.alexdevs.solstice.api.command.LocalGameProfile;
import me.alexdevs.solstice.api.module.ModCommand;
import me.alexdevs.solstice.api.text.Components;
import me.alexdevs.solstice.api.text.Format;
import me.alexdevs.solstice.core.coreModule.CoreModule;
import me.alexdevs.solstice.modules.note.NoteModule;
import me.alexdevs.solstice.modules.note.data.Note;
import me.lucko.fabric.api.permissions.v0.Permissions;
import net.minecraft.class_2168;
import net.minecraft.class_2561;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

public class NotesCommand extends ModCommand<NoteModule> {
    public NotesCommand(NoteModule module) {
        super(module);
    }

    @Override
    public List<String> getNames() {
        return List.of("notes", "note");
    }

    @Override
    public LiteralArgumentBuilder<class_2168> command(String name) {
        return method_9247(name)
                .requires(require(2))
                .then(method_9244("user", StringArgumentType.word())
                        .suggests(LocalGameProfile::suggest)
                        .executes(this::listNotes)
                        .then(method_9247("add")
                                .requires(require("add", 2))
                                .then(method_9244("message", StringArgumentType.greedyString())
                                        .executes(this::addNote)
                                )
                        )
                        .then(method_9247("check")
                                .then(method_9244("index", IntegerArgumentType.integer(0))
                                        .executes(this::checkNote)))
                        .then(method_9247("delete")
                                .requires(require("delete", 2))
                                .then(method_9244("index", IntegerArgumentType.integer(0))
                                        .executes(this::deleteNote)))
                        .then(method_9247("clear")
                                .requires(require("clear", 2))
                                .executes(this::clearNotes))
                );
    }

    private int listNotes(CommandContext<class_2168> context) throws CommandSyntaxException {
        var user = LocalGameProfile.getProfile(context, "user");
        var notes = module.getNotes(user.getId());

        if (notes.isEmpty()) {
            context.getSource().method_9226(() -> module.locale().get("emptyNotes"), false);
            return 0;
        }

        var output = class_2561.method_43473()
                .method_10852(module.locale().get("noteListHeader", Map.of(
                        "user", class_2561.method_30163(user.getName())
                )))
                .method_10852(class_2561.method_30163("\n"));

        for (var i = 0; i < notes.size(); i++) {
            if (i > 0)
                output = output.method_10852(class_2561.method_30163("\n"));

            var note = notes.get(i);

            var checkButton = Components.button(
                    module.locale().raw("checkButton"),
                    module.locale().raw("hoverCheck"),
                    "/notes " + user.getName() + " check " + i
            );

            var senderName = CoreModule.getUsername(note.createdBy);
            var dateFormatter = new SimpleDateFormat(CoreModule.getConfig().dateTimeFormat);
            var placeholders = Map.of(
                    "index", class_2561.method_30163(String.valueOf(i)),
                    "operator", class_2561.method_30163(senderName),
                    "date", class_2561.method_30163(dateFormatter.format(note.creationDate)),
                    "message", Format.parse(note.note),
                    "checkButton", checkButton
            );
            output = output.method_10852(module.locale().get("noteListEntry", placeholders));
        }

        final var finalOutput = output;

        context.getSource().method_9226(() -> finalOutput, false);

        return 1;
    }

    private int checkNote(CommandContext<class_2168> context) throws CommandSyntaxException {
        var user = LocalGameProfile.getProfile(context, "user");
        var notes = module.getNotes(user.getId());
        var index = IntegerArgumentType.getInteger(context, "index");

        if (index < 0 || index >= notes.size()) {
            context.getSource().method_9226(() -> module.locale().get("notFound"), false);
            return 0;
        }

        var note = notes.get(index);

        var deleteButton = Components.button(
                module.locale().raw("deleteButton"),
                module.locale().raw("hoverDelete"),
                "/note " + user.getName() + " delete " + index
        );

        var operator = CoreModule.getUsername(note.createdBy);
        var dateFormatter = new SimpleDateFormat(CoreModule.getConfig().dateTimeFormat);
        var placeholders = Map.of(
                "operator", class_2561.method_30163(operator),
                "date", class_2561.method_30163(dateFormatter.format(note.creationDate)),
                "message", Format.parse(note.note),
                "deleteButton", deleteButton
        );

        context.getSource().method_9226(() -> module.locale().get("noteDetails", placeholders), false);

        return 1;
    }

    private int deleteNote(CommandContext<class_2168> context) throws CommandSyntaxException {
        var user = LocalGameProfile.getProfile(context, "user");
        var notes = module.getNotes(user.getId());
        var index = IntegerArgumentType.getInteger(context, "index");

        if (index < notes.size()) {
            notes.remove(index);
            context.getSource().method_9226(() -> module.locale().get("noteDeleted"), false);
        } else {
            context.getSource().method_9226(() -> module.locale().get("notFound"), false);
            return 0;
        }

        return 1;
    }

    private int addNote(CommandContext<class_2168> context) throws CommandSyntaxException {
        var user = LocalGameProfile.getProfile(context, "user");

        UUID operatorId = new UUID(0, 0);
        if (context.getSource().method_43737())
            operatorId = context.getSource().method_44023().method_5667();

        var message = StringArgumentType.getString(context, "message");

        var note = new Note(message, operatorId);
        var notes = module.getNotes(user.getId());

        notes.add(note);
        var index = notes.size() - 1;

        context.getSource().method_9226(() -> module.locale().get("noteAdded"), false);

        var checkButton = Components.button(
                module.locale().raw("checkButton"),
                module.locale().raw("hoverCheck"),
                "/notes " + user.getName() + " check " + index
        );
        final var text = module.locale().get("addedNotification", Map.of(
                "operator", context.getSource().method_9223(),
                "user", class_2561.method_30163(user.getName()),
                "checkButton", checkButton
        ));

        context.getSource().method_9211().method_3760().method_14571().forEach(pl -> {
            if (Permissions.check(pl, getPermissionNode("notify"), 2)) {
                pl.method_43496(text);
            }
        });

        return 1;
    }

    private int clearNotes(CommandContext<class_2168> context) throws CommandSyntaxException {
        var user = LocalGameProfile.getProfile(context, "user");

        var notes = module.getNotes(user.getId());
        notes.clear();

        context.getSource().method_9226(() -> module.locale().get("notesCleared", Map.of(
                "user", class_2561.method_30163(user.getName())
        )), true);

        return 1;
    }
}
