package me.alexdevs.solstice.modules.rtp.core;

import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import me.alexdevs.solstice.Solstice;
import me.alexdevs.solstice.api.ServerLocation;
import me.alexdevs.solstice.modules.rtp.data.RTPConfig;
import net.minecraft.class_1923;
import net.minecraft.class_1959;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2791;
import net.minecraft.class_2818;
import net.minecraft.class_2902;
import net.minecraft.class_3193;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3230;
import net.minecraft.class_5321;
import org.jetbrains.annotations.Nullable;

import java.util.Comparator;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

public class Locator {

    public static final class_3230<class_2338> RTP_TICKET = class_3230.method_20628("rtp", Comparator.comparingLong(class_1923::method_37232), 300);

    public final class_3222 player;
    public final class_3218 world;
    public final RTPConfig config;
    public @Nullable
    final class_5321<class_1959> biome;

    private Consumer<Result> callback;
    private final Stopwatch stopwatch = Stopwatch.createUnstarted();

    private class_2791 chunk;
    private class_2338 attemptPos;
    private boolean failed = false;
    private int remainingAttempts = 0;

    private static final ImmutableList<class_2248> unsafeBlocks = ImmutableList.of(
            class_2246.field_10164,
            class_2246.field_10092,
            class_2246.field_10029,
            class_2246.field_10036,
            class_2246.field_17350,
            class_2246.field_27098,
            class_2246.field_16999,
            class_2246.field_27879
    );

    private static final ImmutableList<class_2248> nonIdealBlocks = ImmutableList.of(
            class_2246.field_10382
    );

    public Locator(class_3222 player, class_3218 world, RTPConfig config) {
        this(player, world, config, null);
    }

    public Locator(class_3222 player, class_3218 world, RTPConfig config, @Nullable class_5321<class_1959> biome) {
        this.player = player;
        this.world = world;
        this.config = config;
        this.biome = biome;
        this.remainingAttempts = config.attempts;
    }

    public void locate(Consumer<Result> callback) {
        this.callback = callback;
        stopwatch.start();
        attempt();
    }

    private void attempt() {
        for (var i = config.attempts; i >= 0; i--) {
            this.remainingAttempts = i;
            var pos = getRandomPos();

            if (isValid(pos)) {
                Solstice.LOGGER.info("RTP spot found at attempt {} for {}", config.attempts - remainingAttempts, player.method_5477());
                attemptPos = pos;
                load();
                return;
            }
        }

        failed = true;
        callback.accept(new Result(Result.Type.TOO_MANY_ATTEMPTS, Optional.empty()));
    }

    public boolean tick() {
        if (failed) return true;

        if (stopwatch.elapsed(TimeUnit.MILLISECONDS) >= config.timeout) {
            callback.accept(new Result(Result.Type.TIMEOUT, Optional.empty()));
            return true;
        }

        // Not yet started
        if (attemptPos == null) {
            return false;
        }

        var chunk = getChunk(new class_1923(attemptPos));
        if (chunk.isPresent()) {
            this.chunk = chunk.get();
            findValidPlacement();
            return true;
        }

        return false;
    }

    private class_2338 getTopBlock(class_2338 pos) {
        return world.method_8598(class_2902.class_2903.field_13203, pos);
    }

    private class_2338 getEmptySpace(class_2338 pos) {
        var bottom = chunk.method_31607();
        var top = world.method_32819();
        var blockPos = new class_2338.class_2339(pos.method_10263(), top, pos.method_10260());

        var isAir = false;
        var isAirBelow = false;
        while (blockPos.method_10264() >= bottom && isAirBelow || !isAir) {
            isAir = isAirBelow;
            isAirBelow = chunk.method_8320(blockPos.method_10098(class_2350.field_11033)).method_26215();
        }
        return blockPos.method_10084().method_10062();
    }

    private void findValidPlacement() {
        class_2338 pos = attemptPos;
        for (var i = 0; i <= 256; i++) {
            if (world.method_8597().comp_643()) {
                pos = getEmptySpace(pos);
            } else {
                pos = getTopBlock(pos);
            }
            var bs = chunk.method_8320(pos);
            var bsBelow = chunk.method_8320(pos.method_10074());
            if (!unsafeBlocks.contains(bs.method_26204()) && !unsafeBlocks.contains(bsBelow.method_26204())) {
                break;
            }

            var dx = i % 16;
            var dz = i / 16;
            pos = chunk.method_12004().method_35231(dx, chunk.method_31607(), dz);
        }

        if (pos.method_10264() <= chunk.method_31607()) {
            callback.accept(new Result(Result.Type.UNSAFE, Optional.empty()));
            return;
        }

        var vec = pos.method_46558();

        callback.accept(new Result(Result.Type.SUCCESS, Optional.of(new ServerLocation(
                vec.method_10216(), vec.method_10214(), vec.method_10215(), 0, 0, world
        ))));
    }

    private void load() {
        world.method_14178().method_17297(RTP_TICKET, new class_1923(attemptPos), 0, attemptPos);
    }

    private Optional<class_2818> getChunk(class_1923 pos) {
        var holder = world.method_14178().method_14131(pos.method_8324());
        if (holder == null) {
            return Optional.empty();
        } else {
            var chunk = holder.method_20725().getNow(class_3193.field_16427).left().orElse(null);
            if (chunk == null) {
                return Optional.empty();
            }
            return Optional.of(chunk);
        }
    }

    public boolean isValid(class_2338 pos) {
        if (pos == null)
            return false;

        if (this.biome != null) {
            return isInBiome(pos, this.biome);
        }

        var biome = world.method_23753(pos);
        return !config.parseBiomes().contains(biome.method_40230().orElse(null));
    }

    public boolean isInBiome(class_2338 pos, class_5321<class_1959> biome) {
        var biomeAtPos = world.method_23753(pos);
        return biomeAtPos.method_40230().get().equals(biome);
    }

    public class_2338 getRandomPos() {
        var worldBorder = world.method_8621();
        var size = worldBorder.method_11965();

        double centerX, centerZ;
        if (config.aroundPlayer) {
            centerX = player.method_23317();
            centerZ = player.method_23321();
        } else {
            centerX = worldBorder.method_11964();
            centerZ = worldBorder.method_11980();
        }

        var maxDiameter = config.maxRadius * 2;
        var minDiameter = config.minRadius * 2;

        var max = Math.min((int) size, maxDiameter);
        var min = Math.max(0, minDiameter);

        int x = 0;
        int z = 0;
        var limit = 256;
        for (var i = 0; i <= limit; i++) {
            var dist = world.method_8409().method_43058() * (max - min) + min;
            var angle = world.method_8409().method_43058() * Math.PI * 2d;
            x = (int) (Math.cos(angle) * dist + centerX);
            z = (int) (Math.sin(angle) * dist + centerZ);

            if (worldBorder.method_35317(x, z))
                break;

            if (i == limit) {
                return null;
            }
        }

        return new class_2338(x, world.method_32819(), z);
    }

    public record Result(Type type, Optional<ServerLocation> position) {
        public enum Type {
            SUCCESS,
            TOO_MANY_ATTEMPTS,
            TIMEOUT,
            UNSAFE
        }
    }
}
