package me.alexdevs.solstice.modules.kit.commands;

import com.mojang.brigadier.arguments.BoolArgumentType;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.suggestion.Suggestions;
import com.mojang.brigadier.suggestion.SuggestionsBuilder;
import eu.pb4.sgui.api.gui.SimpleGui;
import me.alexdevs.solstice.Solstice;
import me.alexdevs.solstice.api.command.TimeSpan;
import me.alexdevs.solstice.api.module.ModCommand;
import me.alexdevs.solstice.modules.kit.KitInventory;
import me.alexdevs.solstice.modules.kit.KitModule;
import me.alexdevs.solstice.modules.kit.Utils;
import me.alexdevs.solstice.modules.kit.data.KitPlayerData;
import net.minecraft.class_2168;
import net.minecraft.class_2172;
import net.minecraft.class_2561;
import net.minecraft.class_3917;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

public class KitCommand extends ModCommand<KitModule> {
    public KitCommand(KitModule module) {
        super(module);
    }

    @Override
    public List<String> getNames() {
        return List.of("kit");
    }

    @Override
    public LiteralArgumentBuilder<class_2168> command(String name) {
        return method_9247(name)
                .requires(require(true))
                .then(method_9247("list")
                        .executes(this::listKits))
                .then(method_9247("claim")
                        .then(method_9244("name", StringArgumentType.word())
                                .suggests(this::suggestKitList)
                                .executes(this::claimKit))
                )
                .then(method_9247("create")
                        .requires(require("set", 3))
                        .then(method_9244("name", StringArgumentType.word())
                                .executes(this::createKit))
                )
                .then(method_9247("delete")
                        .requires(require("set", 3))
                        .then(method_9244("name", StringArgumentType.word())
                                .suggests(this::suggestAllKits)
                                .executes(this::deleteKit))
                )
                .then(method_9247("edit")
                        .requires(require("set", 3))
                        .then(method_9244("name", StringArgumentType.word())
                                .suggests(this::suggestKitList)
                                .executes(this::editKit))
                )
                .then(method_9247("set")
                        .requires(require("set", 3))
                        .then(method_9244("name", StringArgumentType.word())
                                .suggests(this::suggestAllKits)
                                .then(method_9247("first-join")
                                        .then(method_9244("enable", BoolArgumentType.bool())
                                                .executes(this::setFirstJoin)
                                        ))
                                .then(method_9247("cooldown")
                                        .then(method_9244("timespan", StringArgumentType.word())
                                                .suggests(TimeSpan::suggest)
                                                .executes(this::setCooldown)
                                        ))
                                .then(method_9247("one-time")
                                        .then(method_9244("enable", BoolArgumentType.bool())
                                                .executes(this::setOneTime)
                                        ))
                                .then(method_9247("icon")
                                        .executes(this::setIcon))
                        )
                );
    }

    private int listKits(CommandContext<class_2168> context) throws CommandSyntaxException {
        var player = context.getSource().method_9207();
        var kits = module.getPlayerKitNames(player);

        if(kits.isEmpty()) {
            context.getSource().method_9226(() -> module.locale().get("listNoKits"), false);
            return 0;
        }

        var comma = module.locale().get("listComma");
        var items = class_2561.method_43473();
        for (var i = 0; i < kits.size(); i++) {
            var kit = kits.get(i);
            if (module.couldClaimKit(player, kit)) {
                items.method_10852(module.locale().get("listAvailableKit", Map.of(
                        "kit", class_2561.method_30163(kit)
                )));
            } else {
                items.method_10852(module.locale().get("listUnavailableKit", Map.of(
                        "kit", class_2561.method_30163(kit)
                )));
            }

            if (i < kits.size() - 1) {
                items.method_10852(comma);
            }
        }

        var list = module.locale().get("listHeader", Map.of(
                "list", items
        ));

        context.getSource().method_9226(() -> list, false);

        return 1;
    }

    private int claimKit(CommandContext<class_2168> context) throws CommandSyntaxException {
        var source = context.getSource();
        var player = source.method_9207();
        var name = StringArgumentType.getString(context, "name");

        if (!module.hasKitPermission(player, name)) {
            source.method_9226(() -> module.locale().get("noPermission", Map.of("kit", class_2561.method_30163(name))), false);
            return 0;
        }

        var kits = module.getKits();
        if (!kits.containsKey(name)) {
            source.method_9226(() -> module.locale().get("notFound", Map.of("kit", class_2561.method_30163(name))), false);
            return 0;
        }

        var kit = kits.get(name);

        var playerData = Solstice.playerData.get(player).getData(KitPlayerData.class);
        if (kit.oneTime && playerData.claimedKits.containsKey(name)) {
            source.method_9226(() -> module.locale().get("alreadyClaimed", Map.of("kit", class_2561.method_30163(name))), false);
            return 0;
        }

        if (kit.cooldownSeconds > 0) {
            if (playerData.claimedKits.containsKey(name)) {
                var startDate = playerData.claimedKits.get(name);
                var nowDate = new Date();

                var delta = (nowDate.getTime() - startDate.getTime()) / 1000;
                if (delta < kit.cooldownSeconds) {
                    var remaining = kit.cooldownSeconds - delta;

                    var timespan = TimeSpan.toShortString((int) remaining);
                    source.method_9226(() -> module.locale().get("onCooldown", Map.of(
                            "kit", class_2561.method_30163(name),
                            "timespan", class_2561.method_30163(timespan)
                    )), false);
                    return 0;
                }
            }
        }

        module.claimKit(player, name);

        source.method_9226(() -> module.locale().get("claimed", Map.of("kit", class_2561.method_30163(name))), false);

        return 1;
    }

    private int createKit(CommandContext<class_2168> context) throws CommandSyntaxException {
        var source = context.getSource();
        var player = source.method_9207();
        var name = StringArgumentType.getString(context, "name");

        if (module.getKits().containsKey(name)) {
            source.method_9226(() -> module.locale().get("alreadyExists"), false);
            return 0;
        }

        var kitInventory = new KitInventory();
        var container = new SimpleGui(class_3917.field_17326, player, false) {
            @Override
            public void onClose() {
                if (module.createKit(name, Utils.getItemStacks(kitInventory))) {
                    source.method_9226(() -> module.locale().get("created", Map.of("kit", class_2561.method_30163(name))), true);
                } else {
                    source.method_9226(() -> module.locale().get("alreadyExists"), false);
                }
            }
        };

        Utils.redirect(container, kitInventory);
        container.setTitle(module.locale().get("newKitTitle", Map.of("kit", class_2561.method_30163(name))));
        container.open();

        return 1;
    }

    private int deleteKit(CommandContext<class_2168> context) {
        var name = StringArgumentType.getString(context, "name");
        if (module.getKits().remove(name) != null) {
            context.getSource().method_9226(() -> module.locale().get("deleted", Map.of("kit", class_2561.method_30163(name))), true);
        } else {
            context.getSource().method_9226(() -> module.locale().get("notFound", Map.of("kit", class_2561.method_30163(name))), false);
        }

        return 1;
    }

    private int editKit(CommandContext<class_2168> context) throws CommandSyntaxException {
        var source = context.getSource();
        var player = source.method_9207();
        var name = StringArgumentType.getString(context, "name");

        var kits = module.getKits();
        if (!kits.containsKey(name)) {
            context.getSource().method_9226(() -> module.locale().get("notFound", Map.of("kit", class_2561.method_30163(name))), false);
            return 0;
        }

        var kit = kits.get(name);
        var kitInventory = Utils.createInventory(kit.getItemStacks());

        var container = new SimpleGui(class_3917.field_17326, player, false) {
            @Override
            public void onClose() {
                var items = Utils.getItemStacks(kitInventory);
                kit.itemStacks = items.stream().map(Utils::serializeItemStack).toList();
                source.method_9226(() -> module.locale().get("edited", Map.of("kit", class_2561.method_30163(name))), true);
            }
        };

        Utils.redirect(container, kitInventory);
        container.setTitle(module.locale().get("editKitTitle", Map.of("kit", class_2561.method_30163(name))));
        container.open();

        return 1;
    }

    private int setFirstJoin(CommandContext<class_2168> context) {
        var name = StringArgumentType.getString(context, "name");
        var enable = BoolArgumentType.getBool(context, "enable");

        var kits = module.getKits();
        if (!kits.containsKey(name)) {
            context.getSource().method_9226(() -> module.locale().get("notFound", Map.of("kit", class_2561.method_30163(name))), false);
            return 0;
        }

        var kit = kits.get(name);
        kit.firstJoin = enable;

        context.getSource().method_9226(() -> module.locale().get("setFirstJoin", Map.of(
                "kit", class_2561.method_30163(name),
                "value", class_2561.method_30163(String.valueOf(enable))
        )), true);

        return 1;
    }

    private int setCooldown(CommandContext<class_2168> context) throws CommandSyntaxException {
        var name = StringArgumentType.getString(context, "name");
        var timespan = TimeSpan.getTimeSpan(context, "timespan");

        var kits = module.getKits();
        if (!kits.containsKey(name)) {
            context.getSource().method_9226(() -> module.locale().get("notFound", Map.of("kit", class_2561.method_30163(name))), false);
            return 0;
        }

        var kit = kits.get(name);
        kit.cooldownSeconds = timespan;

        context.getSource().method_9226(() -> module.locale().get("setCooldown", Map.of(
                "kit", class_2561.method_30163(name),
                "value", class_2561.method_30163(TimeSpan.toShortString(timespan))
        )), true);

        return 1;
    }

    private int setOneTime(CommandContext<class_2168> context) {
        var name = StringArgumentType.getString(context, "name");
        var enable = BoolArgumentType.getBool(context, "enable");

        var kits = module.getKits();
        if (!kits.containsKey(name)) {
            context.getSource().method_9226(() -> module.locale().get("notFound", Map.of("kit", class_2561.method_30163(name))), false);
            return 0;
        }

        var kit = kits.get(name);
        kit.oneTime = enable;

        context.getSource().method_9226(() -> module.locale().get("setOneTime", Map.of(
                "kit", class_2561.method_30163(name),
                "value", class_2561.method_30163(String.valueOf(enable))
        )), true);

        return 1;
    }

    private int setIcon(CommandContext<class_2168> context) throws CommandSyntaxException {
        var name = StringArgumentType.getString(context, "name");
        var player = context.getSource().method_9207();

        var kits = module.getKits();
        if (!kits.containsKey(name)) {
            context.getSource().method_9226(() -> module.locale().get("notFound", Map.of("kit", class_2561.method_30163(name))), false);
            return 0;
        }

        var hand = player.method_6058();
        var stack = player.method_5998(hand).method_7972();
        if(stack.method_7960()) {
            context.getSource().method_9226(() -> module.locale().get("noStackInHand"), false);
            return 0;
        }

        var kit = kits.get(name);
        kit.icon = Utils.serializeItemStack(stack);

        context.getSource().method_9226(() -> module.locale().get("setIcon", Map.of(
                "kit", class_2561.method_30163(name)
        )), true);

        return 1;
    }

    private CompletableFuture<Suggestions> suggestAllKits(CommandContext<class_2168> context, SuggestionsBuilder builder) {
        var kits = module.getKits().keySet();
        return class_2172.method_9265(kits, builder);
    }

    private CompletableFuture<Suggestions> suggestKitList(CommandContext<class_2168> context, SuggestionsBuilder builder) throws CommandSyntaxException {
        var player = context.getSource().method_9207();
        return class_2172.method_9265(module.getPlayerKitNames(player), builder);
    }
}
