// SPDX-FileCopyrightText: 2023 klikli-dev
//
// SPDX-License-Identifier: MIT

package com.klikli_dev.theurgy.content.recipe;

import com.klikli_dev.theurgy.content.recipe.input.ItemHandlerRecipeInput;
import com.klikli_dev.theurgy.registry.ItemRegistry;
import com.klikli_dev.theurgy.registry.RecipeSerializerRegistry;
import com.klikli_dev.theurgy.registry.RecipeTypeRegistry;
import com.mojang.serialization.Codec;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import net.minecraft.core.HolderLookup;
import net.minecraft.core.NonNullList;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.codec.ByteBufCodecs;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.crafting.Ingredient;
import net.minecraft.world.item.crafting.Recipe;
import net.minecraft.world.item.crafting.RecipeSerializer;
import net.minecraft.world.item.crafting.RecipeType;
import net.minecraft.world.level.Level;
import org.jetbrains.annotations.NotNull;


public class CatalysationRecipe implements Recipe<ItemHandlerRecipeInput> {
    public static final int DEFAULT_MERCURY_FLUX_PER_TICK = 20;

    public static final MapCodec<CatalysationRecipe> CODEC = RecordCodecBuilder.mapCodec(instance -> instance.group(
                    Ingredient.CODEC.fieldOf("ingredient").forGetter(r -> r.ingredient),
                    Codec.INT.fieldOf("totalMercuryFlux").forGetter(r -> r.totalMercuryFlux),
                    Codec.INT.fieldOf("mercuryFluxPerTick").forGetter(r -> r.mercuryFluxPerTick)
            ).apply(instance, CatalysationRecipe::new)
    );

    public static final StreamCodec<RegistryFriendlyByteBuf, CatalysationRecipe> STREAM_CODEC = StreamCodec.composite(
            Ingredient.CONTENTS_STREAM_CODEC,
            r -> r.ingredient,
            ByteBufCodecs.INT,
            r -> r.totalMercuryFlux,
            ByteBufCodecs.INT,
            r -> r.mercuryFluxPerTick,
            CatalysationRecipe::new
    );
    protected final Ingredient ingredient;

    /**
     * The amount of mercury flux to generate per tick, until the total amount has been reached
     */
    protected final int mercuryFluxPerTick;

    /**
     * The total amount of mercury flux generated by consuming the ingredient.
     */
    protected final int totalMercuryFlux;


    public CatalysationRecipe(Ingredient ingredient, int totalMercuryFlux, int mercuryFluxPerTick) {
        this.ingredient = ingredient;
        this.totalMercuryFlux = totalMercuryFlux;
        this.mercuryFluxPerTick = mercuryFluxPerTick;
    }

    public Ingredient getIngredient() {
        return this.ingredient;
    }

    public int getMercuryFluxPerTick() {
        return this.mercuryFluxPerTick;
    }

    public int getTotalMercuryFlux() {
        return this.totalMercuryFlux;
    }

    @Override
    public boolean isSpecial() {
        return true;
    }

    @Override
    public @NotNull RecipeType<?> getType() {
        return RecipeTypeRegistry.CATALYSATION.get();
    }

    @Override
    public boolean matches(@NotNull ItemHandlerRecipeInput pContainer, @NotNull Level pLevel) {
        var stack = pContainer.getItem(0);
        return this.ingredient.test(stack);
    }

    @Override
    public @NotNull ItemStack assemble(@NotNull ItemHandlerRecipeInput pCraftingContainer, HolderLookup.@NotNull Provider pRegistries) {
        return ItemStack.EMPTY;
    }

    @Override
    public boolean canCraftInDimensions(int pWidth, int pHeight) {
        return true;
    }

    @Override
    public @NotNull ItemStack getResultItem(HolderLookup.@NotNull Provider pRegistries) {
        return ItemStack.EMPTY;
    }

    @Override
    public @NotNull NonNullList<Ingredient> getIngredients() {
        return NonNullList.of(Ingredient.EMPTY, this.ingredient);
    }

    @Override
    public @NotNull ItemStack getToastSymbol() {
        return new ItemStack(ItemRegistry.MERCURY_CATALYST.get());
    }

    @Override
    public @NotNull RecipeSerializer<?> getSerializer() {
        return RecipeSerializerRegistry.CATALYSATION.get();
    }

    public static class Serializer implements RecipeSerializer<CatalysationRecipe> {

        @Override
        public @NotNull MapCodec<CatalysationRecipe> codec() {
            return CODEC;
        }

        @Override
        public @NotNull StreamCodec<RegistryFriendlyByteBuf, CatalysationRecipe> streamCodec() {
            return STREAM_CODEC;
        }
    }
}
