package com.momosoftworks.coldsweat.api.temperature.modifier;

import com.alcatrazescapee.primalwinter.Config;
import com.mojang.datafixers.util.Pair;
import com.momosoftworks.coldsweat.api.util.Temperature;
import com.momosoftworks.coldsweat.config.ConfigSettings;
import com.momosoftworks.coldsweat.compat.CompatManager;
import com.momosoftworks.coldsweat.data.codec.configuration.BiomeTempData;
import com.momosoftworks.coldsweat.data.codec.configuration.DimensionTempData;
import com.momosoftworks.coldsweat.data.codec.configuration.StructureTempData;
import com.momosoftworks.coldsweat.util.math.CSMath;
import com.momosoftworks.coldsweat.util.world.WorldHelper;
import net.minecraft.entity.LivingEntity;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.DimensionType;
import net.minecraft.world.World;
import net.minecraft.world.biome.Biome;
import net.minecraft.world.gen.feature.StructureFeature;

import java.util.Optional;
import java.util.function.Function;

public class BiomeTempModifier extends TempModifier
{
    public BiomeTempModifier()
    {
        this(16);
    }

    public BiomeTempModifier(int samples)
    {   this.getNBT().putInt("Samples", samples);
    }

    @Override
    public Function<Double, Double> calculate(LivingEntity entity, Temperature.Trait trait)
    {
        int samples = this.getNBT().getInt("Samples");
        double worldTemp = 0;
        World level = entity.level;
        DimensionType dimension = level.dimensionType();
        BlockPos entPos = entity.blockPosition();

        // If a structure temperature override is defined, return
        Pair<Double, Double> structureTemp = getStructureTemp(entity.level, entity.blockPosition());
        if (structureTemp.getFirst() != null)
        {   return temp -> structureTemp.getFirst();
        }

        // If the dimension temperature is overridden, return
        DimensionTempData dimTempOverride = ConfigSettings.DIMENSION_TEMPS.get(level.registryAccess()).get(level.dimensionType());
        if (dimTempOverride != null)
        {   return temp -> temp + dimTempOverride.getTemperature();
        }

        DimensionTempData dimTempOffset = ConfigSettings.DIMENSION_OFFSETS.get(level.registryAccess()).get(level.dimensionType());
        double dimOffset = dimTempOffset != null ? dimTempOffset.getTemperature() : 0;

        int biomeCount = 0;
        for (BlockPos blockPos : WorldHelper.getPositionGrid(entPos, samples, 10))
        {
            // Get the holder for the biome
            Biome biome = level.getBiomeManager().getBiome(blockPos);

            if (!dimension.hasCeiling())
            {
                BiomeTempData biomeTempData = ConfigSettings.BIOME_TEMPS.get(level.registryAccess()).get(biome);
                if (CSMath.getIfNotNull(biomeTempData, BiomeTempData::isDisabled, false))
                {   continue;
                }
                // Biome temp with time of day
                double biomeTemp = WorldHelper.getBiomeTemperature(level, biome);

                // Primal Winter compat
                if (CompatManager.isPrimalWinterLoaded() && biomeTempData != null)
                {
                    boolean isWinterBiome = Config.COMMON.isWinterBiome(biome.getRegistryName());
                    boolean isWinterDimension = Config.COMMON.isWinterDimension(level.dimension().getRegistryName());
                    if (isWinterBiome && isWinterDimension)
                    {   biomeTemp = Math.min(biomeTemp, biomeTemp / 2);
                    }
                }
                worldTemp += biomeTemp;
            }
            // If dimension has ceiling (don't use time)
            else worldTemp += CSMath.averagePair(WorldHelper.getBiomeTemperatureRange(level, biome));

            // Tally number of biomes
            biomeCount++;
        }

        worldTemp /= Math.max(1, biomeCount);

        // Add structure offset, if present
        worldTemp += structureTemp.getSecond();

        double finalWorldTemp = worldTemp;
        return temp -> temp + finalWorldTemp + dimOffset;
    }

    public static Pair<Double, Double> getStructureTemp(World level, BlockPos pos)
    {
        Optional<StructureFeature<?, ?>> structure = WorldHelper.getStructureAt(level, pos);
        if (!structure.isPresent()) return Pair.of(null, 0d);

        Double strucTemp = CSMath.getIfNotNull(ConfigSettings.STRUCTURE_TEMPS.get(level.registryAccess()).get(structure.get()), StructureTempData::getTemperature, null);
        Double strucOffset = CSMath.getIfNotNull(ConfigSettings.STRUCTURE_OFFSETS.get(level.registryAccess()).get(structure.get()), StructureTempData::getTemperature, 0d);

        return Pair.of(strucTemp, strucOffset);
    }
}