package com.momosoftworks.coldsweat.data.codec.requirement;

import com.mojang.datafixers.util.Either;
import com.mojang.serialization.Codec;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import com.momosoftworks.coldsweat.data.codec.impl.RequirementHolder;
import com.momosoftworks.coldsweat.data.codec.requirement.sub_type.EntitySubRequirement;
import com.momosoftworks.coldsweat.data.codec.util.IntegerBounds;
import com.momosoftworks.coldsweat.data.codec.util.NegatableList;
import com.momosoftworks.coldsweat.util.entity.EntityHelper;
import net.minecraft.advancements.AdvancementProgress;
import net.minecraft.advancements.CriterionProgress;
import net.minecraft.entity.Entity;
import net.minecraft.entity.player.PlayerEntity;
import net.minecraft.entity.player.ServerPlayerEntity;
import net.minecraft.entity.projectile.ProjectileHelper;
import net.minecraft.stats.Stat;
import net.minecraft.stats.StatType;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.EntityRayTraceResult;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.vector.Vector3d;
import net.minecraft.util.registry.Registry;
import net.minecraft.world.GameType;
import net.minecraft.world.World;

import java.util.Map;
import java.util.Optional;

public class PlayerDataRequirement implements EntitySubRequirement, RequirementHolder
{
    private final Optional<GameType> gameType;
    private final NegatableList<StatRequirement> stats;
    private final Optional<Map<ResourceLocation, Boolean>> recipes;
    private final Optional<Map<ResourceLocation, Either<AdvancementCompletionRequirement, AdvancementCriteriaRequirement>>> advancements;
    private final EntityRequirement lookingAt;

    @Override
    public MapCodec<? extends EntitySubRequirement> getCodec()
    {   return getCodec(EntityRequirement.getCodec());
    }

    public static MapCodec<PlayerDataRequirement> getCodec(Codec<EntityRequirement> lastCodec)
    {
        return RecordCodecBuilder.mapCodec(instance -> instance.group(
                Codec.STRING.xmap(GameType::byName, GameType::getName).optionalFieldOf("game_mode").forGetter(PlayerDataRequirement::gameType),
                NegatableList.listCodec(StatRequirement.CODEC).optionalFieldOf("stats", new NegatableList<>()).forGetter(PlayerDataRequirement::stats),
                Codec.unboundedMap(ResourceLocation.CODEC, Codec.BOOL).optionalFieldOf("recipes").forGetter(PlayerDataRequirement::recipes),
                Codec.unboundedMap(ResourceLocation.CODEC, Codec.either(AdvancementCompletionRequirement.CODEC, AdvancementCriteriaRequirement.CODEC)).optionalFieldOf("advancements").forGetter(PlayerDataRequirement::advancements),
                lastCodec.optionalFieldOf("looking_at", EntityRequirement.NONE).forGetter(PlayerDataRequirement::lookingAt)
        ).apply(instance, PlayerDataRequirement::new));
    }

    public PlayerDataRequirement(Optional<GameType> gameType, NegatableList<StatRequirement> stats,
                                 Optional<Map<ResourceLocation, Boolean>> recipes,
                                 Optional<Map<ResourceLocation, Either<AdvancementCompletionRequirement, AdvancementCriteriaRequirement>>> advancements,
                                 EntityRequirement lookingAt)
    {
        this.gameType = gameType;
        this.stats = stats;
        this.recipes = recipes;
        this.advancements = advancements;
        this.lookingAt = lookingAt;
    }

    public Optional<GameType> gameType()
    {   return gameType;
    }
    public NegatableList<StatRequirement> stats()
    {   return stats;
    }
    public Optional<Map<ResourceLocation, Boolean>> recipes()
    {   return recipes;
    }
    public Optional<Map<ResourceLocation, Either<AdvancementCompletionRequirement, AdvancementCriteriaRequirement>>> advancements()
    {   return advancements;
    }
    public EntityRequirement lookingAt()
    {   return lookingAt;
    }

    @Override
    public boolean test(Entity entity)
    {
        if (!(entity instanceof PlayerEntity)) return false;
        PlayerEntity player = (PlayerEntity) entity;
        ServerPlayerEntity serverPlayer = EntityHelper.getServerPlayer(player);

        if (gameType.isPresent() && EntityHelper.getGameModeForPlayer(player) != gameType.get())
        {   return false;
        }
        if (!stats.test(stat -> stat.test(stat.stat(), serverPlayer.getStats().getValue(stat.stat()))))
        {   return false;
        }
        if (recipes.isPresent())
        {
            for (Map.Entry<ResourceLocation, Boolean> entry : recipes.get().entrySet())
            {
                if (serverPlayer.getRecipeBook().contains(entry.getKey()) != entry.getValue())
                {   return false;
                }
            }
        }
        if (advancements.isPresent())
        {
            for (Map.Entry<ResourceLocation, Either<AdvancementCompletionRequirement, AdvancementCriteriaRequirement>> entry : advancements.get().entrySet())
            {
                AdvancementProgress progress = serverPlayer.getAdvancements().getOrStartProgress(serverPlayer.getServer().getAdvancements().getAdvancement(entry.getKey()));
                if (entry.getValue().map(complete -> complete.test(progress), criteria -> criteria.test(progress)))
                {   return false;
                }
            }
        }
        if (lookingAt != EntityRequirement.NONE)
        {
            Vector3d vec3 = player.getEyePosition(0);
            Vector3d vec31 = player.getViewVector(1.0F);
            Vector3d vec32 = vec3.add(vec31.x * 100.0D, vec31.y * 100.0D, vec31.z * 100.0D);
            EntityRayTraceResult entityhitresult = ProjectileHelper.getEntityHitResult(player.level, player, vec3, vec32, (new AxisAlignedBB(vec3, vec32)).inflate(1.0D), (ent) -> !ent.isSpectator());
            if (entityhitresult == null || entityhitresult.getType() != RayTraceResult.Type.ENTITY)
            {   return false;
            }

            Entity hitEntity = entityhitresult.getEntity();
            if (!this.lookingAt.test(hitEntity))
            {   return false;
            }
        }
        return true;
    }

    @Override
    public boolean test(Entity entity, World level, Vector3d position)
    {   return test(entity);
    }

    @Override
    public String toString()
    {   return CODEC.encodeStart(JsonOps.INSTANCE, this).result().map(Object::toString).orElse("serialize_failed");
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj) return true;
        if (obj == null || getClass() != obj.getClass()) return false;

        PlayerDataRequirement that = (PlayerDataRequirement) obj;
        return gameType.equals(that.gameType)
            && stats.equals(that.stats)
            && recipes.equals(that.recipes)
            && advancements.equals(that.advancements)
            && lookingAt.equals(that.lookingAt);
    }

    public static class StatRequirement
    {
        private final StatType<?> type;
        private ResourceLocation statId;
        private final Stat<?> stat;
        private final IntegerBounds value;

        public static final Codec<StatRequirement> CODEC = RecordCodecBuilder.create(instance -> instance.group(
                Registry.STAT_TYPE.fieldOf("type").forGetter(stat -> stat.type),
                ResourceLocation.CODEC.fieldOf("stat").forGetter(stat -> stat.statId),
                IntegerBounds.CODEC.fieldOf("value").forGetter(stat -> stat.value)
        ).apply(instance, StatRequirement::new));

        public StatRequirement(StatType<?> type, ResourceLocation statId, IntegerBounds value)
        {   this(type, statId, (Stat<?>) type.getRegistry().get(statId), value);
        }

        public StatRequirement(StatType<?> type, ResourceLocation statId, Stat<?> stat, IntegerBounds value)
        {
            this.type = type;
            this.statId = statId;
            this.stat = stat;
            this.value = value;
        }

        public StatType<?> type()
        {   return this.type;
        }
        public Stat<?> stat()
        {   return this.stat;
        }
        public IntegerBounds value()
        {   return this.value;
        }

        public boolean test(Stat<?> stat, int value)
        {   return stat.getType() == type && this.stat.equals(stat) && this.value.test(value);
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj) return true;
            if (obj == null || getClass() != obj.getClass()) return false;

            StatRequirement that = (StatRequirement) obj;
            return type.equals(that.type)
                && statId.equals(that.statId)
                && value.equals(that.value);
        }

        @Override
        public String toString()
        {   return CODEC.encodeStart(JsonOps.INSTANCE, this).result().map(Object::toString).orElse("serialize_failed");
        }
    }

    public static class AdvancementCompletionRequirement
    {
        private final Boolean complete;

        public AdvancementCompletionRequirement(Boolean complete)
        {   this.complete = complete;
        }

        public static final Codec<AdvancementCompletionRequirement> CODEC = Codec.BOOL.xmap(AdvancementCompletionRequirement::new, req -> req.complete);

        public boolean test(AdvancementProgress progress)
        {   return progress.isDone() == this.complete;
        }

        @Override
        public String toString()
        {   return CODEC.encodeStart(JsonOps.INSTANCE, this).result().map(Object::toString).orElse("serialize_failed");
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj) return true;
            if (obj == null || getClass() != obj.getClass()) return false;

            AdvancementCompletionRequirement that = (AdvancementCompletionRequirement) obj;
            return complete.equals(that.complete);
        }
    }
    public static class AdvancementCriteriaRequirement
    {
        private final Map<String, Boolean> criteria;

        public AdvancementCriteriaRequirement(Map<String, Boolean> criteria)
        {   this.criteria = criteria;
        }

        public static final Codec<AdvancementCriteriaRequirement> CODEC = Codec.unboundedMap(Codec.STRING, Codec.BOOL).xmap(AdvancementCriteriaRequirement::new, req -> req.criteria);

        public boolean test(AdvancementProgress progress)
        {
            for (Map.Entry<String, Boolean> entry : this.criteria.entrySet())
            {
                CriterionProgress criterionprogress = progress.getCriterion(entry.getKey());
                if (criterionprogress == null || criterionprogress.isDone() != entry.getValue())
                {   return false;
                }
            }
            return true;
        }

        @Override
        public String toString()
        {   return CODEC.encodeStart(JsonOps.INSTANCE, this).result().map(Object::toString).orElse("serialize_failed");
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj) return true;
            if (obj == null || getClass() != obj.getClass()) return false;

            AdvancementCriteriaRequirement that = (AdvancementCriteriaRequirement) obj;
            return criteria.equals(that.criteria);
        }
    }
}
