package com.momosoftworks.coldsweat.client.event;

import com.mojang.blaze3d.systems.RenderSystem;
import com.momosoftworks.coldsweat.ColdSweat;
import com.momosoftworks.coldsweat.api.event.client.RenderFogEvent;
import com.momosoftworks.coldsweat.api.event.vanilla.RenderLevelEvent;
import com.momosoftworks.coldsweat.api.util.Temperature;
import com.momosoftworks.coldsweat.client.gui.Overlays;
import com.momosoftworks.coldsweat.client.renderer.PostProcessShaderManager;
import com.momosoftworks.coldsweat.common.capability.handler.EntityTempManager;
import com.momosoftworks.coldsweat.config.ConfigSettings;
import com.momosoftworks.coldsweat.util.math.CSMath;
import com.momosoftworks.coldsweat.util.registries.ModEffects;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.BufferBuilder;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.client.shader.ShaderUniform;
import net.minecraft.entity.player.PlayerEntity;
import net.minecraft.potion.Effects;
import net.minecraft.util.ResourceLocation;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.client.event.EntityViewRenderEvent;
import net.minecraftforge.client.event.RenderGameOverlayEvent;
import net.minecraftforge.event.TickEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.fml.common.Mod;

@Mod.EventBusSubscriber(Dist.CLIENT)
public class TempEffectsClient
{
    static float BLEND_TEMP = 0;
    static float PREV_X_SWAY = 0;
    static float PREV_Y_SWAY = 0;
    static float X_SWAY_SPEED = 0;
    static float Y_SWAY_SPEED = 0;
    static float X_SWAY_PHASE = 0;
    static float Y_SWAY_PHASE = 0;
    static float TIME_SINCE_NEW_SWAY = 0;

    static double COLD_IMMUNITY = 0;
    static double HOT_IMMUNITY  = 0;

    // Sway the player's camera when the player is too hot; swaying is more drastic at higher temperatures
    @SubscribeEvent
    public static void setCamera(EntityViewRenderEvent.CameraSetup event)
    {
        PlayerEntity player = Minecraft.getInstance().player;
        if (isPlayerInvalid(player)) return;

        if (!Minecraft.getInstance().isPaused())
        {
            // Get the FPS of the game
            float frameTime = Minecraft.getInstance().getDeltaFrameTime();
            // Get a blended version of the player's temperature
            // More important for fog stuff
            BLEND_TEMP = (float) Overlays.BLEND_BODY_TEMP;

            if (ConfigSettings.DISTORTION_EFFECTS.get())
            {
                // Camera "shivers" when temp is < -50
                if (BLEND_TEMP <= -50 && COLD_IMMUNITY < 1)
                {
                    double tickTime = player.tickCount + event.getRenderPartialTicks();
                    float shiverIntensity = CSMath.blend(((float) Math.sin(tickTime / 10) + 1) * 0.03f + 0.01f,
                                                0f, BLEND_TEMP, -100, -50);
                    // Multiply the effect for lower framerates
                    shiverIntensity *= Minecraft.getInstance().getDeltaFrameTime() * 10;
                    shiverIntensity = (float) CSMath.blend(shiverIntensity, 0, COLD_IMMUNITY, 0, 1);
                    float shiverRotation = (float) (Math.sin(tickTime * 2.5) * shiverIntensity);
                    // Rotate camera
                    player.setYHeadRot(player.getYHeadRot() + shiverRotation);
                }
                else if (BLEND_TEMP >= 50 && HOT_IMMUNITY < 1)
                {
                    float factor = CSMath.blend(0, 20, BLEND_TEMP, 50, 100);
                    factor = (float) CSMath.blend(factor, 0, HOT_IMMUNITY, 0, 1);

                    // Set random sway speed every once in a while
                    if (TIME_SINCE_NEW_SWAY > 100 || X_SWAY_SPEED == 0 || Y_SWAY_SPEED == 0)
                    {
                        TIME_SINCE_NEW_SWAY = 0;
                        X_SWAY_SPEED = (float) (Math.random() * 0.003f + 0.004f);
                        Y_SWAY_SPEED = (float) (Math.random() * 0.003f + 0.004f);
                    }
                    TIME_SINCE_NEW_SWAY += frameTime;

                    // Blend to the new sway speed
                    X_SWAY_PHASE += 2 * Math.PI * frameTime * X_SWAY_SPEED;
                    Y_SWAY_PHASE += 2 * Math.PI * frameTime * Y_SWAY_SPEED;

                    // Apply the sway speed to a sin function
                    float xOffs = (float) (Math.sin(X_SWAY_PHASE) * factor);
                    float yOffs = (float) ((Math.sin(Y_SWAY_PHASE) + Math.cos(Y_SWAY_PHASE / 4) * 2) * factor * 3);

                    // Apply the sway
                    player.xRot = player.xRot + xOffs - PREV_X_SWAY;
                    player.yRot = player.yRot + yOffs - PREV_Y_SWAY;

                    // Save the previous sway
                    PREV_X_SWAY = xOffs;
                    PREV_Y_SWAY = yOffs;
                }
                else
                {
                    PREV_X_SWAY = 0;
                    PREV_Y_SWAY = 0;
                    X_SWAY_PHASE = 0;
                    Y_SWAY_PHASE = 0;
                }
            }
        }
    }

    @SubscribeEvent
    public static void onClientTick(TickEvent.ClientTickEvent event)
    {
        if (event.phase == TickEvent.Phase.END)
        {
            PlayerEntity player = Minecraft.getInstance().player;
            if (isPlayerInvalid(player)) return;
            if (player.tickCount % 5 == 0)
            {
                // Set cold immunity
                if (player.hasEffect(ModEffects.ICE_RESISTANCE) && ConfigSettings.ICE_RESISTANCE_ENABLED.get())
                {   COLD_IMMUNITY = 1;
                }
                else COLD_IMMUNITY = Temperature.get(player, Temperature.Trait.COLD_RESISTANCE);
                // Set heat immunity
                if (player.hasEffect(Effects.FIRE_RESISTANCE) && ConfigSettings.FIRE_RESISTANCE_ENABLED.get())
                {   HOT_IMMUNITY = 1;
                }
                else HOT_IMMUNITY  = Temperature.get(player, Temperature.Trait.HEAT_RESISTANCE);
            }
        }
    }

    @SubscribeEvent
    public static void renderFog(EntityViewRenderEvent event)
    {
        if (!(event instanceof RenderFogEvent || event instanceof EntityViewRenderEvent.FogColors)) return;

        PlayerEntity player = Minecraft.getInstance().player;
        if (isPlayerInvalid(player)) return;

        double fogDistance = Math.max(0.01, ConfigSettings.HEATSTROKE_FOG_DISTANCE.get());
        if (fogDistance >= 64) return;
        if (fogDistance < Double.POSITIVE_INFINITY && BLEND_TEMP >= 50 && HOT_IMMUNITY < 1)
        {
            float tempWithResistance = (float) CSMath.blend(BLEND_TEMP, 50, HOT_IMMUNITY, 0, 1);
            if (event instanceof RenderFogEvent)
            {
                RenderFogEvent fog = (RenderFogEvent) event;
                if (fogDistance > (fog.getFarPlaneDistance())) return;
                fog.setFarPlaneDistance(CSMath.blend(fog.getFarPlaneDistance(), (float) fogDistance, tempWithResistance, 50f, 90f));
                fog.setNearPlaneDistance(CSMath.blend(fog.getNearPlaneDistance(), (float) (fogDistance * 0.3), tempWithResistance, 50f, 90f));
                fog.setCanceled(true);
            }
            else if (event instanceof EntityViewRenderEvent.FogColors)
            {
                EntityViewRenderEvent.FogColors fogColor = (EntityViewRenderEvent.FogColors) event;
                fogColor.setRed(CSMath.blend(fogColor.getRed(), 0.01f, tempWithResistance, 50, 90));
                fogColor.setGreen(CSMath.blend(fogColor.getGreen(), 0.01f, tempWithResistance, 50, 90));
                fogColor.setBlue(CSMath.blend(fogColor.getBlue(), 0.05f, tempWithResistance, 50, 90));
            }
        }
    }

    static final ResourceLocation HAZE_TEXTURE = new ResourceLocation(ColdSweat.MOD_ID, "textures/gui/overlay/haze.png");
    static final ResourceLocation FREEZE_TEXTURE = new ResourceLocation(ColdSweat.MOD_ID, "textures/gui/overlay/freeze_overlay.png");

    @SubscribeEvent
    public static void vignette(RenderGameOverlayEvent.Pre event)
    {
        PlayerEntity player = Minecraft.getInstance().player;
        if (isPlayerInvalid(player)) return;

        if (event.getType() == RenderGameOverlayEvent.ElementType.ALL
        && ((BLEND_TEMP > 0 && HOT_IMMUNITY < 1) || (BLEND_TEMP < 0 && COLD_IMMUNITY < 1)))
        {
            float resistance = (float) CSMath.blend(1, 0, BLEND_TEMP > 0 ? HOT_IMMUNITY : COLD_IMMUNITY, 0, 1);
            float opacity = CSMath.blend(0f, 1f, Math.abs(BLEND_TEMP), 50, 100) * resistance;
            float tickTime = player.tickCount + event.getPartialTicks();
            if (opacity == 0) return;
            double width = event.getWindow().getWidth();
            double height = event.getWindow().getHeight();
            double scale = event.getWindow().getGuiScale();

            RenderSystem.enableBlend();
            RenderSystem.defaultBlendFunc();
            if (BLEND_TEMP > 0)
            {   float vignetteBrightness = opacity + ((float) Math.sin((tickTime + 3) / (Math.PI * 1.0132f)) / 5f - 0.2f) * opacity;
                RenderSystem.color4f(0.231f, 0f, 0f, vignetteBrightness);
                Minecraft.getInstance().textureManager.bind(HAZE_TEXTURE);
            }
            else
            {   RenderSystem.color4f(1f, 1f, 1f, opacity);
                Minecraft.getInstance().textureManager.bind(FREEZE_TEXTURE);
            }
            Tessellator tessellator = Tessellator.getInstance();
            BufferBuilder bufferbuilder = tessellator.getBuilder();
            bufferbuilder.begin(7, DefaultVertexFormats.POSITION_TEX);
            bufferbuilder.vertex(0.0D, height / scale, -90.0D).uv(0.0F, 1.0F).endVertex();
            bufferbuilder.vertex(width / scale, height / scale, -90.0D).uv(1.0F, 1.0F).endVertex();
            bufferbuilder.vertex(width / scale, 0.0D, -90.0D).uv(1.0F, 0.0F).endVertex();
            bufferbuilder.vertex(0.0D, 0.0D, -90.0D).uv(0.0F, 0.0F).endVertex();
            tessellator.end();
            RenderSystem.depthMask(true);
            RenderSystem.enableDepthTest();
            RenderSystem.color4f(1.0F, 1.0F, 1.0F, 1.0F);
            RenderSystem.defaultBlendFunc();
        }
    }

    @SubscribeEvent
    public static void onRenderBlur(RenderLevelEvent.Post event)
    {
        if (isPlayerInvalid(Minecraft.getInstance().player)) return;
        PostProcessShaderManager shaderManager = PostProcessShaderManager.getInstance();

        if (ConfigSettings.DISTORTION_EFFECTS.get() && BLEND_TEMP >= 50 && HOT_IMMUNITY < 1)
        {
            float blur = CSMath.blend(0f, 12f, BLEND_TEMP, 50, 100);
            blur = (float) CSMath.blend(blur, 0, HOT_IMMUNITY, 0, 1);
            if (!shaderManager.hasEffect("heat_blur"))
            {   shaderManager.loadEffect("heat_blur", PostProcessShaderManager.BLOBS);
            }
            ShaderUniform blurRadius = shaderManager.getPostPasses("heat_blur").get(0).getEffect().getUniform("Radius");
            if (blurRadius != null)
            {   blurRadius.set(blur);
            }
        }
        else if (shaderManager.hasEffect("heat_blur"))
        {   shaderManager.closeEffect("heat_blur");
        }

        shaderManager.process(event.getPartialTick());
    }

    private static boolean isPlayerInvalid(PlayerEntity player)
    {   return player == null || !player.isAlive() || EntityTempManager.isPeacefulMode(player) || player.hasEffect(ModEffects.GRACE);
    }
}
