package com.momosoftworks.coldsweat.util.world;

import com.mojang.datafixers.util.Pair;
import com.momosoftworks.coldsweat.api.registry.BlockTempRegistry;
import com.momosoftworks.coldsweat.api.temperature.block_temp.BlockTemp;
import com.momosoftworks.coldsweat.api.temperature.modifier.*;
import com.momosoftworks.coldsweat.api.util.Temperature;
import com.momosoftworks.coldsweat.common.blockentity.HearthBlockEntity;
import com.momosoftworks.coldsweat.common.capability.handler.EntityTempManager;
import com.momosoftworks.coldsweat.config.ConfigSettings;
import com.momosoftworks.coldsweat.data.codec.configuration.BiomeTempData;
import com.momosoftworks.coldsweat.data.tag.ModBiomeTags;
import com.momosoftworks.coldsweat.util.entity.DummyEntity;
import com.momosoftworks.coldsweat.util.entity.DummyPlayer;
import com.momosoftworks.coldsweat.util.math.FastMap;
import com.momosoftworks.coldsweat.util.serialization.DynamicHolder;
import com.momosoftworks.coldsweat.core.network.ColdSweatPacketHandler;
import com.momosoftworks.coldsweat.core.network.message.BlockDataUpdateMessage;
import com.momosoftworks.coldsweat.core.network.message.ParticleBatchMessage;
import com.momosoftworks.coldsweat.core.network.message.PlayEntityAttachedSoundMessage;
import com.momosoftworks.coldsweat.core.network.message.SyncForgeDataMessage;
import com.momosoftworks.coldsweat.util.ClientOnlyHelper;
import com.momosoftworks.coldsweat.compat.CompatManager;
import com.momosoftworks.coldsweat.util.math.CSMath;
import com.momosoftworks.coldsweat.util.serialization.OptionalHolder;
import it.unimi.dsi.fastutil.longs.LongSet;
import net.minecraft.core.*;
import net.minecraft.core.particles.ParticleOptions;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.sounds.SoundEvent;
import net.minecraft.sounds.SoundSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.item.ItemEntity;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.*;
import net.minecraft.world.level.biome.Biome;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.CampfireBlock;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.properties.BlockStateProperties;
import net.minecraft.world.level.chunk.ChunkAccess;
import net.minecraft.world.level.chunk.ChunkStatus;
import net.minecraft.world.level.chunk.LevelChunk;
import net.minecraft.world.level.chunk.LevelChunkSection;
import net.minecraft.world.level.levelgen.Heightmap;
import net.minecraft.world.level.levelgen.structure.Structure;
import net.minecraft.world.level.levelgen.structure.StructureStart;
import net.minecraft.world.level.material.Fluid;
import net.minecraft.world.level.material.Fluids;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.Vec3;
import net.minecraft.world.phys.shapes.CollisionContext;
import net.minecraft.world.phys.shapes.Shapes;
import net.minecraft.world.phys.shapes.VoxelShape;
import net.minecraftforge.event.server.ServerStoppedEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.fml.common.Mod;
import net.minecraftforge.fml.util.ObfuscationReflectionHelper;
import net.minecraftforge.network.PacketDistributor;
import net.minecraftforge.server.ServerLifecycleHooks;

import javax.annotation.Nullable;
import java.lang.reflect.Field;
import java.util.*;
import java.util.function.*;

@Mod.EventBusSubscriber
public abstract class WorldHelper
{
    static Map<ResourceKey<Level>, DummyPlayer> DUMMY_PLAYERS = new HashMap<>();
    static Map<ResourceKey<Level>, DummyEntity> DUMMY_ENTITIES = new HashMap<>();
    static Map<ResourceKey<Level>, Map<BlockPos, TempSnapshot>> TEMPERATURE_CHECKS = new FastMap<>();

    @SubscribeEvent
    public static void clearCachesOnUnload(ServerStoppedEvent event)
    {   DUMMY_PLAYERS.clear();
        DUMMY_ENTITIES.clear();
        TEMPERATURE_CHECKS.clear();
    }

    public static int getHeight(BlockPos pos, Level level, Heightmap.Types heightmap)
    {
        int minHeight = level.getMinBuildHeight();
        int maxHeight = level.getMaxBuildHeight();
        int seaLevel = level.getSeaLevel();
        // If chunk isn't loaded, return sea level
        if (!level.isLoaded(pos)) return seaLevel;

        ChunkAccess chunk = getChunk(level, pos);
        if (chunk == null) return seaLevel;

        if (level.isClientSide())
        {
            int y = level.getMaxBuildHeight();
            BlockPos.MutableBlockPos mutable = pos.mutable();
            mutable.setY(CSMath.clamp(mutable.getY(), minHeight, maxHeight));

            BlockState state = null;

            for (; state == null || !heightmap.isOpaque().test(state); y--)
            {
                if (!CSMath.betweenInclusive(mutable.getY(), minHeight, maxHeight))
                {   return seaLevel;
                }
                mutable.setY(y);
                state = chunk.getBlockState(mutable);
            }
            return y;
        }
        else return chunk.getHeight(heightmap, pos.getX(), pos.getZ());
    }

    public static int getHeight(BlockPos pos, Level level)
    {   return getHeight(pos, level, Heightmap.Types.MOTION_BLOCKING);
    }

    public static int getAverageHeight(BlockPos pos, Level level, Heightmap.Types... heightmaps)
    {
        int totalHeight = 0;
        for (Heightmap.Types heightmap : heightmaps)
        {   totalHeight += getHeight(pos, level, heightmap);
        }
        return totalHeight / heightmaps.length;
    }

    /**
     * Returns all block positions in a grid of the specified size<br>
     * Search area scales with the number of samples
     * @param pos The center of the search area
     * @param samples The total number of checks performed.
     * @param interval How far apart each check is. Higher values = less dense and larger search area
     */
    public static List<BlockPos> getPositionGrid(BlockPos pos, int samples, int interval)
    {
        List<BlockPos> posList = new ArrayList<>();
        int sampleRoot = (int) Math.sqrt(samples);
        int radius = (sampleRoot * interval) / 2;

        for (int x = -radius; x < radius; x += interval)
        {
            for (int z = -radius; z < radius; z += interval)
            {   posList.add(pos.offset(x + interval / 2, 0, z + interval / 2));
            }
        }

        return posList;
    }

    /**
     * Returns a cube of BlockPos of the specified size and density
     * @param pos The center of the cube
     * @param size The side length of the cube, in intervals
     * @param interval The length of one interval (distance between each BlockPos)
     * @return A list of BlockPos
     */
    public static List<BlockPos> getPositionCube(BlockPos pos, int size, int interval)
    {
        List<BlockPos> posList = new ArrayList<>();
        int radius = (size * interval) / 2;
        int halfInterval = interval / 2;

        for (int x = -radius + halfInterval; x < radius + halfInterval; x += interval)
        {
            for (int y = -radius + halfInterval; y < radius + halfInterval; y += interval)
            {
                for (int z = -radius + halfInterval; z < radius + halfInterval; z += interval)
                {   posList.add(pos.offset(x, y, z));
                }
            }
        }

        return posList;
    }

    /**
     * More accurate method for detecting skylight access. Relies on block hitbox shape instead of light level.
     * @param pos The position to check
     * @param maxDistance The maximum distance to check
     * @return True if the specified position can see the sky (if no full y-axis block faces are within the detection range)
     */
    public static boolean canSeeSky(LevelAccessor level, BlockPos pos, int maxDistance)
    {
        BlockPos.MutableBlockPos pos2 = pos.mutable();
        int iterations = Math.min(maxDistance, level.getMaxBuildHeight() - pos.getY());
        ChunkAccess chunk = getChunk(level, pos);
        if (chunk == null) return true;

        for (int i = 0; i < iterations; i++)
        {
            try
            {
                BlockState state = chunk.getBlockState(pos2);
                if (ConfigSettings.THERMAL_SOURCE_SPREAD_BLACKLIST.get().contains(state.getBlock()))
                {   return false;
                }
                if (state.isAir() || state.getMaterial().isLiquid() || ConfigSettings.THERMAL_SOURCE_SPREAD_WHITELIST.get().contains(state.getBlock()))
                {   continue;
                }
                VoxelShape shape = state.getShape(level, pos, CollisionContext.empty());
                if (shape.equals(Shapes.block())) return false;

                if (isFullSide(CSMath.flattenShape(Direction.Axis.Y, shape), Direction.UP))
                {   return false;
                }
            }
            finally
            {   pos2.move(0, 1, 0);
            }
        }
        return true;
    }

    public static boolean isSpreadBlocked(LevelAccessor level, BlockState state, BlockPos pos, Direction fromDir, Direction toDir)
    {
        Block block = state.getBlock();

        if (state.isAir() || ConfigSettings.THERMAL_SOURCE_SPREAD_WHITELIST.get().contains(block))
        {   return false;
        }
        if (ConfigSettings.THERMAL_SOURCE_SPREAD_BLACKLIST.get().contains(block)) return true;

        VoxelShape shape = state.getCollisionShape(level, pos, CollisionContext.empty());
        if (shape.equals(Shapes.block())) return true;

               // Should it have spread here in the first place?
        return isFullSide(shape.getFaceShape(fromDir.getOpposite()), fromDir)
               // Can it spread out?
            || isFullSide(CSMath.flattenShape(toDir.getAxis(), shape), toDir);
    }

    public static boolean isFullSide(VoxelShape shape, Direction dir)
    {
        if (shape.isEmpty()) return false;

        // Return true if the 2D x/y area of the shape is >= 1
        double[] area = new double[1];
        switch (dir.getAxis())
        {
            case X -> shape.forAllBoxes((x1, y1, z1, x2, y2, z2) -> area[0] += Math.abs(y2 - y1) * Math.abs(z2 - z1));
            case Y -> shape.forAllBoxes((x1, y1, z1, x2, y2, z2) -> area[0] += Math.abs(x2 - x1) * Math.abs(z2 - z1));
            case Z -> shape.forAllBoxes((x1, y1, z1, x2, y2, z2) -> area[0] += Math.abs(x2 - x1) * Math.abs(y2 - y1));
        }

        return area[0] >= 1;
    }

    @Nullable
    public static ChunkAccess getChunk(LevelAccessor level, BlockPos pos)
    {   return getChunk(level, pos.getX() >> 4, pos.getZ() >> 4);
    }

    @Nullable
    public static ChunkAccess getChunk(LevelAccessor level, ChunkPos pos)
    {   return getChunk(level, pos.x, pos.z);
    }

    @Nullable
    public static ChunkAccess getChunk(LevelAccessor level, int chunkX, int chunkZ)
    {   return level.getChunkSource().getChunkNow(chunkX, chunkZ);
    }

    public static LevelChunkSection getChunkSection(ChunkAccess chunk, int y)
    {   LevelChunkSection[] sections = chunk.getSections();

        return sections[CSMath.clamp(chunk.getSectionIndex(y), 0, sections.length - 1)];
    }

    public static Optional<Holder<Structure>> getStructureAt(Level level, BlockPos pos)
    {
        if (!(level instanceof ServerLevel serverLevel) || !level.isLoaded(pos)) return Optional.empty();

        StructureManager structureManager = serverLevel.structureManager();
        Registry<Structure> structureRegistry = serverLevel.registryAccess().registryOrThrow(Registry.STRUCTURE_REGISTRY);

        // Iterate over all structures at the position (ignores Y level)
        for (Map.Entry<Structure, LongSet> entry : structureManager.getAllStructuresAt(pos).entrySet())
        {
            Structure structure = entry.getKey();
            LongSet strucCoordinates = entry.getValue();

            // Iterate over all chunk coordinates within the structures
            for (long coordinate : strucCoordinates)
            {
                SectionPos sectionpos = SectionPos.of(new ChunkPos(coordinate), level.getMinSection());
                // Get the structure start
                StructureStart structurestart = structureManager.getStartForStructure(sectionpos, structure, level.getChunk(sectionpos.x(), sectionpos.z(), ChunkStatus.STRUCTURE_STARTS));

                if (structurestart != null && structurestart.isValid())
                {
                    // If the structure has a piece at the position, get the structure's holder
                    if (structureManager.structureHasPieceAt(pos, structurestart))
                    {
                        ResourceLocation structureId = structureRegistry.getKey(structure);
                        if (structureId == null)
                        {   return Optional.empty();
                        }
                        return structureRegistry.getHolder(ResourceKey.create(Registry.STRUCTURE_REGISTRY, structureId));
                    }
                }
            }
        }
        return Optional.empty();
    }

    /**
     * Plays a sound for all tracking clients that follows the source entity around.<br>
     * Why this isn't in Vanilla Minecraft is beyond me
     * @param sound The SoundEvent to play
     * @param entity The entity to attach the sound to (all tracking entities will hear the sound)
     * @param volume The volume of the sound
     * @param pitch The pitch of the sound
     */
    public static void playEntitySound(SoundEvent sound, Entity entity, SoundSource source, float volume, float pitch)
    {
        if (!entity.isSilent())
        {
            if (entity.level.isClientSide)
            {   ClientOnlyHelper.playEntitySound(sound, source, volume, pitch, entity);
            }
            else
            {   ColdSweatPacketHandler.INSTANCE.send(PacketDistributor.TRACKING_ENTITY_AND_SELF.with(() -> entity),
                        new PlayEntityAttachedSoundMessage(sound, source, volume, pitch, entity.getId()));
            }
        }
    }

    public static boolean isInWater(Entity entity)
    {   BlockPos pos = entity.blockPosition();
        ChunkAccess chunk = WorldHelper.getChunk(entity.level, pos);
        if (chunk == null) return false;

        return entity.isInWater() || chunk.getBlockState(pos).getBlock() == Blocks.BUBBLE_COLUMN;
    }

    public static boolean isRainingAt(Level level, BlockPos pos)
    {
        return level.isRaining() && level.getBiomeManager().getBiome(pos).value().getPrecipitation() == Biome.Precipitation.RAIN
            && canSeeSky(level, pos.above(), level.getMaxBuildHeight())
            && !CompatManager.SereneSeasons.isColdEnoughToSnow(level, pos);
    }

    /**
     * Iterates through every block along the given vector
     * @param from The starting position
     * @param to The ending position
     * @param rayTracer function to run on each found block
     * @param maxHits the maximum number of blocks to act upon before the ray expires
     */
    public static void forBlocksInRay(Vec3 from, Vec3 to, Level level, ChunkAccess chunk, Map<BlockPos, BlockState> stateCache,
                                      BiConsumer<BlockState, BlockPos> rayTracer, int maxHits)
    {
        // Don't bother if the ray has no length
        if (!from.equals(to))
        {
            Vec3 ray = to.subtract(from);
            Vec3 normalRay = ray.normalize();
            BlockPos.MutableBlockPos pos = new BlockPos(from).mutable();
            ChunkAccess workingChunk = chunk;

            // Iterate over every block-long segment of the ray
            for (int i = 0; i < ray.length(); i++)
            {
                // Get the position of the current segment
                Vec3 vec = from.add(normalRay.scale(i));

                // Skip if the position is the same as the last one
                if (new BlockPos(vec).equals(pos)) continue;
                pos.set(vec.x, vec.y, vec.z);

                // Get the blockstate at the current position
                BlockState state = stateCache.get(pos);

                if (state == null)
                {   // Set new workingChunk if the ray travels outside the current one
                    if (workingChunk == null || !workingChunk.getPos().equals(new ChunkPos(pos)))
                    {   workingChunk = getChunk(level, pos);
                    }
                    if (workingChunk == null) continue;

                    state = workingChunk.getBlockState(pos);
                    stateCache.put(pos.immutable(), state);
                }


                // If the block isn't air, then we hit something
                if (!state.isAir() && --maxHits <= 0)
                {   break;
                }

                rayTracer.accept(state, pos);
            }
        }
    }

    /**
     * Overload for {@link #forBlocksInRay(Vec3, Vec3, Level, ChunkAccess, Map, BiConsumer, int)} with less bloated params
     */
    public static void forBlocksInRay(Vec3 from, Vec3 to, Level level, BiConsumer<BlockState, BlockPos> rayTracer, int maxHits)
    {   forBlocksInRay(from, to, level, getChunk(level, new BlockPos(from)), new HashMap<>(), rayTracer, maxHits);
    }

    public static Entity raycastEntity(Vec3 from, Vec3 to, Level level, Predicate<Entity> filter)
    {
        // Don't bother if the ray has no length
        if (!from.equals(to))
        {
            Vec3 ray = to.subtract(from);
            Vec3 normalRay = ray.normalize();
            BlockPos.MutableBlockPos pos = new BlockPos.MutableBlockPos();

            // Iterate over every block-long segment of the ray
            for (int i = 0; i < ray.length(); i++)
            {
                // Get the position of the current segment
                Vec3 vec = from.add(normalRay.scale(i));

                // Skip if the position is the same as the last one
                if (new BlockPos(vec).equals(pos)) continue;
                pos.set(vec.x, vec.y, vec.z);

                // Return the first entity in the current block, or continue if there is none
                List<Entity> entities = level.getEntitiesOfClass(Entity.class, new AABB(pos), filter);
                if (!entities.isEmpty()) return entities.get(0);
            }
        }
        return null;
    }

    public static void spawnParticle(Level level, ParticleOptions particle, double x, double y, double z,
                                     double xSpeed, double ySpeed, double zSpeed)
    {
        if (!level.isClientSide)
        {
            ParticleBatchMessage particles = new ParticleBatchMessage();
            particles.addParticle(particle, new ParticleBatchMessage.ParticlePlacement(x, y, z, xSpeed, ySpeed, zSpeed));
            ColdSweatPacketHandler.INSTANCE.send(PacketDistributor.TRACKING_CHUNK.with(() -> (LevelChunk) getChunk(level, (int) x >> 4, (int) z >> 4)), particles);
        }
        else
        {   level.addParticle(particle, x, y, z, xSpeed, ySpeed, zSpeed);
        }
    }

    public static void spawnParticleBatch(Level level, ParticleOptions particle, double x, double y, double z,
                                          double xSpread, double ySpread, double zSpread, double count, double speed)
    {
        Random rand = new Random();

        if (!level.isClientSide)
        {
            ParticleBatchMessage particles = new ParticleBatchMessage();
            for (int i = 0; i < count; i++)
            {
                Vec3 vec = new Vec3(Math.random() - 0.5, Math.random() - 0.5, Math.random() - 0.5).normalize().scale(speed);
                particles.addParticle(particle, new ParticleBatchMessage.ParticlePlacement(
                        x + xSpread - rand.nextDouble() * (xSpread * 2),
                        y + ySpread - rand.nextDouble() * (ySpread * 2),
                        z + zSpread - rand.nextDouble() * (zSpread * 2), vec.x, vec.y, vec.z));
            }
            ColdSweatPacketHandler.INSTANCE.send(PacketDistributor.DIMENSION.with(level::dimension), particles);
        }
        else
        {
            for (int i = 0; i < count; i++)
            {
                Vec3 vec = new Vec3(Math.random() - 0.5, Math.random() - 0.5, Math.random() - 0.5).normalize().scale(speed);
                level.addParticle(particle,
                        x + xSpread - rand.nextDouble() * (xSpread * 2),
                        y + ySpread - rand.nextDouble() * (ySpread * 2),
                        z + zSpread - rand.nextDouble() * (zSpread * 2), vec.x, vec.y, vec.z);
            }
        }
    }

    public static ItemEntity dropItem(Level level, BlockPos pos, ItemStack stack)
    {   return dropItem(level, pos, stack, 6000);
    }

    public static ItemEntity dropItem(Level level, BlockPos pos, ItemStack stack, int lifeTime)
    {
        Random rand = new Random();
        ItemEntity item = new ItemEntity(level, pos.getX(), pos.getY(), pos.getZ(), stack);

        item.setDeltaMovement(item.getDeltaMovement().add(((rand.nextFloat() - rand.nextFloat()) * 0.1F), (rand.nextFloat() * 0.05F), ((rand.nextFloat() - rand.nextFloat()) * 0.1F)));
        Field age = ObfuscationReflectionHelper.findField(ItemEntity.class, "f_31985_");
        age.setAccessible(true);
        try
        {   age.set(item, 6000 - lifeTime);
        }
        catch (Exception e)
        {   e.printStackTrace();
        }
        return item;
    }

    /**
     * Drops an item with random velocity from the entity's position
     * @return The dropped item entity
     */
    public static ItemEntity entityDropItem(Entity entity, ItemStack stack)
    {   return entityDropItem(entity, stack, 6000);
    }

    /**
     * Drops an item with random velocity from the entity's position
     * @param lifeTime The despawn time of the item, in ticks
     * @return The dropped item entity
     */
    public static ItemEntity entityDropItem(Entity entity, ItemStack stack, int lifeTime)
    {
        Random rand = new Random();
        ItemEntity item = entity.spawnAtLocation(stack, entity.getBbHeight());
        if (item != null)
        {   item.setDeltaMovement(item.getDeltaMovement().add(((rand.nextFloat() - rand.nextFloat()) * 0.1F), (rand.nextFloat() * 0.05F), ((rand.nextFloat() - rand.nextFloat()) * 0.1F)));
            item.lifespan = lifeTime;
        }
        return item;
    }

    /**
     * @return The closest Vec3, contained in the entity's bounding box, to the given pos.
     */
    public static Vec3 getClosestPointOnEntity(LivingEntity entity, Vec3 pos)
    {
        double playerRadius = entity.getBbWidth() / 2;
        return new Vec3(CSMath.clamp(pos.x, entity.getX() - playerRadius, entity.getX() + playerRadius),
                        CSMath.clamp(pos.y, entity.getY(), entity.getY() + entity.getBbHeight()),
                        CSMath.clamp(pos.z, entity.getZ() - playerRadius, entity.getZ() + playerRadius));
    }

    /**
     * Merges the entity's server-side persistent data into the client-side persistent data
     * @param destination The player to send the data to. If null, sends to all tracking entities
     */
    public static void syncEntityForgeData(Entity entity, ServerPlayer destination)
    {
        ColdSweatPacketHandler.INSTANCE.send(destination != null ? PacketDistributor.PLAYER.with(() -> destination)
                                                                 : PacketDistributor.TRACKING_ENTITY.with(() -> entity),
                                             new SyncForgeDataMessage(entity));
    }

    /**
     * Manually sends a block entity's update tag to tracking clients
     */
    public static void syncBlockEntityData(BlockEntity be)
    {
        if (be.getLevel() == null || be.getLevel().isClientSide) return;

        ChunkAccess ichunk = getChunk(be.getLevel(), be.getBlockPos());
        if (ichunk instanceof LevelChunk chunk)
        {   ColdSweatPacketHandler.INSTANCE.send(PacketDistributor.TRACKING_CHUNK.with(() -> chunk), new BlockDataUpdateMessage(be));
        }
    }

    /**
     * Allows the server world to be accessed if you only have a generic "Level" object<br>
     * ONLY USE ON THE SERVER THREAD
     * @return The server world
     */
    public static ServerLevel getServerLevel(Level level)
    {   return ServerLifecycleHooks.getCurrentServer().getLevel(level.dimension());
    }

    /**
     * Gets the server instance<br>
     * ONLY USE ON THE SERVER THREAD
     * @return The server instance
     */
    public static MinecraftServer getServer()
    {   return ServerLifecycleHooks.getCurrentServer();
    }

    /**
     * Gets the min and max temperature of the biome.
     * @return A pair of the min and max temperature of the biome
     */
    public static Pair<Double, Double> getBiomeTemperatureRange(LevelAccessor level, Holder<Biome> biome)
    {   return getBiomeTemperatureRange(level.registryAccess(), biome);
    }

    /**
     * Gets the min and max temperature of the biome.
     * @return A pair of the min and max temperature of the biome
     */
    public static Pair<Double, Double> getBiomeTemperatureRange(RegistryAccess registryAccess, Holder<Biome> biome)
    {
        double variance = 1 / Math.max(1, 2 + biome.value().getModifiedClimateSettings().downfall() * 2);
        double baseTemp = biome.value().getBaseTemperature();

        Pair<Double, Double> biomeTemp = Optional.ofNullable(ConfigSettings.BIOME_TEMPS.get(registryAccess).get(biome))
                                                 .filter(data -> !data.isDisabled())
                                                 .map(data -> Pair.of(data.minTemp(), data.maxTemp()))
                                                 .orElse(Pair.of(baseTemp - variance, baseTemp + variance));
        Pair<Double, Double> configOffset = Optional.ofNullable(ConfigSettings.BIOME_OFFSETS.get(registryAccess).get(biome))
                                                    .filter(data -> !data.isDisabled())
                                                    .map(data -> Pair.of(data.minTemp(), data.maxTemp()))
                                                    .orElse(Pair.of(0d, 0d));
        return CSMath.addPairs(Pair.of(biomeTemp.getFirst(), biomeTemp.getSecond()),
                               Pair.of(configOffset.getFirst(), configOffset.getSecond()));
    }

    /**
     * Gets the temperature of the biome at the specified position, including time of day.
     * @return The temperature of the biome at the specified position
     */
    public static double getBiomeTemperature(LevelAccessor level, Holder<Biome> biome)
    {
        Pair<Double, Double> temps = getBiomeTemperatureRange(level, biome);
        return CSMath.blend(temps.getFirst(), temps.getSecond(), getTimeMultiplier(level), -1, 1);
    }

    public static double getTimeMultiplier(LevelAccessor level)
    {   return Math.sin(level.dayTime() / (12000 / Math.PI));
    }

    public static double getWaterTemperatureAt(Level level, BlockPos pos)
    {
        double defaultWaterTemp = ConfigSettings.DEFAULT_WATER_TEMPERATURE.get();
        BiomeTempData biomeTemp = ConfigSettings.BIOME_TEMPS.get(level.registryAccess()).get(level.getBiome(pos));
        if (biomeTemp == null)
        {   return defaultWaterTemp;
        }
        return biomeTemp.isOffset() ? defaultWaterTemp + biomeTemp.waterTemp()
                                    : biomeTemp.waterTemp();
    }

    /**
     * Returns a cached temperature value<br>
     * <br>
     * <b>Flags:</b><br>
     * 1 = sensitive<br>
     * 2 = force update<br>
     */
    public static double getRoughTemperatureAt(Level level, BlockPos pos, int flags)
    {
        boolean sensitive = (flags & 1) != 0;
        boolean forceUpdate = (flags & 2) != 0;

        Map<BlockPos, TempSnapshot> snapshots = TEMPERATURE_CHECKS.computeIfAbsent(level.dimension(), dim -> new HashMap<>());
        int tickSpeedMultiplier = 1 + level.getGameRules().getInt(GameRules.RULE_RANDOMTICKING) / 20;

        BlockPos segment = new BlockPos(pos.getX() >> 3, pos.getY() >> 3, pos.getZ() >> 3);

        // Use cached value if not forced update
        if (!forceUpdate)
        {
            int interval = sensitive ? 200 : 1000;
            TempSnapshot snapshot = snapshots.get(segment);
            if (snapshot != null)
            {
                long gameTime = level.getGameTime();
                if (gameTime - snapshot.timestamp < interval / tickSpeedMultiplier)
                {   return snapshot.temperature();
                }
            }
        }

        /* Get temperature based on dummy entity */
        // Init dummy at this location
        DummyEntity dummy = getDummyEntity(level);
        dummy.setPos(CSMath.getCenterPos(pos));
        // Get modifiers
        List<TempModifier> modifiers = new ArrayList<>(Temperature.getModifiers(dummy, Temperature.Trait.WORLD));

        // Get insulation from hearths
        Pair<Integer, Integer> maxCoolingHeating = getInsulationAt(level, pos, 2);
        if (maxCoolingHeating.getFirst() > 0)
        {   modifiers.add(new FrigidnessTempModifier(maxCoolingHeating.getFirst()));
        }
        if (maxCoolingHeating.getSecond() > 0)
        {   modifiers.add(new WarmthTempModifier(maxCoolingHeating.getSecond()));
        }
        // Get & store temperature
        double tempAt = Temperature.apply(0, dummy, Temperature.Trait.WORLD, modifiers, true);
        snapshots.put(segment, new TempSnapshot(level.getGameTime(), tempAt));

        return tempAt;
    }

    public static double getRoughTemperatureAt(Level level, BlockPos pos)
    {   return getRoughTemperatureAt(level, pos, 0);
    }

    /**
     * Gets the "raw" temperature for a block, ignoring distance and occlusion
     */
    public static double getBlockTemperature(Level level, BlockState block)
    {
        Collection<BlockTemp> blockTemps = BlockTempRegistry.getBlockTempsFor(block);
        for (BlockTemp blockTemp : blockTemps)
        {
            if (blockTemp.isValid(level, BlockPos.ZERO, block))
            {   return blockTemp.getTemperature(level, null, block, BlockPos.ZERO, 0);
            }
        }
        return 0;
    }

    public static double getTemperatureAt(Level level, BlockPos pos)
    {
        DummyPlayer dummy = getDummyPlayer(level);
        // Move the dummy to the position being tested
        dummy.setPos(CSMath.getCenterPos(pos));
        List<TempModifier> modifiers = new ArrayList<>(Temperature.getModifiers(dummy, Temperature.Trait.WORLD));

        // Get insulation from hearths
        Pair<Integer, Integer> maxCoolingHeating = getInsulationAt(level, pos, 2);
        if (maxCoolingHeating.getFirst() > 0)
        {   modifiers.add(new FrigidnessTempModifier(maxCoolingHeating.getFirst()));
        }
        if (maxCoolingHeating.getSecond() > 0)
        {   modifiers.add(new WarmthTempModifier(maxCoolingHeating.getSecond()));
        }
        return Temperature.apply(0, dummy, Temperature.Trait.WORLD, modifiers, true);
    }

    public static DummyPlayer getDummyPlayer(Level level)
    {
        ResourceKey<Level> dimension = level.dimension();
        // There is one "dummy" entity per world, which TempModifiers are applied to
        DummyPlayer dummy = DUMMY_PLAYERS.get(dimension);
        // If the dummy for this dimension is invalid, make a new one
        if (dummy == null || dummy.level != level)
        {
            WorldHelper.DUMMY_PLAYERS.put(dimension, dummy = new DummyPlayer(level));
            // Use default player modifiers to determine the temperature
            Map<Temperature.Trait, List<TempModifier>> defaultModifiers = EntityTempManager.gatherTempModifiers(dummy);
            defaultModifiers.get(Temperature.Trait.WORLD).forEach(mod -> mod.tickRate(1));
            Temperature.getModifiers(dummy).putAll(defaultModifiers);
        }
        return dummy;
    }

    public static DummyEntity getDummyEntity(Level level)
    {
        ResourceKey<Level> dimension = level.dimension();
        // There is one "dummy" entity per world, which TempModifiers are applied to
        DummyEntity dummy = DUMMY_ENTITIES.get(dimension);
        // If the dummy for this dimension is invalid, make a new one
        if (dummy == null || dummy.level != level)
        {
            WorldHelper.DUMMY_ENTITIES.put(dimension, dummy = new DummyEntity(level));
            // Use default player modifiers to determine the temperature
            Map<Temperature.Trait, List<TempModifier>> defaultModifiers = EntityTempManager.gatherTempModifiers(dummy);
            defaultModifiers.get(Temperature.Trait.WORLD).forEach(mod -> mod.tickRate(1));
            Temperature.getModifiers(dummy).putAll(defaultModifiers);
        }
        return dummy;
    }

    public Map<ResourceKey<Level>, DummyPlayer> getDummyPlayers()
    {   return DUMMY_PLAYERS;
    }

    public Map<ResourceKey<Level>, DummyEntity> getDummyEntities()
    {   return DUMMY_ENTITIES;
    }

    public Map<ResourceKey<Level>, Map<BlockPos, TempSnapshot>> getWorldTempCache()
    {   return TEMPERATURE_CHECKS;
    }

    public static boolean allAdjacentBlocksMatch(BlockPos pos, Predicate<BlockPos> predicate)
    {
        BlockPos.MutableBlockPos pos2 = pos.mutable();
        for (int i = 0; i < Direction.values().length; i++)
        {
            BlockPos offset = pos2.setWithOffset(pos, Direction.values()[i]);
            if (!predicate.test(offset)) return false;
        }
        return true;
    }

    public static BlockState waterlog(BlockState state, Level level, BlockPos pos)
    {
        boolean waterAt = level.getFluidState(pos).getType() == Fluids.WATER;
        return state.setValue(BlockStateProperties.WATERLOGGED, waterAt);
    }

    public static boolean shouldFreeze(LevelAccessor levelReader, BlockPos pos, boolean mustBeAtEdge)
    {
        if (pos.getY() >= levelReader.getMinBuildHeight() && pos.getY() < levelReader.getMaxBuildHeight()
        && levelReader instanceof ServerLevel serverLevel)
        {
            Holder<Biome> biome = levelReader.getBiome(pos);
            if (biome.is(ModBiomeTags.HAS_HOT_WATER)) return false;

            if (surroundedByBlock(levelReader, pos, Blocks.ICE))
            {   return true;
            }
            DynamicHolder<Boolean> freezingTemp = DynamicHolder.create(null, () -> getRoughTemperatureAt(serverLevel, pos) < 0f);

            if (!mustBeAtEdge)
            {   return freezingTemp.get();
            }
            return !surroundedByFluid(levelReader, pos, Fluids.WATER) && freezingTemp.get();
        }
        return false;
    }

    public static boolean shouldMelt(LevelAccessor levelReader, BlockPos pos, boolean mustBeAtEdge)
    {
        if (pos.getY() >= levelReader.getMinBuildHeight() && pos.getY() < levelReader.getMaxBuildHeight()
        && levelReader instanceof ServerLevel serverLevel)
        {
            Holder<Biome> biome = levelReader.getBiome(pos);
            if (biome.is(ModBiomeTags.HAS_HOT_WATER)) return true;

            if (mustBeAtEdge && surroundedByBlock(levelReader, pos, Blocks.ICE))
            {   return false;
            }
            return getRoughTemperatureAt(serverLevel, pos) >= 0f;
        }
        return false;
    }

    public static boolean surroundedByBlock(LevelAccessor level, BlockPos pos, Block block)
    {
        return level.getBlockState(pos.north()).is(block)
            && level.getBlockState(pos.south()).is(block)
            && level.getBlockState(pos.east()).is(block)
            && level.getBlockState(pos.west()).is(block);
    }

    public static boolean surroundedByFluid(LevelAccessor level, BlockPos pos, Fluid fluid)
    {
        return level.getBlockState(pos.north()).getFluidState().is(fluid)
            && level.getBlockState(pos.south()).getFluidState().is(fluid)
            && level.getBlockState(pos.east()).getFluidState().is(fluid)
            && level.getBlockState(pos.west()).getFluidState().is(fluid);
    }

    public static boolean nextToSoulFire(LevelAccessor level, BlockPos pos)
    {
        BlockPos.MutableBlockPos pos2 = pos.mutable();
        for (int x = -1; x <= 1; x++)
        for (int y =  0; y <= 1; y++)
        for (int z = -1; z <= 1; z++)
        {
            BlockState state = level.getBlockState(pos2.setWithOffset(pos, x, y, z));
            if (state.is(Blocks.SOUL_FIRE) || state.is(Blocks.SOUL_CAMPFIRE) && state.getValue(CampfireBlock.LIT))
            {   return true;
            }
        }
        return false;
    }

    public static Pair<Integer, Integer> getInsulationAt(Level level, BlockPos pos, int chunkRadius)
    {
        int maxCoolingLevel = 0;
        int maxHeatingLevel = 0;
        ChunkPos chunkPos = new ChunkPos(pos);
        for (int x = -chunkRadius; x <= chunkRadius; x++)
        for (int z = -chunkRadius; z <= chunkRadius; z++)
        {
            ChunkAccess chunk = getChunk(level, chunkPos.x + x, chunkPos.z + z);
            if (chunk == null) continue;

            for (BlockPos bePos : chunk.getBlockEntitiesPos())
            {
                BlockEntity be = chunk.getBlockEntity(bePos);
                if (be instanceof HearthBlockEntity hearth && hearth.getPathLookup().contains(pos))
                {
                    maxCoolingLevel = Math.max(maxCoolingLevel, hearth.getCoolingLevel());
                    maxHeatingLevel = Math.max(maxHeatingLevel, hearth.getHeatingLevel());
                }
            }
        }
        return Pair.of(maxCoolingLevel, maxHeatingLevel);
    }

    public static List<BlockPos> getOccupiedPositions(AABB bb)
    {
        List<BlockPos> positions = new ArrayList<>();
        int minX = (int) Math.floor(bb.minX);
        int minY = (int) Math.floor(bb.minY);
        int minZ = (int) Math.floor(bb.minZ);
        int maxX = (int) Math.ceil(bb.maxX);
        int maxY = (int) Math.ceil(bb.maxY);
        int maxZ = (int) Math.ceil(bb.maxZ);

        for (int x = minX; x < maxX; x++)
        for (int y = minY; y < maxY; y++)
        for (int z = minZ; z < maxZ; z++)
        {   positions.add(new BlockPos(x, y, z));
        }
        return positions;
    }

    public record TempSnapshot(long timestamp, double temperature) {}
}
