package thelm.packagedauto.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.function.BooleanSupplier;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;

import it.unimi.dsi.fastutil.Hash;
import it.unimi.dsi.fastutil.objects.Object2IntLinkedOpenCustomHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.ObjectRBTreeSet;
import net.minecraft.client.Minecraft;
import net.minecraft.core.Direction;
import net.minecraft.core.NonNullList;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.Container;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.crafting.RecipeManager;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.fml.DistExecutor;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.ItemHandlerHelper;
import thelm.packagedauto.api.IMiscHelper;
import thelm.packagedauto.api.IPackagePattern;
import thelm.packagedauto.api.IPackageRecipeInfo;
import thelm.packagedauto.api.IPackageRecipeType;
import thelm.packagedauto.api.IVolumePackageItem;
import thelm.packagedauto.api.IVolumeStackWrapper;
import thelm.packagedauto.api.IVolumeType;
import thelm.packagedauto.api.PackagedAutoApi;
import thelm.packagedauto.item.VolumePackageItem;

public class MiscHelper implements IMiscHelper {

	public static final MiscHelper INSTANCE = new MiscHelper();

	private static final Cache<CompoundTag, IPackageRecipeInfo> RECIPE_CACHE = CacheBuilder.newBuilder().maximumSize(500).build();
	private static final Logger LOGGER = LogManager.getLogger();

	private static MinecraftServer server;

	private MiscHelper() {}

	@Override
	public List<ItemStack> condenseStacks(Container container) {
		List<ItemStack> stacks = new ArrayList<>(container.m_6643_());
		for(int i = 0; i < container.m_6643_(); ++i) {
			stacks.add(container.m_8020_(i));
		}
		return condenseStacks(stacks);
	}

	@Override
	public List<ItemStack> condenseStacks(IItemHandler itemHandler) {
		List<ItemStack> stacks = new ArrayList<>(itemHandler.getSlots());
		for(int i = 0; i < itemHandler.getSlots(); ++i) {
			stacks.add(itemHandler.getStackInSlot(i));
		}
		return condenseStacks(stacks);
	}

	@Override
	public List<ItemStack> condenseStacks(ItemStack... stacks) {
		return condenseStacks(List.of(stacks));
	}

	@Override
	public List<ItemStack> condenseStacks(Stream<ItemStack> stacks) {
		return condenseStacks(stacks.toList());
	}

	@Override
	public List<ItemStack> condenseStacks(Iterable<ItemStack> stacks) {
		return condenseStacks(stacks instanceof List<?> ? (List<ItemStack>)stacks : Lists.newArrayList(stacks));
	}

	@Override
	public List<ItemStack> condenseStacks(List<ItemStack> stacks) {
		return condenseStacks(stacks, false);
	}

	@Override
	public List<ItemStack> condenseStacks(List<ItemStack> stacks, boolean ignoreStackSize) {
		Object2IntLinkedOpenCustomHashMap<Pair<Item, CompoundTag>> map = new Object2IntLinkedOpenCustomHashMap<>(new Hash.Strategy<>() {
			@Override
			public int hashCode(Pair<Item, CompoundTag> o) {
				return Objects.hash(Item.m_41393_(o.getLeft()), o.getRight());
			}
			@Override
			public boolean equals(Pair<Item, CompoundTag> a, Pair<Item, CompoundTag> b) {
				return a.equals(b);
			}
		});
		for(ItemStack stack : stacks) {
			if(stack.m_41619_()) {
				continue;
			}
			Pair<Item, CompoundTag> pair = Pair.of(stack.m_41720_(), stack.m_41783_());
			if(!map.containsKey(pair)) {
				map.put(pair, 0);
			}
			map.addTo(pair, stack.m_41613_());
		}
		List<ItemStack> list = new ArrayList<>();
		for(Object2IntMap.Entry<Pair<Item, CompoundTag>> entry : map.object2IntEntrySet()) {
			Pair<Item, CompoundTag> pair = entry.getKey();
			int count = entry.getIntValue();
			Item item = pair.getLeft();
			CompoundTag nbt = pair.getRight();
			if(ignoreStackSize) {
				ItemStack toAdd = new ItemStack(item, count);
				toAdd.m_41751_(nbt);
				list.add(toAdd);
			}
			else {
				while(count > 0) {
					ItemStack toAdd = new ItemStack(item, 1);
					toAdd.m_41751_(nbt);
					int limit = item.getItemStackLimit(toAdd);
					toAdd.m_41764_(Math.min(count, limit));
					list.add(toAdd);
					count -= limit;
				}
			}
		}
		map.clear();
		return list;
	}

	@Override
	public ListTag saveAllItems(ListTag tagList, List<ItemStack> list) {
		return saveAllItems(tagList, list, "Index");
	}

	@Override
	public ListTag saveAllItems(ListTag tagList, List<ItemStack> list, String indexKey) {
		for(int i = 0; i < list.size(); ++i) {
			ItemStack stack = list.get(i);
			boolean empty = stack.m_41619_();
			if(!empty || i == list.size()-1) {
				if(empty) {
					// Ensure that the end-of-list stack if empty is always the default empty stack
					stack = new ItemStack((Item)null);
				}
				CompoundTag nbt = new CompoundTag();
				nbt.m_128344_(indexKey, (byte)i);
				saveItemWithLargeCount(nbt, stack);
				tagList.add(nbt);
			}
		}
		return tagList;
	}

	@Override
	public void loadAllItems(ListTag tagList, List<ItemStack> list) {
		loadAllItems(tagList, list, "Index");
	}

	@Override
	public void loadAllItems(ListTag tagList, List<ItemStack> list, String indexKey) {
		list.clear();
		try {
			for(int i = 0; i < tagList.size(); ++i) {
				CompoundTag nbt = tagList.m_128728_(i);
				int j = nbt.m_128445_(indexKey) & 255;
				while(j >= list.size()) {
					list.add(ItemStack.f_41583_);
				}
				if(j >= 0)  {
					ItemStack stack = loadItemWithLargeCount(nbt);
					list.set(j, stack.m_41619_() ? ItemStack.f_41583_ : stack);
				}
			}
		}
		catch(UnsupportedOperationException | IndexOutOfBoundsException e) {}
	}

	@Override
	public CompoundTag saveItemWithLargeCount(CompoundTag nbt, ItemStack stack) {
		stack.m_41739_(nbt);
		int count = stack.m_41613_();
		if((byte)count == count) {
			nbt.m_128344_("Count", (byte)count);
		}
		else if((short)count == count) {
			nbt.m_128376_("Count", (short)count);
		}
		else {
			nbt.m_128405_("Count", (short)count);
		}
		return nbt;
	}

	@Override
	public ItemStack loadItemWithLargeCount(CompoundTag nbt) {
		ItemStack stack = ItemStack.m_41712_(nbt);
		stack.m_41764_(nbt.m_128451_("Count"));
		return stack;
	}

	@Override
	public void writeItemWithLargeCount(FriendlyByteBuf buf, ItemStack stack) {
		if(stack.m_41619_()) {
			buf.writeBoolean(false);
			return;
		}
		buf.writeBoolean(true);
		buf.m_130130_(Item.m_41393_(stack.m_41720_()));
		buf.m_130130_(stack.m_41613_());
		CompoundTag nbt = null;
		if(stack.m_41720_().isDamageable(stack) || stack.m_41720_().m_41468_()) {
			nbt = stack.getShareTag();
		}
		buf.m_130079_(nbt);
	}

	@Override
	public ItemStack readItemWithLargeCount(FriendlyByteBuf buf) {
		if(!buf.readBoolean()) {
			return ItemStack.f_41583_;
		}
		int id = buf.m_130242_();
		int count = buf.m_130242_();
		ItemStack stack = new ItemStack(Item.m_41445_(id), count);
		stack.m_41720_().readShareTag(stack, buf.m_130260_());
		return stack;
	}

	@Override
	public IPackagePattern getPattern(IPackageRecipeInfo recipeInfo, int index) {
		return new PackagePattern(recipeInfo, index);
	}

	@Override
	public List<ItemStack> getRemainingItems(Container container) {
		return getRemainingItems(IntStream.range(0, container.m_6643_()).mapToObj(container::m_8020_).toList());
	}

	@Override
	public List<ItemStack> getRemainingItems(Container container, int minInclusive, int maxExclusive) {
		return getRemainingItems(IntStream.range(minInclusive, maxExclusive).mapToObj(container::m_8020_).toList());
	}

	@Override
	public List<ItemStack> getRemainingItems(ItemStack... stacks) {
		return getRemainingItems(List.of(stacks));
	}

	@Override
	public List<ItemStack> getRemainingItems(List<ItemStack> stacks) {
		NonNullList<ItemStack> ret = NonNullList.m_122780_(stacks.size(), ItemStack.f_41583_);
		for(int i = 0; i < ret.size(); i++) {
			ret.set(i, getContainerItem(stacks.get(i)));
		}
		return ret;
	}

	@Override
	public ItemStack getContainerItem(ItemStack stack) {
		if(stack.m_41619_()) {
			return ItemStack.f_41583_;
		}
		if(stack.m_41720_().hasContainerItem(stack)) {
			stack = stack.m_41720_().getContainerItem(stack);
			if(!stack.m_41619_() && stack.m_41763_() && stack.m_41773_() > stack.m_41776_()) {
				return ItemStack.f_41583_;
			}
			return stack;
		}
		else {
			if(stack.m_41613_() > 1) {
				stack = stack.m_41777_();
				stack.m_41764_(stack.m_41613_() - 1);
				return stack;
			}
			return ItemStack.f_41583_;
		}
	}

	@Override
	public ItemStack cloneStack(ItemStack stack, int stackSize) {
		if(stack.m_41619_()) {
			return ItemStack.f_41583_;
		}
		ItemStack retStack = stack.m_41777_();
		retStack.m_41764_(stackSize);
		return retStack;
	}

	@Override
	public boolean isEmpty(IItemHandler itemHandler) {
		if(itemHandler == null || itemHandler.getSlots() == 0) {
			return false;
		}
		for(int i = 0; i < itemHandler.getSlots(); ++i) {
			if(!itemHandler.getStackInSlot(i).m_41619_()) {
				return false;
			}
		}
		return true;
	}

	@Override
	public ItemStack makeVolumePackage(IVolumeStackWrapper volumeStack) {
		return VolumePackageItem.makeVolumePackage(volumeStack);
	}

	@Override
	public ItemStack tryMakeVolumePackage(Object volumeStack) {
		return VolumePackageItem.tryMakeVolumePackage(volumeStack);
	}

	@Override
	public CompoundTag saveRecipe(CompoundTag nbt, IPackageRecipeInfo recipe) {
		nbt.m_128359_("RecipeType", recipe.getRecipeType().getName().toString());
		recipe.save(nbt);
		return nbt;
	}

	@Override
	public IPackageRecipeInfo loadRecipe(CompoundTag nbt) {
		IPackageRecipeType recipeType = PackagedAutoApi.instance().getRecipeType(new ResourceLocation(nbt.m_128461_("RecipeType")));
		if(recipeType != null) {
			IPackageRecipeInfo recipe = RECIPE_CACHE.getIfPresent(nbt);
			if(recipe != null) {
				return recipe;
			}
			recipe = recipeType.getNewRecipeInfo();
			recipe.load(nbt);
			RECIPE_CACHE.put(nbt, recipe);
			return recipe;
		}
		return null;
	}

	@Override
	public ListTag saveRecipeList(ListTag tagList, List<IPackageRecipeInfo> recipes) {
		for(IPackageRecipeInfo recipe : recipes) {
			tagList.add(saveRecipe(new CompoundTag(), recipe));
		}
		return tagList;
	}

	@Override
	public List<IPackageRecipeInfo> loadRecipeList(ListTag tagList) {
		List<IPackageRecipeInfo> recipes = new ArrayList<>(tagList.size());
		for(int i = 0; i < tagList.size(); ++i) {
			IPackageRecipeInfo recipe = loadRecipe(tagList.m_128728_(i));
			if(recipe != null) {
				recipes.add(recipe);
			}
		}
		return recipes;
	}

	@Override
	public boolean recipeEquals(IPackageRecipeInfo recipeA, Object recipeInternalA, IPackageRecipeInfo recipeB, Object recipeInternalB) {
		if(!Objects.equals(recipeInternalA, recipeInternalB)) {
			return false;
		}
		List<ItemStack> inputsA = recipeA.getInputs();
		List<ItemStack> inputsB = recipeB.getInputs();
		if(inputsA.size() != inputsB.size()) {
			return false;
		}
		List<ItemStack> outputsA = recipeA.getOutputs();
		List<ItemStack> outputsB = recipeB.getOutputs();
		if(outputsA.size() != outputsB.size()) {
			return false;
		}
		for(int i = 0; i < inputsA.size(); ++i) {
			if(!ItemStack.m_41728_(inputsA.get(i), inputsB.get(i))) {
				return false;
			}
		}
		for(int i = 0; i < outputsA.size(); ++i) {
			if(!ItemStack.m_41728_(outputsA.get(i), outputsB.get(i))) {
				return false;
			}
		}
		return true;
	}

	@Override
	public int recipeHashCode(IPackageRecipeInfo recipe, Object recipeInternal) {
		List<ItemStack> inputs = recipe.getInputs();
		List<ItemStack> outputs = recipe.getOutputs();
		Function<ItemStack, Object[]> decompose = stack->new Object[] {
				stack.m_41720_(), stack.m_41613_(), stack.m_41783_(),
		};
		Object[] toHash = {
				recipeInternal, inputs.stream().map(decompose).toArray(), outputs.stream().map(decompose).toArray(),
		};
		return Arrays.deepHashCode(toHash);
	}

	//Modified from Forestry
	@Override
	public boolean removeExactSet(List<ItemStack> offered, List<ItemStack> required, boolean simulate) {
		List<ItemStack> condensedRequired = condenseStacks(required, true);
		List<ItemStack> condensedOffered = condenseStacks(offered, true);
		f:for(ItemStack req : condensedRequired) {
			for(ItemStack offer : condensedOffered) {
				if(req.m_41613_() <= offer.m_41613_() && req.m_41656_(offer) &&
						(!req.m_41782_() || ItemStack.m_41658_(req, offer))) {
					continue f;
				}
			}
			return false;
		}
		if(simulate) {
			return true;
		}
		for(ItemStack req : condensedRequired) {
			int count = req.m_41613_();
			for(ItemStack offer : offered) {
				if(!offer.m_41619_()) {
					if(req.m_41656_(offer) && (!req.m_41782_() || ItemStack.m_41658_(req, offer))) {
						int toRemove = Math.min(count, offer.m_41613_());
						offer.m_41774_(toRemove);
						count -= toRemove;
						if(count == 0) {
							continue;
						}
					}
				}
			}
		}
		return true;
	}

	@Override
	public boolean arePatternsDisjoint(List<IPackagePattern> patternList) {
		ObjectRBTreeSet<Pair<Item, CompoundTag>> set = new ObjectRBTreeSet<>(
				Comparator.comparing(pair->Pair.of(pair.getLeft().getRegistryName(), ""+pair.getRight())));
		for(IPackagePattern pattern : patternList) {
			List<ItemStack> condensedInputs = condenseStacks(pattern.getInputs(), true);
			for(ItemStack stack : condensedInputs) {
				Pair<Item, CompoundTag> toAdd = Pair.of(stack.m_41720_(), stack.m_41783_());
				if(set.contains(toAdd)) {
					return false;
				}
				set.add(toAdd);
			}
		}
		set.clear();
		return true;
	}

	@Override
	public ItemStack insertItem(IItemHandler itemHandler, ItemStack stack, boolean requireEmptySlot, boolean simulate) {
		if(itemHandler == null || stack.m_41619_()) {
			return stack;
		}
		if(!requireEmptySlot) {
			return ItemHandlerHelper.insertItem(itemHandler, stack, simulate);
		}
		for(int slot = 0; slot < itemHandler.getSlots(); ++slot) {
			if(itemHandler.getStackInSlot(slot).m_41619_()) {
				stack = itemHandler.insertItem(slot, stack, simulate);
				if(stack.m_41619_()) {
					return ItemStack.f_41583_;
				}
			}
		}
		return stack;
	}

	@Override
	public ItemStack fillVolume(BlockEntity blockEntity, Direction direction, ItemStack stack, boolean simulate) {
		if(blockEntity == null || stack.m_41619_()) {
			return stack;
		}
		if(stack.m_41720_() instanceof IVolumePackageItem vPackage &&
				vPackage.getVolumeType(stack) != null &&
				vPackage.getVolumeType(stack).hasBlockCapability(blockEntity, direction)) {
			IVolumeType vType = vPackage.getVolumeType(stack);
			stack = stack.m_41777_();
			IVolumeStackWrapper vStack = vPackage.getVolumeStack(stack);
			while(!stack.m_41619_()) {
				int simulateFilled = vType.fill(blockEntity, direction, vStack, true);
				if(simulateFilled == vStack.getAmount()) {
					if(!simulate) {
						vType.fill(blockEntity, direction, vStack, false);
					}
					stack.m_41774_(1);
					if(stack.m_41619_()) {
						return ItemStack.f_41583_;
					}
				}
				else {
					break;
				}
			}
		}
		return stack;
	}

	@Override
	public Runnable conditionalRunnable(BooleanSupplier conditionSupplier, Supplier<Runnable> trueRunnable, Supplier<Runnable> falseRunnable) {
		return ()->(conditionSupplier.getAsBoolean() ? trueRunnable : falseRunnable).get().run();
	}

	@Override
	public <T> Supplier<T> conditionalSupplier(BooleanSupplier conditionSupplier, Supplier<Supplier<T>> trueSupplier, Supplier<Supplier<T>> falseSupplier) {
		return ()->(conditionSupplier.getAsBoolean() ? trueSupplier : falseSupplier).get().get();
	}

	public void setServer(MinecraftServer server) {
		MiscHelper.server = server;
	}

	@Override
	public RecipeManager getRecipeManager() {
		return server != null ? server.m_129894_() :
			DistExecutor.unsafeCallWhenOn(Dist.CLIENT, ()->()->Minecraft.m_91087_().f_91073_.m_7465_());
	}
}
