package thelm.packagedauto.client;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.lwjgl.opengl.GL11;

import com.google.common.primitives.Doubles;

import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.BufferBuilder;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3d;
import net.minecraftforge.client.event.RenderWorldLastEvent;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.fml.common.eventhandler.SubscribeEvent;
import net.minecraftforge.fml.common.gameevent.TickEvent;
import thelm.packagedauto.api.DirectionalGlobalPos;
import thelm.packagedauto.item.ItemDistributorMarker;
import thelm.packagedauto.item.ItemProxyMarker;

// Based on Botania, Scannables, and AE2
public class WorldOverlayRenderer {

	public static final WorldOverlayRenderer INSTANCE = new WorldOverlayRenderer();
	public static final Vec3d BLOCK_SIZE = new Vec3d(1, 1, 1);

	private WorldOverlayRenderer() {}

	private Minecraft mc;
	private List<DirectionalMarkerInfo> directionalMarkers = new LinkedList<>();
	private List<SizedMarkerInfo> sizedMarkers = new LinkedList<>();
	private List<BeamInfo> beams = new LinkedList<>();

	public void onConstruct() {
		mc = Minecraft.func_71410_x();
		MinecraftForge.EVENT_BUS.register(this);
	}

	@SubscribeEvent
	public void onClientTick(TickEvent.ClientTickEvent event) {
		if(event.phase != TickEvent.Phase.END || mc.field_71441_e == null || mc.field_71439_g == null || mc.func_147113_T()) {
			return;
		}
		for(EnumHand hand : EnumHand.values()) {
			ItemStack stack = mc.field_71439_g.func_184586_b(hand);
			if(stack.func_77973_b() == ItemDistributorMarker.INSTANCE) {
				DirectionalGlobalPos globalPos = ItemDistributorMarker.INSTANCE.getDirectionalGlobalPos(stack);
				if(globalPos != null) {
					addDirectionalMarkers(Collections.singletonList(globalPos), 0x00FFFF, 1);
				}
			}
			if(stack.func_77973_b() == ItemProxyMarker.INSTANCE) {
				DirectionalGlobalPos globalPos = ItemProxyMarker.INSTANCE.getDirectionalGlobalPos(stack);
				if(globalPos != null) {
					addDirectionalMarkers(Collections.singletonList(globalPos), 0xFF7F00, 1);
				}
			}
		}
	}

	@SubscribeEvent
	public void onRenderWorldLast(RenderWorldLastEvent event) {
		render(event.getPartialTicks());
	}

	public void addDirectionalMarkers(List<DirectionalGlobalPos> positions, int color, int lifetime) {
		directionalMarkers.add(new DirectionalMarkerInfo(positions, color, lifetime));
	}

	public void addSizedMarker(Vec3d lowerCorner, Vec3d size, int color, int lifetime) {
		sizedMarkers.add(new SizedMarkerInfo(lowerCorner, size, color, lifetime));
	}

	public void addBeams(Vec3d source, List<Vec3d> deltas, int color, int lifetime, boolean fadeout) {
		beams.add(new BeamInfo(source, deltas, color, lifetime, fadeout));
	}

	public void render(float partialTick) {
		int currentTick = RenderTimer.INSTANCE.getTicks();
		directionalMarkers.removeIf(marker->marker.shouldRemove(currentTick));
		sizedMarkers.removeIf(marker->marker.shouldRemove(currentTick));
		beams.removeIf(beam->beam.shouldRemove(currentTick));

		float renderTick = currentTick+partialTick;
		double viewerPosX = mc.func_175598_ae().field_78730_l;
		double viewerPosY = mc.func_175598_ae().field_78731_m;
		double viewerPosZ = mc.func_175598_ae().field_78728_n;

		GlStateManager.func_179094_E();
		GL11.glPushAttrib(GL11.GL_LIGHTING_BIT);
		GlStateManager.func_179129_p();
		GlStateManager.func_179090_x();
		GlStateManager.func_179147_l();
		GL11.glLineWidth(4);

		GlStateManager.func_179097_i();

		for(DirectionalMarkerInfo marker : directionalMarkers) {
			int r = marker.color>>16&0xFF;
			int g = marker.color>> 8&0xFF;
			int b = marker.color    &0xFF;

			for(DirectionalGlobalPos globalPos : marker.positions) {
				if(globalPos.dimension() != mc.field_71441_e.field_73011_w.getDimension()) {
					continue;
				}

				int range = 64;
				BlockPos blockPos = globalPos.blockPos();
				double distX = viewerPosX-blockPos.func_177958_n()-0.5;
				double distY = viewerPosY-blockPos.func_177956_o()-0.5;
				double distZ = viewerPosZ-blockPos.func_177952_p()-0.5;
				if(Doubles.max(Math.abs(distX), Math.abs(distY), Math.abs(distZ)) > range) {
					continue;
				}

				GlStateManager.func_179094_E();
				GlStateManager.func_179137_b(blockPos.func_177958_n()-viewerPosX, blockPos.func_177956_o()-viewerPosY, blockPos.func_177952_p()-viewerPosZ);

				EnumFacing direction = globalPos.direction();
				drawMarker(BLOCK_SIZE, direction, r, g, b, 127);
				drawMarker(BLOCK_SIZE, null, r, g, b, 255);

				GlStateManager.func_179121_F();
			}
		}

		GlStateManager.func_179126_j();

		for(SizedMarkerInfo marker : sizedMarkers) {
			Vec3d lowerCorner = marker.lowerCorner;

			GlStateManager.func_179094_E();
			GlStateManager.func_179137_b(lowerCorner.field_72450_a-viewerPosX, lowerCorner.field_72448_b-viewerPosY, lowerCorner.field_72449_c-viewerPosZ);

			int r = marker.color>>16&0xFF;
			int g = marker.color>> 8&0xFF;
			int b = marker.color    &0xFF;
			drawMarker(marker.size, null, r, g, b, 255);

			GlStateManager.func_179121_F();
		}

		GL11.glLineWidth(3);

		for(BeamInfo beam : beams) {
			Vec3d source = beam.source;

			GlStateManager.func_179094_E();
			GlStateManager.func_179137_b(source.field_72450_a-viewerPosX, source.field_72448_b-viewerPosY, source.field_72449_c-viewerPosZ);

			int r = beam.color>>16&0xFF;
			int g = beam.color>> 8&0xFF;
			int b = beam.color    &0xFF;
			int a = (int)(beam.getAlpha(renderTick)*255);
			for(Vec3d delta : beam.deltas) {
				drawBeam(delta, r, g, b, a);
			}

			GlStateManager.func_179121_F();
		}

		GlStateManager.func_179089_o();
		GlStateManager.func_179098_w();
		GlStateManager.func_179084_k();
		GL11.glPopAttrib();
		GlStateManager.func_179121_F();
	}

	public void drawMarker(Vec3d delta, EnumFacing direction, int r, int g, int b, int a) {
		Tessellator tessellator = Tessellator.func_178181_a();
		BufferBuilder buffer = tessellator.func_178180_c();
		float x = (float)delta.field_72450_a;
		float y = (float)delta.field_72448_b;
		float z = (float)delta.field_72449_c;
		buffer.func_181668_a(direction == null ? GL11.GL_LINES : GL11.GL_QUADS, DefaultVertexFormats.field_181706_f);
		if(direction == null || direction == EnumFacing.NORTH) {
			// Face North, Edge Bottom
			buffer.func_181662_b(0, 0, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, 0, 0).func_181669_b(r, g, b, a).func_181675_d();
			// Face North, Edge Top
			buffer.func_181662_b(x, y, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(0, y, 0).func_181669_b(r, g, b, a).func_181675_d();
		}
		if(direction == null || direction == EnumFacing.SOUTH) {
			// Face South, Edge Bottom
			buffer.func_181662_b(x, 0, z).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(0, 0, z).func_181669_b(r, g, b, a).func_181675_d();
			// Face South, Edge Top
			buffer.func_181662_b(0, y, z).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, y, z).func_181669_b(r, g, b, a).func_181675_d();
		}
		if(direction == null || direction == EnumFacing.WEST) {
			// Face West, Edge Bottom
			buffer.func_181662_b(0, 0, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(0, 0, z).func_181669_b(r, g, b, a).func_181675_d();
			// Face West, Edge Top
			buffer.func_181662_b(0, y, z).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(0, y, 0).func_181669_b(r, g, b, a).func_181675_d();
		}
		if(direction == null || direction == EnumFacing.EAST) {
			// Face East, Edge Bottom
			buffer.func_181662_b(x, 0, z).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, 0, 0).func_181669_b(r, g, b, a).func_181675_d();
			// Face East, Edge Top
			buffer.func_181662_b(x, y, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, y, z).func_181669_b(r, g, b, a).func_181675_d();
		}
		if(direction == EnumFacing.DOWN) {
			// Face Down
			buffer.func_181662_b(0, 0, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, 0, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, 0, z).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(0, 0, z).func_181669_b(r, g, b, a).func_181675_d();
		}
		if(direction == EnumFacing.UP) {
			// Face Up
			buffer.func_181662_b(0, y, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, y, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, y, z).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(0, y, z).func_181669_b(r, g, b, a).func_181675_d();
		}
		if(direction == null) {
			// Face North, Edge West
			buffer.func_181662_b(0, 0, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(0, y, 0).func_181669_b(r, g, b, a).func_181675_d();
			// Face North, Edge East
			buffer.func_181662_b(x, y, 0).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, 0, 0).func_181669_b(r, g, b, a).func_181675_d();
			// Face South, Edge East
			buffer.func_181662_b(x, 0, z).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(x, y, z).func_181669_b(r, g, b, a).func_181675_d();
			// Face South, Edge West
			buffer.func_181662_b(0, y, z).func_181669_b(r, g, b, a).func_181675_d();
			buffer.func_181662_b(0, 0, z).func_181669_b(r, g, b, a).func_181675_d();
		}
		tessellator.func_78381_a();
	}

	public void drawBeam(Vec3d delta, int r, int g, int b, int a) {
		Tessellator tessellator = Tessellator.func_178181_a();
		BufferBuilder buffer = tessellator.func_178180_c();
		float x = (float)delta.field_72450_a;
		float y = (float)delta.field_72448_b;
		float z = (float)delta.field_72449_c;
		buffer.func_181668_a(GL11.GL_LINES, DefaultVertexFormats.field_181706_f);
		buffer.func_181662_b(0, 0, 0).func_181669_b(r, g, b, a).func_181675_d();
		buffer.func_181662_b(x, y, z).func_181669_b(r, g, b, a).func_181675_d();
		tessellator.func_78381_a();
	}

	public static class DirectionalMarkerInfo {

		private List<DirectionalGlobalPos> positions;
		private int color;
		private int lifetime;
		private int startTick;

		public DirectionalMarkerInfo(List<DirectionalGlobalPos> positions, int color, int lifetime) {
			this.positions = positions;
			this.color = color;
			this.lifetime = lifetime;
			startTick = RenderTimer.INSTANCE.getTicks();
		}

		public boolean shouldRemove(int currentTick) {
			if(currentTick < startTick) {
				currentTick += 0x1FFFFF;
			}
			return currentTick-startTick >= lifetime;
		}
	}

	public static class SizedMarkerInfo {

		private Vec3d lowerCorner;
		private Vec3d size;
		private int color;
		private int lifetime;
		private int startTick;

		public SizedMarkerInfo(Vec3d lowerCorner, Vec3d size, int color, int lifetime) {
			this.lowerCorner = lowerCorner;
			this.size = size;
			this.color = color;
			this.lifetime = lifetime;
			startTick = RenderTimer.INSTANCE.getTicks();
		}

		public boolean shouldRemove(int currentTick) {
			if(currentTick < startTick) {
				currentTick += 0x1FFFFF;
			}
			return currentTick-startTick >= lifetime;
		}
	}

	public static class BeamInfo {

		private Vec3d source;
		private List<Vec3d> deltas;
		private int color;
		private int lifetime;
		private int startTick;
		private boolean fadeout;

		public BeamInfo(Vec3d source, List<Vec3d> deltas, int color, int lifetime, boolean fadeout) {
			this.source = source;
			this.deltas = deltas;
			this.color = color;
			this.lifetime = lifetime;
			this.fadeout = fadeout;
			startTick = RenderTimer.INSTANCE.getTicks();
		}

		public boolean shouldRemove(int currentTick) {
			if(currentTick < startTick) {
				currentTick += 0x1FFFFF;
			}
			return currentTick-startTick >= lifetime;
		}

		public float getAlpha(float renderTick) {
			if(!fadeout) {
				return 1;
			}
			float diff = renderTick-startTick;
			if(diff < 0) {
				diff += 0x1FFFFF;
			}
			float factor = Math.min(diff/lifetime, 1);
			return 1-factor*factor;
		}
	}
}
