package thelm.packagedauto.client;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.OptionalDouble;

import org.lwjgl.opengl.GL11;

import com.google.common.collect.ImmutableMap;
import com.google.common.primitives.Doubles;
import com.mojang.blaze3d.matrix.MatrixStack;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.IVertexBuilder;

import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.BufferBuilder;
import net.minecraft.client.renderer.IRenderTypeBuffer;
import net.minecraft.client.renderer.RenderType;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.client.renderer.vertex.VertexFormat;
import net.minecraft.item.ItemStack;
import net.minecraft.util.Direction;
import net.minecraft.util.Hand;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.vector.Matrix4f;
import net.minecraft.util.math.vector.Vector3d;
import net.minecraftforge.client.event.RenderWorldLastEvent;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.event.TickEvent;
import thelm.packagedauto.api.DirectionalGlobalPos;
import thelm.packagedauto.item.DistributorMarkerItem;
import thelm.packagedauto.item.ProxyMarkerItem;

// Based on Botania, Scannables, and AE2
public class WorldOverlayRenderer {

	public static final WorldOverlayRenderer INSTANCE = new WorldOverlayRenderer();
	public static final Vector3d BLOCK_SIZE = new Vector3d(1, 1, 1);

	private WorldOverlayRenderer() {}

	private Minecraft mc;
	private List<DirectionalMarkerInfo> directionalMarkers = new LinkedList<>();
	private List<SizedMarkerInfo> sizedMarkers = new LinkedList<>();
	private List<BeamInfo> beams = new LinkedList<>();

	public void onConstruct() {
		mc = Minecraft.func_71410_x();
		MinecraftForge.EVENT_BUS.addListener(this::onClientTick);
		MinecraftForge.EVENT_BUS.addListener(this::onRenderWorldLast);
	}

	public void onClientTick(TickEvent.ClientTickEvent event) {
		if(event.phase != TickEvent.Phase.END || mc.field_71441_e == null || mc.field_71439_g == null || mc.func_147113_T()) {
			return;
		}
		for(Hand hand : Hand.values()) {
			ItemStack stack = mc.field_71439_g.func_184586_b(hand);
			if(stack.func_77973_b() == DistributorMarkerItem.INSTANCE) {
				DirectionalGlobalPos globalPos = DistributorMarkerItem.INSTANCE.getDirectionalGlobalPos(stack);
				if(globalPos != null) {
					addDirectionalMarkers(Collections.singletonList(globalPos), 0x00FFFF, 1);
				}
			}
			if(stack.func_77973_b() == ProxyMarkerItem.INSTANCE) {
				DirectionalGlobalPos globalPos = ProxyMarkerItem.INSTANCE.getDirectionalGlobalPos(stack);
				if(globalPos != null) {
					addDirectionalMarkers(Collections.singletonList(globalPos), 0xFF7F00, 1);
				}
			}
		}
	}

	public void onRenderWorldLast(RenderWorldLastEvent event) {
		render(event.getMatrixStack(), event.getPartialTicks());
	}

	public void addDirectionalMarkers(List<DirectionalGlobalPos> positions, int color, int lifetime) {
		directionalMarkers.add(new DirectionalMarkerInfo(positions, color, lifetime));
	}

	public void addSizedMarker(Vector3d lowerCorner, Vector3d size, int color, int lifetime) {
		sizedMarkers.add(new SizedMarkerInfo(lowerCorner, size, color, lifetime));
	}

	public void addBeams(Vector3d source, List<Vector3d> deltas, int color, int lifetime, boolean fadeout) {
		beams.add(new BeamInfo(source, deltas, color, lifetime, fadeout));
	}

	public void render(MatrixStack matrixStack, float partialTick) {
		int currentTick = RenderTimer.INSTANCE.getTicks();
		directionalMarkers.removeIf(marker->marker.shouldRemove(currentTick));
		sizedMarkers.removeIf(marker->marker.shouldRemove(currentTick));
		beams.removeIf(beam->beam.shouldRemove(currentTick));

		float renderTick = currentTick+partialTick;
		Vector3d cameraPos = mc.field_71460_t.func_215316_n().func_216785_c();

		IRenderTypeBuffer.Impl buffers = RenderTypeHelper.BUFFERS;
		IVertexBuilder quadBuffer = buffers.getBuffer(RenderTypeHelper.MARKER_QUAD);
		IVertexBuilder lineBuffer = buffers.getBuffer(RenderTypeHelper.MARKER_LINE_4);

		for(DirectionalMarkerInfo marker : directionalMarkers) {
			int r = marker.color>>16&0xFF;
			int g = marker.color>> 8&0xFF;
			int b = marker.color    &0xFF;

			for(DirectionalGlobalPos globalPos : marker.positions) {
				if(!globalPos.dimension().equals(mc.field_71441_e.func_234923_W_())) {
					continue;
				}

				int range = 64;
				BlockPos blockPos = globalPos.blockPos();
				Vector3d distVec = cameraPos.func_178788_d(Vector3d.func_237489_a_(blockPos));
				if(Doubles.max(Math.abs(distVec.field_72450_a), Math.abs(distVec.field_72448_b), Math.abs(distVec.field_72449_c)) > range) {
					continue;
				}

				matrixStack.func_227860_a_();
				matrixStack.func_227861_a_(blockPos.func_177958_n()-cameraPos.field_72450_a, blockPos.func_177956_o()-cameraPos.field_72448_b, blockPos.func_177952_p()-cameraPos.field_72449_c);

				Direction direction = globalPos.direction();
				addMarkerVertices(matrixStack, quadBuffer, BLOCK_SIZE, direction, r, g, b, 127);
				addMarkerVertices(matrixStack, lineBuffer, BLOCK_SIZE, null, r, g, b, 255);

				matrixStack.func_227865_b_();
			}
		}

		RenderSystem.disableDepthTest();
		buffers.func_228461_a_();
		RenderSystem.enableDepthTest();

		lineBuffer = buffers.getBuffer(RenderTypeHelper.MARKER_LINE_4);
		
		for(SizedMarkerInfo marker : sizedMarkers) {
			Vector3d lowerCorner = marker.lowerCorner;

			matrixStack.func_227860_a_();
			matrixStack.func_227861_a_(lowerCorner.field_72450_a-cameraPos.field_72450_a, lowerCorner.field_72448_b-cameraPos.field_72448_b, lowerCorner.field_72449_c-cameraPos.field_72449_c);

			int r = marker.color>>16&0xFF;
			int g = marker.color>> 8&0xFF;
			int b = marker.color    &0xFF;
			addMarkerVertices(matrixStack, lineBuffer, marker.size, null, r, g, b, 255);

			matrixStack.func_227865_b_();
		}

		buffers.func_228461_a_();
		
		lineBuffer = buffers.getBuffer(RenderTypeHelper.BEAM_LINE_3);
		
		for(BeamInfo beam : beams) {
			Vector3d source = beam.source;

			matrixStack.func_227860_a_();
			matrixStack.func_227861_a_(source.field_72450_a-cameraPos.field_72450_a, source.field_72448_b-cameraPos.field_72448_b, source.field_72449_c-cameraPos.field_72449_c);

			int r = beam.color>>16&0xFF;
			int g = beam.color>> 8&0xFF;
			int b = beam.color    &0xFF;
			int a = (int)(beam.getAlpha(renderTick)*255);
			for(Vector3d delta : beam.deltas) {
				addBeamVertices(matrixStack, lineBuffer, delta, r, g, b, a);
			}

			matrixStack.func_227865_b_();
		}

		buffers.func_228461_a_();
	}

	public void addMarkerVertices(MatrixStack matrixStack, IVertexBuilder buffer, Vector3d delta, Direction direction, int r, int g, int b, int a) {
		Matrix4f pose = matrixStack.func_227866_c_().func_227870_a_();
		float x = (float)delta.field_72450_a;
		float y = (float)delta.field_72448_b;
		float z = (float)delta.field_72449_c;
		if(direction == null || direction == Direction.NORTH) {
			// Face North, Edge Bottom
			buffer.func_227888_a_(pose, 0, 0, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, 0, 0).func_225586_a_(r, g, b, a).func_181675_d();
			// Face North, Edge Top
			buffer.func_227888_a_(pose, x, y, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, 0, y, 0).func_225586_a_(r, g, b, a).func_181675_d();
		}
		if(direction == null || direction == Direction.SOUTH) {
			// Face South, Edge Bottom
			buffer.func_227888_a_(pose, x, 0, z).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, 0, 0, z).func_225586_a_(r, g, b, a).func_181675_d();
			// Face South, Edge Top
			buffer.func_227888_a_(pose, 0, y, z).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, y, z).func_225586_a_(r, g, b, a).func_181675_d();
		}
		if(direction == null || direction == Direction.WEST) {
			// Face West, Edge Bottom
			buffer.func_227888_a_(pose, 0, 0, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, 0, 0, z).func_225586_a_(r, g, b, a).func_181675_d();
			// Face West, Edge Top
			buffer.func_227888_a_(pose, 0, y, z).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, 0, y, 0).func_225586_a_(r, g, b, a).func_181675_d();
		}
		if(direction == null || direction == Direction.EAST) {
			// Face East, Edge Bottom
			buffer.func_227888_a_(pose, x, 0, z).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, 0, 0).func_225586_a_(r, g, b, a).func_181675_d();
			// Face East, Edge Top
			buffer.func_227888_a_(pose, x, y, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, y, z).func_225586_a_(r, g, b, a).func_181675_d();
		}
		if(direction == Direction.DOWN) {
			// Face Down
			buffer.func_227888_a_(pose, 0, 0, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, 0, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, 0, z).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, 0, 0, z).func_225586_a_(r, g, b, a).func_181675_d();
		}
		if(direction == Direction.UP) {
			// Face Up
			buffer.func_227888_a_(pose, 0, y, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, y, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, y, z).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, 0, y, z).func_225586_a_(r, g, b, a).func_181675_d();
		}
		if(direction == null) {
			// Face North, Edge West
			buffer.func_227888_a_(pose, 0, 0, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, 0, y, 0).func_225586_a_(r, g, b, a).func_181675_d();
			// Face North, Edge East
			buffer.func_227888_a_(pose, x, y, 0).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, 0, 0).func_225586_a_(r, g, b, a).func_181675_d();
			// Face South, Edge East
			buffer.func_227888_a_(pose, x, 0, z).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, x, y, z).func_225586_a_(r, g, b, a).func_181675_d();
			// Face South, Edge West
			buffer.func_227888_a_(pose, 0, y, z).func_225586_a_(r, g, b, a).func_181675_d();
			buffer.func_227888_a_(pose, 0, 0, z).func_225586_a_(r, g, b, a).func_181675_d();
		}
	}

	public void addBeamVertices(MatrixStack matrixStack, IVertexBuilder buffer, Vector3d delta, int r, int g, int b, int a) {
		Matrix4f pose = matrixStack.func_227866_c_().func_227870_a_();
		float x = (float)delta.field_72450_a;
		float y = (float)delta.field_72448_b;
		float z = (float)delta.field_72449_c;
		buffer.func_227888_a_(pose, 0, 0, 0).func_225586_a_(r, g, b, a).func_181675_d();
		buffer.func_227888_a_(pose, x, y, z).func_225586_a_(r, g, b, a).func_181675_d();
	}

	public static class DirectionalMarkerInfo {

		private List<DirectionalGlobalPos> positions;
		private int color;
		private int lifetime;
		private int startTick;

		public DirectionalMarkerInfo(List<DirectionalGlobalPos> positions, int color, int lifetime) {
			this.positions = positions;
			this.color = color;
			this.lifetime = lifetime;
			startTick = RenderTimer.INSTANCE.getTicks();
		}

		public boolean shouldRemove(int currentTick) {
			if(currentTick < startTick) {
				currentTick += 0x1FFFFF;
			}
			return currentTick-startTick >= lifetime;
		}
	}

	public static class SizedMarkerInfo {

		private Vector3d lowerCorner;
		private Vector3d size;
		private int color;
		private int lifetime;
		private int startTick;

		public SizedMarkerInfo(Vector3d lowerCorner, Vector3d size, int color, int lifetime) {
			this.lowerCorner = lowerCorner;
			this.size = size;
			this.color = color;
			this.lifetime = lifetime;
			startTick = RenderTimer.INSTANCE.getTicks();
		}

		public boolean shouldRemove(int currentTick) {
			if(currentTick < startTick) {
				currentTick += 0x1FFFFF;
			}
			return currentTick-startTick >= lifetime;
		}
	}

	public static class BeamInfo {

		private Vector3d source;
		private List<Vector3d> deltas;
		private int color;
		private int lifetime;
		private int startTick;
		private boolean fadeout;

		public BeamInfo(Vector3d source, List<Vector3d> deltas, int color, int lifetime, boolean fadeout) {
			this.source = source;
			this.deltas = deltas;
			this.color = color;
			this.lifetime = lifetime;
			this.fadeout = fadeout;
			startTick = RenderTimer.INSTANCE.getTicks();
		}

		public boolean shouldRemove(int currentTick) {
			if(currentTick < startTick) {
				currentTick += 0x1FFFFF;
			}
			return currentTick-startTick >= lifetime;
		}

		public float getAlpha(float renderTick) {
			if(!fadeout) {
				return 1;
			}
			float diff = renderTick-startTick;
			if(diff < 0) {
				diff += 0x1FFFFF;
			}
			float factor = Math.min(diff/lifetime, 1);
			return 1-factor*factor;
		}
	}

	public static class RenderTypeHelper extends RenderType {

		private RenderTypeHelper(String name, VertexFormat format, int mode, int bufferSize, boolean affectsCrumbling, boolean sortOnUpload, Runnable setupState, Runnable clearState) {
			super(name, format, mode, bufferSize, affectsCrumbling, sortOnUpload, setupState, clearState);
		}

		public static final RenderType MARKER_LINE_4;
		public static final RenderType MARKER_QUAD;
		public static final RenderType BEAM_LINE_3;
		public static final IRenderTypeBuffer.Impl BUFFERS;

		static {
			MARKER_LINE_4 = func_228633_a_("packagedauto:marker_line_4",
					DefaultVertexFormats.field_181706_f, GL11.GL_LINES, 8192, false, false,
					State.func_228694_a_().
					func_228720_a_(new LineState(OptionalDouble.of(4))).
					func_228718_a_(field_239235_M_).
					func_228727_a_(field_228496_F_).
					func_228715_a_(field_228492_B_).
					func_228714_a_(field_228491_A_).
					func_228728_a_(false));
			MARKER_QUAD = func_228633_a_("packagedauto:marker_quad",
					DefaultVertexFormats.field_181706_f, GL11.GL_QUADS, 1024, false, false,
					State.func_228694_a_().
					func_228718_a_(field_239235_M_).
					func_228726_a_(field_228515_g_).
					func_228727_a_(field_228496_F_).
					func_228715_a_(field_228492_B_).
					func_228714_a_(field_228491_A_).
					func_228728_a_(false));
			BEAM_LINE_3 = func_228633_a_("packagedauto:beam_line_3",
					DefaultVertexFormats.field_181706_f, GL11.GL_LINES, 8192, false, false,
					State.func_228694_a_().
					func_228720_a_(new LineState(OptionalDouble.of(3))).
					func_228718_a_(field_239235_M_).
					func_228726_a_(field_228515_g_).
					func_228721_a_(field_241712_U_).
					func_228727_a_(field_228495_E_).
					func_228714_a_(field_228491_A_).
					func_228728_a_(false));
			BUFFERS = IRenderTypeBuffer.func_228456_a_(
					ImmutableMap.of(MARKER_LINE_4, new BufferBuilder(MARKER_LINE_4.func_228662_o_()),
							MARKER_QUAD, new BufferBuilder(MARKER_QUAD.func_228662_o_()),
							BEAM_LINE_3, new BufferBuilder(BEAM_LINE_3.func_228662_o_())),
					Tessellator.func_178181_a().func_178180_c());
		}
	}
}
