package thelm.packagedauto.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.function.BooleanSupplier;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;

import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntRBTreeMap;
import it.unimi.dsi.fastutil.objects.ObjectRBTreeSet;
import net.minecraft.client.Minecraft;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.RecipeManager;
import net.minecraft.nbt.CompoundNBT;
import net.minecraft.nbt.ListNBT;
import net.minecraft.network.PacketBuffer;
import net.minecraft.server.MinecraftServer;
import net.minecraft.util.NonNullList;
import net.minecraft.util.ResourceLocation;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.fml.DistExecutor;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.ItemHandlerHelper;
import thelm.packagedauto.api.IMiscHelper;
import thelm.packagedauto.api.IPackagePattern;
import thelm.packagedauto.api.IPackageRecipeInfo;
import thelm.packagedauto.api.IPackageRecipeType;
import thelm.packagedauto.api.PackagedAutoApi;

public class MiscHelper implements IMiscHelper {

	public static final MiscHelper INSTANCE = new MiscHelper();

	private static final Cache<CompoundNBT, IPackageRecipeInfo> RECIPE_CACHE = CacheBuilder.newBuilder().maximumSize(500).build();
	private static final Logger LOGGER = LogManager.getLogger();

	private static MinecraftServer server;

	private MiscHelper() {}

	@Override
	public List<ItemStack> condenseStacks(IInventory inventory) {
		List<ItemStack> stacks = new ArrayList<>(inventory.func_70302_i_());
		for(int i = 0; i < inventory.func_70302_i_(); ++i) {
			stacks.add(inventory.func_70301_a(i));
		}
		return condenseStacks(stacks);
	}

	@Override
	public List<ItemStack> condenseStacks(IItemHandler itemHandler) {
		List<ItemStack> stacks = new ArrayList<>(itemHandler.getSlots());
		for(int i = 0; i < itemHandler.getSlots(); ++i) {
			stacks.add(itemHandler.getStackInSlot(i));
		}
		return condenseStacks(stacks);
	}

	@Override
	public List<ItemStack> condenseStacks(ItemStack... stacks) {
		return condenseStacks(Arrays.asList(stacks));
	}

	@Override
	public List<ItemStack> condenseStacks(Stream<ItemStack> stacks) {
		return condenseStacks(stacks.collect(Collectors.toList()));
	}

	@Override
	public List<ItemStack> condenseStacks(Iterable<ItemStack> stacks) {
		return condenseStacks(stacks instanceof List<?> ? (List<ItemStack>)stacks : Lists.newArrayList(stacks));
	}

	@Override
	public List<ItemStack> condenseStacks(List<ItemStack> stacks) {
		return condenseStacks(stacks, false);
	}

	@Override
	public List<ItemStack> condenseStacks(List<ItemStack> stacks, boolean ignoreStackSize) {
		Object2IntRBTreeMap<Pair<Item, CompoundNBT>> map = new Object2IntRBTreeMap<>(
				Comparator.comparing(pair->Pair.of(pair.getLeft().getRegistryName(), ""+pair.getRight())));
		for(ItemStack stack : stacks) {
			if(stack.func_190926_b()) {
				continue;
			}
			Pair<Item, CompoundNBT> pair = Pair.of(stack.func_77973_b(), stack.func_77978_p());
			if(!map.containsKey(pair)) {
				map.put(pair, 0);
			}
			map.addTo(pair, stack.func_190916_E());
		}
		List<ItemStack> list = new ArrayList<>();
		for(Object2IntMap.Entry<Pair<Item, CompoundNBT>> entry : map.object2IntEntrySet()) {
			Pair<Item, CompoundNBT> pair = entry.getKey();
			int count = entry.getIntValue();
			Item item = pair.getLeft();
			CompoundNBT nbt = pair.getRight();
			if(ignoreStackSize) {
				ItemStack toAdd = new ItemStack(item, count);
				toAdd.func_77982_d(nbt);
				list.add(toAdd);
			}
			else {
				while(count > 0) {
					ItemStack toAdd = new ItemStack(item, 1);
					toAdd.func_77982_d(nbt);
					int limit = item.getItemStackLimit(toAdd);
					toAdd.func_190920_e(Math.min(count, limit));
					list.add(toAdd);
					count -= limit;
				}
			}
		}
		map.clear();
		return list;
	}

	@Override
	public ListNBT saveAllItems(ListNBT tagList, List<ItemStack> list) {
		return saveAllItems(tagList, list, "Index");
	}

	@Override
	public ListNBT saveAllItems(ListNBT tagList, List<ItemStack> list, String indexKey) {
		for(int i = 0; i < list.size(); ++i) {
			ItemStack stack = list.get(i);
			boolean empty = stack.func_190926_b();
			if(!empty || i == list.size()-1) {
				if(empty) {
					// Ensure that the end-of-list stack if empty is always the default empty stack
					stack = new ItemStack((Item)null);
				}
				CompoundNBT nbt = new CompoundNBT();
				nbt.func_74774_a(indexKey, (byte)i);
				saveItemWithLargeCount(nbt, stack);
				tagList.add(nbt);
			}
		}
		return tagList;
	}

	@Override
	public void loadAllItems(ListNBT tagList, List<ItemStack> list) {
		loadAllItems(tagList, list, "Index");
	}

	@Override
	public void loadAllItems(ListNBT tagList, List<ItemStack> list, String indexKey) {
		list.clear();
		try {
			for(int i = 0; i < tagList.size(); ++i) {
				CompoundNBT nbt = tagList.func_150305_b(i);
				int j = nbt.func_74771_c(indexKey) & 255;
				while(j >= list.size()) {
					list.add(ItemStack.field_190927_a);
				}
				if(j >= 0)  {
					ItemStack stack = loadItemWithLargeCount(nbt);
					list.set(j, stack.func_190926_b() ? ItemStack.field_190927_a : stack);
				}
			}
		}
		catch(UnsupportedOperationException | IndexOutOfBoundsException e) {}
	}

	@Override
	public CompoundNBT saveItemWithLargeCount(CompoundNBT nbt, ItemStack stack) {
		stack.func_77955_b(nbt);
		int count = stack.func_190916_E();
		if((byte)count == count) {
			nbt.func_74774_a("Count", (byte)count);
		}
		else if((short)count == count) {
			nbt.func_74777_a("Count", (short)count);
		}
		else {
			nbt.func_74768_a("Count", (short)count);
		}
		return nbt;
	}

	@Override
	public ItemStack loadItemWithLargeCount(CompoundNBT nbt) {
		ItemStack stack = ItemStack.func_199557_a(nbt);
		stack.func_190920_e(nbt.func_74762_e("Count"));
		return stack;
	}

	@Override
	public void writeItemWithLargeCount(PacketBuffer buf, ItemStack stack) {
		if(stack.func_190926_b()) {
			buf.writeBoolean(false);
			return;
		}
		buf.writeBoolean(true);
		buf.func_150787_b(Item.func_150891_b(stack.func_77973_b()));
		buf.func_150787_b(stack.func_190916_E());
		CompoundNBT nbt = null;
		if(stack.func_77973_b().isDamageable(stack) || stack.func_77973_b().func_77651_p()) {
			nbt = stack.getShareTag();
		}
		buf.func_150786_a(nbt);
	}

	@Override
	public ItemStack readItemWithLargeCount(PacketBuffer buf) {
		if(!buf.readBoolean()) {
			return ItemStack.field_190927_a;
		}
		int id = buf.func_150792_a();
		int count = buf.func_150792_a();
		ItemStack stack = new ItemStack(Item.func_150899_d(id), count);
		stack.func_77973_b().readShareTag(stack, buf.func_150793_b());
		return stack;
	}

	@Override
	public IPackagePattern getPattern(IPackageRecipeInfo recipeInfo, int index) {
		return new PackagePattern(recipeInfo, index);
	}

	@Override
	public List<ItemStack> getRemainingItems(IInventory inventory) {
		return getRemainingItems(IntStream.range(0, inventory.func_70302_i_()).mapToObj(inventory::func_70301_a).collect(Collectors.toList()));
	}

	@Override
	public List<ItemStack> getRemainingItems(IInventory inventory, int minInclusive, int maxExclusive) {
		return getRemainingItems(IntStream.range(minInclusive, maxExclusive).mapToObj(inventory::func_70301_a).collect(Collectors.toList()));
	}

	@Override
	public List<ItemStack> getRemainingItems(ItemStack... stacks) {
		return getRemainingItems(Arrays.asList(stacks));
	}

	@Override
	public List<ItemStack> getRemainingItems(List<ItemStack> stacks) {
		NonNullList<ItemStack> ret = NonNullList.func_191197_a(stacks.size(), ItemStack.field_190927_a);
		for(int i = 0; i < ret.size(); i++) {
			ret.set(i, getContainerItem(stacks.get(i)));
		}
		return ret;
	}

	@Override
	public ItemStack getContainerItem(ItemStack stack) {
		if(stack.func_190926_b()) {
			return ItemStack.field_190927_a;
		}
		if(stack.func_77973_b().hasContainerItem(stack)) {
			stack = stack.func_77973_b().getContainerItem(stack);
			if(!stack.func_190926_b() && stack.func_77984_f() && stack.func_77952_i() > stack.func_77958_k()) {
				return ItemStack.field_190927_a;
			}
			return stack;
		}
		else {
			if(stack.func_190916_E() > 1) {
				stack = stack.func_77946_l();
				stack.func_190920_e(stack.func_190916_E() - 1);
				return stack;
			}
			return ItemStack.field_190927_a;
		}
	}

	@Override
	public ItemStack cloneStack(ItemStack stack, int stackSize) {
		if(stack.func_190926_b()) {
			return ItemStack.field_190927_a;
		}
		ItemStack retStack = stack.func_77946_l();
		retStack.func_190920_e(stackSize);
		return retStack;
	}

	@Override
	public boolean isEmpty(IItemHandler itemHandler) {
		if(itemHandler == null) {
			return false;
		}
		if(itemHandler.getSlots() == 0) {
			return false;
		}
		for(int i = 0; i < itemHandler.getSlots(); ++i) {
			if(!itemHandler.getStackInSlot(i).func_190926_b()) {
				return false;
			}
		}
		return true;
	}

	@Override
	public CompoundNBT writeRecipe(CompoundNBT nbt, IPackageRecipeInfo recipe) {
		nbt.func_74778_a("RecipeType", recipe.getRecipeType().getName().toString());
		recipe.write(nbt);
		return nbt;
	}

	@Override
	public IPackageRecipeInfo readRecipe(CompoundNBT nbt) {
		IPackageRecipeType recipeType = PackagedAutoApi.instance().getRecipeType(new ResourceLocation(nbt.func_74779_i("RecipeType")));
		if(recipeType != null) {
			IPackageRecipeInfo recipe = RECIPE_CACHE.getIfPresent(nbt);
			if(recipe != null) {
				return recipe;
			}
			recipe = recipeType.getNewRecipeInfo();
			recipe.read(nbt);
			RECIPE_CACHE.put(nbt, recipe);
			return recipe;
		}
		return null;
	}

	@Override
	public ListNBT writeRecipeList(ListNBT tagList, List<IPackageRecipeInfo> recipes) {
		for(IPackageRecipeInfo recipe : recipes) {
			tagList.add(writeRecipe(new CompoundNBT(), recipe));
		}
		return tagList;
	}

	@Override
	public List<IPackageRecipeInfo> readRecipeList(ListNBT tagList) {
		List<IPackageRecipeInfo> recipes = new ArrayList<>(tagList.size());
		for(int i = 0; i < tagList.size(); ++i) {
			IPackageRecipeInfo recipe = readRecipe(tagList.func_150305_b(i));
			if(recipe != null) {
				recipes.add(recipe);
			}
		}
		return recipes;
	}

	@Override
	public boolean recipeEquals(IPackageRecipeInfo recipeA, Object recipeInternalA, IPackageRecipeInfo recipeB, Object recipeInternalB) {
		if(!Objects.equals(recipeInternalA, recipeInternalB)) {
			return false;
		}
		List<ItemStack> inputsA = recipeA.getInputs();
		List<ItemStack> inputsB = recipeB.getInputs();
		if(inputsA.size() != inputsB.size()) {
			return false;
		}
		List<ItemStack> outputsA = recipeA.getOutputs();
		List<ItemStack> outputsB = recipeB.getOutputs();
		if(outputsA.size() != outputsB.size()) {
			return false;
		}
		for(int i = 0; i < inputsA.size(); ++i) {
			if(!ItemStack.func_77989_b(inputsA.get(i), inputsB.get(i))) {
				return false;
			}
		}
		for(int i = 0; i < outputsA.size(); ++i) {
			if(!ItemStack.func_77989_b(outputsA.get(i), outputsB.get(i))) {
				return false;
			}
		}
		return true;
	}

	@Override
	public int recipeHashCode(IPackageRecipeInfo recipe, Object recipeInternal) {
		List<ItemStack> inputs = recipe.getInputs();
		List<ItemStack> outputs = recipe.getOutputs();
		Function<ItemStack, Object[]> decompose = stack->new Object[] {
				stack.func_77973_b(), stack.func_190916_E(), stack.func_77978_p(),
		};
		Object[] toHash = {
				recipeInternal, inputs.stream().map(decompose).toArray(), outputs.stream().map(decompose).toArray(),
		};
		return Arrays.deepHashCode(toHash);
	}

	//Modified from Forestry
	@Override
	public boolean removeExactSet(List<ItemStack> offered, List<ItemStack> required, boolean simulate) {
		List<ItemStack> condensedRequired = condenseStacks(required, true);
		List<ItemStack> condensedOffered = condenseStacks(offered, true);
		f:for(ItemStack req : condensedRequired) {
			for(ItemStack offer : condensedOffered) {
				if(req.func_190916_E() <= offer.func_190916_E() && req.func_77969_a(offer) &&
						(!req.func_77942_o() || ItemStack.func_77970_a(req, offer))) {
					continue f;
				}
			}
			return false;
		}
		if(simulate) {
			return true;
		}
		for(ItemStack req : condensedRequired) {
			int count = req.func_190916_E();
			for(ItemStack offer : offered) {
				if(!offer.func_190926_b()) {
					if(req.func_77969_a(offer) && (!req.func_77942_o() || ItemStack.func_77970_a(req, offer))) {
						int toRemove = Math.min(count, offer.func_190916_E());
						offer.func_190918_g(toRemove);
						count -= toRemove;
						if(count == 0) {
							continue;
						}
					}
				}
			}
		}
		return true;
	}

	@Override
	public boolean arePatternsDisjoint(List<IPackagePattern> patternList) {
		ObjectRBTreeSet<Pair<Item, CompoundNBT>> set = new ObjectRBTreeSet<>(
				Comparator.comparing(pair->Pair.of(pair.getLeft().getRegistryName(), ""+pair.getRight())));
		for(IPackagePattern pattern : patternList) {
			List<ItemStack> condensedInputs = condenseStacks(pattern.getInputs(), true);
			for(ItemStack stack : condensedInputs) {
				Pair<Item, CompoundNBT> toAdd = Pair.of(stack.func_77973_b(), stack.func_77978_p());
				if(set.contains(toAdd)) {
					return false;
				}
				set.add(toAdd);
			}
		}
		set.clear();
		return true;
	}

	@Override
	public ItemStack insertItem(IItemHandler itemHandler, ItemStack stack, boolean requireEmptySlot, boolean simulate) {
		if(itemHandler == null || stack.func_190926_b()) {
			return stack;
		}
		if(!requireEmptySlot) {
			return ItemHandlerHelper.insertItem(itemHandler, stack, simulate);
		}
		for(int slot = 0; slot < itemHandler.getSlots(); ++slot) {
			if(itemHandler.getStackInSlot(slot).func_190926_b()) {
				stack = itemHandler.insertItem(slot, stack, simulate);
				if(stack.func_190926_b()) {
					return ItemStack.field_190927_a;
				}
			}
		}
		return stack;
	}

	@Override
	public Runnable conditionalRunnable(BooleanSupplier conditionSupplier, Supplier<Runnable> trueRunnable, Supplier<Runnable> falseRunnable) {
		return ()->(conditionSupplier.getAsBoolean() ? trueRunnable : falseRunnable).get().run();
	}

	@Override
	public <T> Supplier<T> conditionalSupplier(BooleanSupplier conditionSupplier, Supplier<Supplier<T>> trueSupplier, Supplier<Supplier<T>> falseSupplier) {
		return ()->(conditionSupplier.getAsBoolean() ? trueSupplier : falseSupplier).get().get();
	}

	public void setServer(MinecraftServer server) {
		MiscHelper.server = server;
	}

	@Override
	public RecipeManager getRecipeManager() {
		return server != null ? server.func_199529_aN() :
			DistExecutor.unsafeCallWhenOn(Dist.CLIENT, ()->()->Minecraft.func_71410_x().field_71441_e.func_199532_z());
	}
}
