package thelm.packagedauto.client;

import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.BufferBuilder;
import com.mojang.blaze3d.vertex.DefaultVertexFormat;
import com.mojang.blaze3d.vertex.PoseStack;
import com.mojang.blaze3d.vertex.Tesselator;
import com.mojang.blaze3d.vertex.VertexFormat;
import com.mojang.math.Matrix4f;

import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.GameRenderer;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.inventory.InventoryMenu;
import net.minecraft.world.level.material.Fluid;
import net.minecraftforge.client.extensions.common.IClientFluidTypeExtensions;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidType;

// Code from Refined Storage
public class FluidRenderer {

	public static final FluidRenderer INSTANCE = new FluidRenderer(16, 16, 16);

	private static final int TEX_WIDTH = 16;
	private static final int TEX_HEIGHT = 16;

	private final int width;
	private final int height;
	private final int minHeight;

	public FluidRenderer(int width, int height, int minHeight) {
		this.width = width;
		this.height = height;
		this.minHeight = minHeight;
	}

	private static TextureAtlasSprite getStillFluidSprite(FluidStack fluidStack) {
		ResourceLocation fluidStill = IClientFluidTypeExtensions.of(fluidStack.getFluid()).getStillTexture(fluidStack);
		return Minecraft.m_91087_().m_91258_(InventoryMenu.f_39692_).apply(fluidStill);
	}

	private static void setGLColorFromInt(int color) {
		float red = (color >> 16 & 0xFF) / 255.0F;
		float green = (color >> 8 & 0xFF) / 255.0F;
		float blue = (color & 0xFF) / 255.0F;
		float alpha = ((color >> 24) & 0xFF) / 255F;
		RenderSystem.m_157429_(red, green, blue, alpha);
	}

	private static void drawTextureWithMasking(Matrix4f matrix, float xCoord, float yCoord, TextureAtlasSprite textureSprite, int maskTop, int maskRight, float zLevel) {
		float uMin = textureSprite.m_118409_();
		float uMax = textureSprite.m_118410_();
		float vMin = textureSprite.m_118411_();
		float vMax = textureSprite.m_118412_();
		uMax = uMax - (maskRight / 16F * (uMax - uMin));
		vMin = vMin + (maskTop / 16F * (vMax - vMin));
		RenderSystem.m_157427_(GameRenderer::m_172817_);
		Tesselator tessellator = Tesselator.m_85913_();
		BufferBuilder bufferBuilder = tessellator.m_85915_();
		bufferBuilder.m_166779_(VertexFormat.Mode.QUADS, DefaultVertexFormat.f_85817_);
		bufferBuilder.m_85982_(matrix, xCoord, yCoord + 16, zLevel).m_7421_(uMin, vMax).m_5752_();
		bufferBuilder.m_85982_(matrix, xCoord + 16 - maskRight, yCoord + 16, zLevel).m_7421_(uMax, vMax).m_5752_();
		bufferBuilder.m_85982_(matrix, xCoord + 16 - maskRight, yCoord + maskTop, zLevel).m_7421_(uMax, vMin).m_5752_();
		bufferBuilder.m_85982_(matrix, xCoord, yCoord + maskTop, zLevel).m_7421_(uMin, vMin).m_5752_();
		tessellator.m_85914_();
	}

	public void render(PoseStack poseStack, int xPosition, int yPosition, FluidStack fluidStack) {
		render(poseStack, xPosition, yPosition, fluidStack, FluidType.BUCKET_VOLUME);
	}

	public void render(PoseStack poseStack, int xPosition, int yPosition, FluidStack fluidStack, int capacity) {
		RenderSystem.m_69478_();
		drawFluid(poseStack, xPosition, yPosition, fluidStack, capacity);
		RenderSystem.m_157429_(1, 1, 1, 1);
		RenderSystem.m_69461_();
	}

	private void drawFluid(PoseStack poseStack, int xPosition, int yPosition, FluidStack fluidStack, int capacity) {
		if(capacity <= 0 || fluidStack == null || fluidStack.isEmpty()) {
			return;
		}
		Fluid fluid = fluidStack.getFluid();
		TextureAtlasSprite fluidStillSprite = getStillFluidSprite(fluidStack);
		int fluidColor = IClientFluidTypeExtensions.of(fluid).getTintColor(fluidStack);
		int amount = fluidStack.getAmount();
		int scaledAmount = (amount * height) / capacity;
		if(amount > 0 && scaledAmount < minHeight) {
			scaledAmount = minHeight;
		}
		if(scaledAmount > height) {
			scaledAmount = height;
		}
		drawTiledSprite(poseStack, xPosition, yPosition, width, height, fluidColor, scaledAmount, fluidStillSprite);
	}

	private void drawTiledSprite(PoseStack poseStack, int xPosition, int yPosition, int tiledWidth, int tiledHeight, int color, int scaledAmount, TextureAtlasSprite sprite) {
		RenderSystem.m_157456_(0, InventoryMenu.f_39692_);
		Matrix4f matrix = poseStack.m_85850_().m_85861_();
		setGLColorFromInt(color);
		int xTileCount = tiledWidth / TEX_WIDTH;
		int xRemainder = tiledWidth - (xTileCount * TEX_WIDTH);
		int yTileCount = scaledAmount / TEX_HEIGHT;
		int yRemainder = scaledAmount - (yTileCount * TEX_HEIGHT);
		int yStart = yPosition + tiledHeight;
		for(int xTile = 0; xTile <= xTileCount; xTile++) {
			for(int yTile = 0; yTile <= yTileCount; yTile++) {
				int width = (xTile == xTileCount) ? xRemainder : TEX_WIDTH;
				int height = (yTile == yTileCount) ? yRemainder : TEX_HEIGHT;
				int x = xPosition + (xTile * TEX_WIDTH);
				int y = yStart - ((yTile + 1) * TEX_HEIGHT);
				if(width > 0 && height > 0) {
					int maskTop = TEX_HEIGHT - height;
					int maskRight = TEX_WIDTH - width;
					drawTextureWithMasking(matrix, x, y, sprite, maskTop, maskRight, 100);
				}
			}
		}
	}
}
