package thelm.packagedauto.client;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.OptionalDouble;

import com.google.common.primitives.Doubles;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.BufferBuilder;
import com.mojang.blaze3d.vertex.DefaultVertexFormat;
import com.mojang.blaze3d.vertex.PoseStack;
import com.mojang.blaze3d.vertex.Tesselator;
import com.mojang.blaze3d.vertex.VertexConsumer;
import com.mojang.blaze3d.vertex.VertexFormat;
import com.mojang.blaze3d.vertex.VertexFormat.Mode;
import com.mojang.math.Matrix3f;
import com.mojang.math.Matrix4f;

import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.MultiBufferSource;
import net.minecraft.client.renderer.RenderStateShard;
import net.minecraft.client.renderer.RenderType;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.phys.Vec3;
import net.minecraftforge.client.event.RenderLevelLastEvent;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.event.TickEvent;
import thelm.packagedauto.api.DirectionalGlobalPos;
import thelm.packagedauto.item.DistributorMarkerItem;
import thelm.packagedauto.item.ProxyMarkerItem;

// Based on Botania, Scannables, and AE2
@SuppressWarnings("removal")
public class WorldOverlayRenderer {

	public static final WorldOverlayRenderer INSTANCE = new WorldOverlayRenderer();
	public static final Vec3 BLOCK_SIZE = new Vec3(1, 1, 1);

	private WorldOverlayRenderer() {}

	private Minecraft mc;
	private List<DirectionalMarkerInfo> directionalMarkers = new LinkedList<>();
	private List<SizedMarkerInfo> sizedMarkers = new LinkedList<>();
	private List<BeamInfo> beams = new LinkedList<>();

	public void onConstruct() {
		mc = Minecraft.m_91087_();
		MinecraftForge.EVENT_BUS.addListener(this::onClientTick);
		MinecraftForge.EVENT_BUS.addListener(this::onRenderLevelLast);
	}

	public void onClientTick(TickEvent.ClientTickEvent event) {
		if(event.phase != TickEvent.Phase.END || mc.f_91073_ == null || mc.f_91074_ == null || mc.m_91104_()) {
			return;
		}
		for(InteractionHand hand : InteractionHand.values()) {
			ItemStack stack = mc.f_91074_.m_21120_(hand);
			if(stack.m_150930_(DistributorMarkerItem.INSTANCE)) {
				DirectionalGlobalPos globalPos = DistributorMarkerItem.INSTANCE.getDirectionalGlobalPos(stack);
				if(globalPos != null) {
					addDirectionalMarkers(List.of(globalPos), 0x00FFFF, 1);
				}
			}
			if(stack.m_150930_(ProxyMarkerItem.INSTANCE)) {
				DirectionalGlobalPos globalPos = ProxyMarkerItem.INSTANCE.getDirectionalGlobalPos(stack);
				if(globalPos != null) {
					addDirectionalMarkers(List.of(globalPos), 0xFF7F00, 1);
				}
			}
		}
	}

	public void onRenderLevelLast(RenderLevelLastEvent event) {
		render(event.getPoseStack(), event.getPartialTick());
	}

	public void addDirectionalMarkers(List<DirectionalGlobalPos> positions, int color, int lifetime) {
		directionalMarkers.add(new DirectionalMarkerInfo(positions, color, lifetime));
	}

	public void addSizedMarker(Vec3 lowerCorner, Vec3 size, int color, int lifetime) {
		sizedMarkers.add(new SizedMarkerInfo(lowerCorner, size, color, lifetime));
	}

	public void addBeams(Vec3 source, List<Vec3> deltas, int color, int lifetime, boolean fadeout) {
		beams.add(new BeamInfo(source, deltas, color, lifetime, fadeout));
	}

	public void render(PoseStack poseStack, float partialTick) {
		int currentTick = RenderTimer.INSTANCE.getTicks();
		directionalMarkers.removeIf(marker->marker.shouldRemove(currentTick));
		sizedMarkers.removeIf(marker->marker.shouldRemove(currentTick));
		beams.removeIf(beam->beam.shouldRemove(currentTick));

		float renderTick = currentTick+partialTick;
		Vec3 cameraPos = mc.f_91063_.m_109153_().m_90583_();

		MultiBufferSource.BufferSource buffers = RenderTypeHelper.BUFFERS;
		VertexConsumer quadBuffer = buffers.m_6299_(RenderTypeHelper.MARKER_QUAD);
		VertexConsumer lineBuffer = buffers.m_6299_(RenderTypeHelper.MARKER_LINE_4);

		for(DirectionalMarkerInfo marker : directionalMarkers) {
			int r = marker.color>>16&0xFF;
			int g = marker.color>> 8&0xFF;
			int b = marker.color    &0xFF;

			for(DirectionalGlobalPos globalPos : marker.positions) {
				if(!globalPos.dimension().equals(mc.f_91073_.m_46472_())) {
					continue;
				}

				int range = 64;
				BlockPos blockPos = globalPos.blockPos();
				Vec3 distVec = cameraPos.m_82546_(Vec3.m_82512_(blockPos));
				if(Doubles.max(Math.abs(distVec.f_82479_), Math.abs(distVec.f_82480_), Math.abs(distVec.f_82481_)) > range) {
					continue;
				}

				poseStack.m_85836_();
				poseStack.m_85837_(blockPos.m_123341_()-cameraPos.f_82479_, blockPos.m_123342_()-cameraPos.f_82480_, blockPos.m_123343_()-cameraPos.f_82481_);

				Direction direction = globalPos.direction();
				addMarkerVertices(poseStack, quadBuffer, BLOCK_SIZE, direction, r, g, b, 127);
				addMarkerVertices(poseStack, lineBuffer, BLOCK_SIZE, null, r, g, b, 255);

				poseStack.m_85849_();
			}
		}

		RenderSystem.m_69465_();
		buffers.m_109911_();
		RenderSystem.m_69482_();

		lineBuffer = buffers.m_6299_(RenderTypeHelper.MARKER_LINE_4);

		for(SizedMarkerInfo marker : sizedMarkers) {
			Vec3 lowerCorner = marker.lowerCorner;

			poseStack.m_85836_();
			poseStack.m_85837_(lowerCorner.f_82479_-cameraPos.f_82479_, lowerCorner.f_82480_-cameraPos.f_82480_, lowerCorner.f_82481_-cameraPos.f_82481_);

			int r = marker.color>>16&0xFF;
			int g = marker.color>> 8&0xFF;
			int b = marker.color    &0xFF;
			addMarkerVertices(poseStack, lineBuffer, marker.size, null, r, g, b, 255);

			poseStack.m_85849_();
		}

		buffers.m_109911_();

		lineBuffer = buffers.m_6299_(RenderTypeHelper.BEAM_LINE_3);

		for(BeamInfo beam : beams) {
			Vec3 source = beam.source();

			poseStack.m_85836_();
			poseStack.m_85837_(source.f_82479_-cameraPos.f_82479_, source.f_82480_-cameraPos.f_82480_, source.f_82481_-cameraPos.f_82481_);

			int r = beam.color>>16&0xFF;
			int g = beam.color>> 8&0xFF;
			int b = beam.color    &0xFF;
			int a = (int)(beam.getAlpha(renderTick)*255);
			for(Vec3 delta : beam.deltas) {
				addBeamVertices(poseStack, lineBuffer, delta, r, g, b, a);
			}

			poseStack.m_85849_();
		}

		buffers.m_109911_();
	}

	public void addMarkerVertices(PoseStack poseStack, VertexConsumer buffer, Vec3 delta, Direction direction, int r, int g, int b, int a) {
		Matrix4f pose = poseStack.m_85850_().m_85861_();
		Matrix3f normal = poseStack.m_85850_().m_85864_();
		float x = (float)delta.f_82479_;
		float y = (float)delta.f_82480_;
		float z = (float)delta.f_82481_;
		if(direction == null || direction == Direction.NORTH) {
			// Face North, Edge Bottom
			buffer.m_85982_(pose, 0, 0, 0).m_6122_(r, g, b, a).m_85977_(normal, 1, 0, 0).m_5752_();
			buffer.m_85982_(pose, x, 0, 0).m_6122_(r, g, b, a).m_85977_(normal, 1, 0, 0).m_5752_();
			// Face North, Edge Top
			buffer.m_85982_(pose, x, y, 0).m_6122_(r, g, b, a).m_85977_(normal, -1, 0, 0).m_5752_();
			buffer.m_85982_(pose, 0, y, 0).m_6122_(r, g, b, a).m_85977_(normal, -1, 0, 0).m_5752_();
		}
		if(direction == null || direction == Direction.SOUTH) {
			// Face South, Edge Bottom
			buffer.m_85982_(pose, x, 0, z).m_6122_(r, g, b, a).m_85977_(normal, -1, 0, 0).m_5752_();
			buffer.m_85982_(pose, 0, 0, z).m_6122_(r, g, b, a).m_85977_(normal, -1, 0, 0).m_5752_();
			// Face South, Edge Top
			buffer.m_85982_(pose, 0, y, z).m_6122_(r, g, b, a).m_85977_(normal, 1, 0, 0).m_5752_();
			buffer.m_85982_(pose, x, y, z).m_6122_(r, g, b, a).m_85977_(normal, 1, 0, 0).m_5752_();
		}
		if(direction == null || direction == Direction.WEST) {
			// Face West, Edge Bottom
			buffer.m_85982_(pose, 0, 0, 0).m_6122_(r, g, b, a).m_85977_(normal, 0, 0, 1).m_5752_();
			buffer.m_85982_(pose, 0, 0, z).m_6122_(r, g, b, a).m_85977_(normal, 0, 0, 1).m_5752_();
			// Face West, Edge Top
			buffer.m_85982_(pose, 0, y, z).m_6122_(r, g, b, a).m_85977_(normal, 0, 0, -1).m_5752_();
			buffer.m_85982_(pose, 0, y, 0).m_6122_(r, g, b, a).m_85977_(normal, 0, 0, -1).m_5752_();
		}
		if(direction == null || direction == Direction.EAST) {
			// Face East, Edge Bottom
			buffer.m_85982_(pose, x, 0, z).m_6122_(r, g, b, a).m_85977_(normal, 0, 0, -1).m_5752_();
			buffer.m_85982_(pose, x, 0, 0).m_6122_(r, g, b, a).m_85977_(normal, 0, 0, -1).m_5752_();
			// Face East, Edge Top
			buffer.m_85982_(pose, x, y, 0).m_6122_(r, g, b, a).m_85977_(normal, 0, 0, 1).m_5752_();
			buffer.m_85982_(pose, x, y, z).m_6122_(r, g, b, a).m_85977_(normal, 0, 0, 1).m_5752_();
		}
		if(direction == Direction.DOWN) {
			// Face Down
			buffer.m_85982_(pose, 0, 0, 0).m_6122_(r, g, b, a).m_85977_(normal, 1, 0, 0).m_5752_();
			buffer.m_85982_(pose, x, 0, 0).m_6122_(r, g, b, a).m_85977_(normal, 1, 0, 0).m_5752_();
			buffer.m_85982_(pose, x, 0, z).m_6122_(r, g, b, a).m_85977_(normal, -1, 0, 0).m_5752_();
			buffer.m_85982_(pose, 0, 0, z).m_6122_(r, g, b, a).m_85977_(normal, -1, 0, 0).m_5752_();
		}
		if(direction == Direction.UP) {
			// Face Up
			buffer.m_85982_(pose, 0, y, 0).m_6122_(r, g, b, a).m_85977_(normal, 1, 0, 0).m_5752_();
			buffer.m_85982_(pose, x, y, 0).m_6122_(r, g, b, a).m_85977_(normal, 1, 0, 0).m_5752_();
			buffer.m_85982_(pose, x, y, z).m_6122_(r, g, b, a).m_85977_(normal, -1, 0, 0).m_5752_();
			buffer.m_85982_(pose, 0, y, z).m_6122_(r, g, b, a).m_85977_(normal, -1, 0, 0).m_5752_();
		}
		if(direction == null) {
			// Face North, Edge West
			buffer.m_85982_(pose, 0, 0, 0).m_6122_(r, g, b, a).m_85977_(normal, 0, 1, 0).m_5752_();
			buffer.m_85982_(pose, 0, y, 0).m_6122_(r, g, b, a).m_85977_(normal, 0, 1, 0).m_5752_();
			// Face North, Edge East
			buffer.m_85982_(pose, x, y, 0).m_6122_(r, g, b, a).m_85977_(normal, 0, -1, 0).m_5752_();
			buffer.m_85982_(pose, x, 0, 0).m_6122_(r, g, b, a).m_85977_(normal, 0, -1, 0).m_5752_();
			// Face South, Edge East
			buffer.m_85982_(pose, x, 0, z).m_6122_(r, g, b, a).m_85977_(normal, 0, 1, 0).m_5752_();
			buffer.m_85982_(pose, x, y, z).m_6122_(r, g, b, a).m_85977_(normal, 0, 1, 0).m_5752_();
			// Face South, Edge West
			buffer.m_85982_(pose, 0, y, z).m_6122_(r, g, b, a).m_85977_(normal, 0, -1, 0).m_5752_();
			buffer.m_85982_(pose, 0, 0, z).m_6122_(r, g, b, a).m_85977_(normal, 0, -1, 0).m_5752_();
		}
	}

	public void addBeamVertices(PoseStack poseStack, VertexConsumer buffer, Vec3 delta, int r, int g, int b, int a) {
		Vec3 normalVec = delta.m_82541_();
		Matrix4f pose = poseStack.m_85850_().m_85861_();
		Matrix3f normal = poseStack.m_85850_().m_85864_();
		float x = (float)delta.f_82479_;
		float y = (float)delta.f_82480_;
		float z = (float)delta.f_82481_;
		float xn = (float)normalVec.f_82479_;
		float yn = (float)normalVec.f_82480_;
		float zn = (float)normalVec.f_82481_;
		buffer.m_85982_(pose, 0, 0, 0).m_6122_(r, g, b, a).m_85977_(normal, xn, yn, zn).m_5752_();
		buffer.m_85982_(pose, x, y, z).m_6122_(r, g, b, a).m_85977_(normal, xn, yn, zn).m_5752_();
	}

	public static record DirectionalMarkerInfo(List<DirectionalGlobalPos> positions, int color, int lifetime, int startTick) {

		public DirectionalMarkerInfo(List<DirectionalGlobalPos> positions, int color, int lifetime) {
			this(positions, color, lifetime, RenderTimer.INSTANCE.getTicks());
		}

		public boolean shouldRemove(int currentTick) {
			if(currentTick < startTick) {
				currentTick += 0x1FFFFF;
			}
			return currentTick-startTick >= lifetime;
		}
	}

	public static record SizedMarkerInfo(Vec3 lowerCorner, Vec3 size, int color, int lifetime, int startTick) {

		public SizedMarkerInfo(Vec3 lowerCorner, Vec3 size, int color, int lifetime) {
			this(lowerCorner, size, color, lifetime, RenderTimer.INSTANCE.getTicks());
		}

		public boolean shouldRemove(int currentTick) {
			if(currentTick < startTick) {
				currentTick += 0x1FFFFF;
			}
			return currentTick-startTick >= lifetime;
		}
	}

	public static record BeamInfo(Vec3 source, List<Vec3> deltas, int color, int lifetime, boolean fadeout, int startTick) {

		public BeamInfo(Vec3 source, List<Vec3> deltas, int color, int lifetime, boolean fadeout) {
			this(source, deltas, color, lifetime, fadeout, RenderTimer.INSTANCE.getTicks());
		}

		public boolean shouldRemove(int currentTick) {
			if(currentTick < startTick) {
				currentTick += 0x1FFFFF;
			}
			return currentTick-startTick >= lifetime;
		}

		public float getAlpha(float renderTick) {
			if(!fadeout) {
				return 1;
			}
			float diff = renderTick-startTick;
			if(diff < 0) {
				diff += 0x1FFFFF;
			}
			float factor = Math.min(diff/lifetime, 1);
			return 1-factor*factor;
		}
	}

	public static class RenderTypeHelper extends RenderType {

		private RenderTypeHelper(String name, VertexFormat format, Mode mode, int bufferSize, boolean affectsCrumbling, boolean sortOnUpload, Runnable setupState, Runnable clearState) {
			super(name, format, mode, bufferSize, affectsCrumbling, sortOnUpload, setupState, clearState);
		}

		public static final RenderType MARKER_LINE_4;
		public static final RenderType MARKER_QUAD;
		public static final RenderType BEAM_LINE_3;
		public static final MultiBufferSource.BufferSource BUFFERS;

		static {
			MARKER_LINE_4 = m_173215_("packagedauto:marker_line_4",
					DefaultVertexFormat.f_166851_, VertexFormat.Mode.LINES, 8192, false, false,
					CompositeState.m_110628_().
					m_173292_(f_173095_).
					m_110673_(new LineStateShard(OptionalDouble.of(4))).
					m_110669_(f_110119_).
					m_110687_(f_110115_).
					m_110663_(f_110111_).
					m_110661_(f_110110_).
					m_110691_(false));
			MARKER_QUAD = m_173215_("packagedauto:marker_quad",
					DefaultVertexFormat.f_166851_, VertexFormat.Mode.QUADS, 1024, false, false,
					CompositeState.m_110628_().
					m_173292_(RenderStateShard.f_173104_).
					m_110669_(f_110119_).
					m_110685_(f_110139_).
					m_110687_(f_110115_).
					m_110663_(f_110111_).
					m_110661_(f_110110_).
					m_110691_(false));
			BEAM_LINE_3 = m_173215_("packagedauto:beam_line_3",
					DefaultVertexFormat.f_166851_, VertexFormat.Mode.LINES, 8192, false, false,
					CompositeState.m_110628_().
					m_173292_(f_173095_).
					m_110673_(new LineStateShard(OptionalDouble.of(3))).
					m_110669_(f_110119_).
					m_110685_(f_110139_).
					m_110675_(f_110129_).
					m_110687_(f_110114_).
					m_110661_(f_110110_).
					m_110691_(false));
			BUFFERS = MultiBufferSource.m_109900_(
					Map.of(MARKER_LINE_4, new BufferBuilder(MARKER_LINE_4.m_110507_()),
							MARKER_QUAD, new BufferBuilder(MARKER_QUAD.m_110507_()),
							BEAM_LINE_3, new BufferBuilder(BEAM_LINE_3.m_110507_())),
					Tesselator.m_85913_().m_85915_());
		}
	}
}
