package thelm.packagedauto.api;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.function.BooleanSupplier;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.apache.commons.lang3.tuple.Triple;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;

import io.netty.buffer.ByteBuf;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntRBTreeMap;
import it.unimi.dsi.fastutil.objects.ObjectRBTreeSet;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.NonNullList;
import net.minecraft.util.ResourceLocation;
import net.minecraftforge.fml.common.network.ByteBufUtils;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.ItemHandlerHelper;

public class MiscUtil {

	private static final Cache<NBTTagCompound, IRecipeInfo> RECIPE_CACHE = CacheBuilder.newBuilder().maximumSize(500).build();
	private static final Logger LOGGER = LogManager.getLogger();

	private MiscUtil() {}

	public static List<ItemStack> condenseStacks(IInventory inventory) {
		List<ItemStack> stacks = new ArrayList<>(inventory.func_70302_i_());
		for(int i = 0; i < inventory.func_70302_i_(); ++i) {
			stacks.add(inventory.func_70301_a(i));
		}
		return condenseStacks(stacks);
	}

	public static List<ItemStack> condenseStacks(IItemHandler itemHandler) {
		List<ItemStack> stacks = new ArrayList<>(itemHandler.getSlots());
		for(int i = 0; i < itemHandler.getSlots(); ++i) {
			stacks.add(itemHandler.getStackInSlot(i));
		}
		return condenseStacks(stacks);
	}

	public static List<ItemStack> condenseStacks(ItemStack... stacks) {
		return condenseStacks(Arrays.asList(stacks));
	}

	public static List<ItemStack> condenseStacks(Stream<ItemStack> stacks) {
		return condenseStacks(stacks.collect(Collectors.toList()));
	}

	public static List<ItemStack> condenseStacks(Iterable<ItemStack> stacks) {
		return condenseStacks(stacks instanceof List<?> ? (List<ItemStack>)stacks : Lists.newArrayList(stacks));
	}

	public static List<ItemStack> condenseStacks(List<ItemStack> stacks) {
		return condenseStacks(stacks, false);
	}

	public static List<ItemStack> condenseStacks(List<ItemStack> stacks, boolean ignoreStackSize) {
		Object2IntRBTreeMap<Triple<Item, Integer, NBTTagCompound>> map = new Object2IntRBTreeMap<>(
				Comparator.comparing(triple->Triple.of(triple.getLeft().getRegistryName(), triple.getMiddle(), ""+triple.getRight())));
		for(ItemStack stack : stacks) {
			if(stack.func_190926_b()) {
				continue;
			}
			Triple<Item, Integer, NBTTagCompound> triple = Triple.of(stack.func_77973_b(), stack.func_77960_j(), stack.func_77978_p());
			if(!map.containsKey(triple)) {
				map.put(triple, 0);
			}
			map.addTo(triple, stack.func_190916_E());
		}
		List<ItemStack> list = new ArrayList<>();
		for(Object2IntMap.Entry<Triple<Item, Integer, NBTTagCompound>> entry : map.object2IntEntrySet()) {
			Triple<Item, Integer, NBTTagCompound> triple = entry.getKey();
			int count = entry.getIntValue();
			Item item = triple.getLeft();
			int meta = triple.getMiddle();
			NBTTagCompound nbt = triple.getRight();
			if(ignoreStackSize) {
				ItemStack toAdd = new ItemStack(item, count, meta);
				toAdd.func_77982_d(nbt);
				list.add(toAdd);
			}
			else {
				while(count > 0) {
					ItemStack toAdd = new ItemStack(item, 1, meta);
					toAdd.func_77982_d(nbt);
					int limit = item.getItemStackLimit(toAdd);
					toAdd.func_190920_e(Math.min(count, limit));
					list.add(toAdd);
					count -= limit;
				}
			}
		}
		map.clear();
		return list;
	}

	public static NBTTagList saveAllItems(NBTTagList tagList, List<ItemStack> list) {
		return saveAllItems(tagList, list, "Index");
	}

	public static NBTTagList saveAllItems(NBTTagList tagList, List<ItemStack> list, String indexKey) {
		for(int i = 0; i < list.size(); ++i) {
			ItemStack stack = list.get(i);
			boolean empty = stack.func_190926_b();
			if(!empty || i == list.size()-1) {
				if(empty) {
					// Ensure that the end-of-list stack if empty is always the default empty stack
					stack = new ItemStack((Item)null);
				}
				NBTTagCompound nbt = new NBTTagCompound();
				nbt.func_74774_a(indexKey, (byte)i);
				saveItemWithLargeCount(nbt, stack);
				tagList.func_74742_a(nbt);
			}
		}
		return tagList;
	}

	public static void loadAllItems(NBTTagList tagList, List<ItemStack> list) {
		loadAllItems(tagList, list, "Index");
	}

	public static void loadAllItems(NBTTagList tagList, List<ItemStack> list, String indexKey) {
		list.clear();
		try {
			for(int i = 0; i < tagList.func_74745_c(); ++i) {
				NBTTagCompound nbt = tagList.func_150305_b(i);
				int j = nbt.func_74771_c(indexKey) & 255;
				while(j >= list.size()) {
					list.add(ItemStack.field_190927_a);
				}
				if(j >= 0)  {
					ItemStack stack = loadItemWithLargeCount(nbt);
					list.set(j, stack.func_190926_b() ? ItemStack.field_190927_a : stack);
				}
			}
		}
		catch(UnsupportedOperationException | IndexOutOfBoundsException e) {}
	}

	public static NBTTagCompound saveItemWithLargeCount(NBTTagCompound nbt, ItemStack stack) {
		stack.func_77955_b(nbt);
		int count = stack.func_190916_E();
		if((byte)count == count) {
			nbt.func_74774_a("Count", (byte)count);
		}
		else if((short)count == count) {
			nbt.func_74777_a("Count", (short)count);
		}
		else {
			nbt.func_74768_a("Count", (short)count);
		}
		return nbt;
	}

	public static ItemStack loadItemWithLargeCount(NBTTagCompound nbt) {
		ItemStack stack = new ItemStack(nbt);
		stack.func_190920_e(nbt.func_74762_e("Count"));
		return stack;
	}

	public static void writeItemWithLargeCount(ByteBuf buf, ItemStack stack) {
		if(stack.func_190926_b()) {
			buf.writeBoolean(false);
			return;
		}
		buf.writeBoolean(true);
		ByteBufUtils.writeVarInt(buf, Item.func_150891_b(stack.func_77973_b()), 5);
		ByteBufUtils.writeVarInt(buf, stack.func_190916_E(), 5);
		buf.writeShort(stack.func_77960_j());
		NBTTagCompound nbt = null;
		if(stack.func_77973_b().func_77645_m() || stack.func_77973_b().func_77651_p()) {
			nbt = stack.func_77973_b().getNBTShareTag(stack);
		}
		ByteBufUtils.writeTag(buf, nbt);
	}

	public static ItemStack readItemWithLargeCount(ByteBuf buf) {
		if(!buf.readBoolean()) {
			return ItemStack.field_190927_a;
		}
		int id = ByteBufUtils.readVarInt(buf, 5);
		int count = ByteBufUtils.readVarInt(buf, 5);
		int meta = buf.readShort();
		ItemStack stack = new ItemStack(Item.func_150899_d(id), count, meta);
		stack.func_77973_b().readNBTShareTag(stack, ByteBufUtils.readTag(buf));
		return stack;
	}

	public static IPackagePattern getPatternHelper(IRecipeInfo recipeInfo, int index) {
		try {
			Class<? extends IPackagePattern> helperClass = (Class<? extends IPackagePattern>)Class.forName("thelm.packagedauto.util.PatternHelper");
			Constructor<? extends IPackagePattern> helperConstructor = helperClass.getConstructor(IRecipeInfo.class, int.class);
			return helperConstructor.newInstance(recipeInfo, index);
		}
		catch(Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	public static List<ItemStack> getRemainingItems(IInventory inventory) {
		return getRemainingItems(IntStream.range(0, inventory.func_70302_i_()).mapToObj(inventory::func_70301_a).collect(Collectors.toList()));
	}

	public static List<ItemStack> getRemainingItems(IInventory inventory, int minInclusive, int maxExclusive) {
		return getRemainingItems(IntStream.range(minInclusive, maxExclusive).mapToObj(inventory::func_70301_a).collect(Collectors.toList()));
	}

	public static List<ItemStack> getRemainingItems(ItemStack... stacks) {
		return getRemainingItems(Arrays.asList(stacks));
	}

	public static List<ItemStack> getRemainingItems(List<ItemStack> stacks) {
		NonNullList<ItemStack> ret = NonNullList.func_191197_a(stacks.size(), ItemStack.field_190927_a);
		for(int i = 0; i < ret.size(); i++) {
			ret.set(i, getContainerItem(stacks.get(i)));
		}
		return ret;
	}

	public static ItemStack getContainerItem(ItemStack stack) {
		if(stack.func_190926_b()) {
			return ItemStack.field_190927_a;
		}
		if(stack.func_77973_b().hasContainerItem(stack)) {
			stack = stack.func_77973_b().getContainerItem(stack);
			if(!stack.func_190926_b() && stack.func_77984_f() && stack.func_77960_j() > stack.func_77958_k()) {
				return ItemStack.field_190927_a;
			}
			return stack;
		}
		else {
			if(stack.func_190916_E() > 1) {
				stack = stack.func_77946_l();
				stack.func_190920_e(stack.func_190916_E() - 1);
				return stack;
			}
			return ItemStack.field_190927_a;
		}
	}

	public static ItemStack cloneStack(ItemStack stack, int stackSize) {
		if(stack.func_190926_b()) {
			return ItemStack.field_190927_a;
		}
		ItemStack retStack = stack.func_77946_l();
		retStack.func_190920_e(stackSize);
		return retStack;
	}

	public static boolean isEmpty(IItemHandler itemHandler) {
		for(int i = 0; i < itemHandler.getSlots(); ++i) {
			if(!itemHandler.getStackInSlot(i).func_190926_b()) {
				return false;
			}
		}
		return true;
	}

	public static NBTTagCompound writeRecipeToNBT(NBTTagCompound nbt, IRecipeInfo recipe) {
		nbt.func_74778_a("RecipeType", recipe.getRecipeType().getName().toString());
		recipe.writeToNBT(nbt);
		return nbt;
	}

	public static IRecipeInfo readRecipeFromNBT(NBTTagCompound nbt) {
		IRecipeType recipeType = RecipeTypeRegistry.getRecipeType(new ResourceLocation(nbt.func_74779_i("RecipeType")));
		if(recipeType != null) {
			IRecipeInfo recipe = RECIPE_CACHE.getIfPresent(nbt);
			if(recipe != null) {
				return recipe;
			}
			recipe = recipeType.getNewRecipeInfo();
			recipe.readFromNBT(nbt);
			RECIPE_CACHE.put(nbt, recipe);
			return recipe;
		}
		return null;
	}

	public static NBTTagList writeRecipeListToNBT(NBTTagList tagList, List<IRecipeInfo> recipes) {
		for(IRecipeInfo recipe : recipes) {
			tagList.func_74742_a(writeRecipeToNBT(new NBTTagCompound(), recipe));
		}
		return tagList;
	}

	public static List<IRecipeInfo> readRecipeListFromNBT(NBTTagList tagList) {
		List<IRecipeInfo> recipes = new ArrayList<>(tagList.func_74745_c());
		for(int i = 0; i < tagList.func_74745_c(); ++i) {
			IRecipeInfo recipe = readRecipeFromNBT(tagList.func_150305_b(i));
			if(recipe != null) {
				recipes.add(recipe);
			}
		}
		return recipes;
	}

	public static boolean recipeEquals(IRecipeInfo recipeA, Object recipeInternalA, IRecipeInfo recipeB, Object recipeInternalB) {
		if(!Objects.equals(recipeInternalA, recipeInternalB)) {
			return false;
		}
		List<ItemStack> inputsA = recipeA.getInputs();
		List<ItemStack> inputsB = recipeB.getInputs();
		if(inputsA.size() != inputsB.size()) {
			return false;
		}
		List<ItemStack> outputsA = recipeA.getOutputs();
		List<ItemStack> outputsB = recipeB.getOutputs();
		if(outputsA.size() != outputsB.size()) {
			return false;
		}
		for(int i = 0; i < inputsA.size(); ++i) {
			if(!ItemStack.areItemStacksEqualUsingNBTShareTag(inputsA.get(i), inputsB.get(i))) {
				return false;
			}
		}
		for(int i = 0; i < outputsA.size(); ++i) {
			if(!ItemStack.areItemStacksEqualUsingNBTShareTag(outputsA.get(i), outputsB.get(i))) {
				return false;
			}
		}
		return true;
	}

	public static int recipeHashCode(IRecipeInfo recipe, Object recipeInternal) {
		List<ItemStack> inputs = recipe.getInputs();
		List<ItemStack> outputs = recipe.getOutputs();
		Function<ItemStack, Object[]> decompose = stack->new Object[] {
				stack.func_77973_b(), stack.func_77952_i(), stack.func_190916_E(), stack.func_77978_p(),
		};
		Object[] toHash = {
				recipeInternal, inputs.stream().map(decompose).toArray(), outputs.stream().map(decompose).toArray(),
		};
		return Arrays.deepHashCode(toHash);
	}

	//Modified from Forestry
	public static boolean removeExactSet(List<ItemStack> offered, List<ItemStack> required, boolean simulate) {
		List<ItemStack> condensedRequired = condenseStacks(required, true);
		List<ItemStack> condensedOffered = condenseStacks(offered, true);
		f:for(ItemStack req : condensedRequired) {
			for(ItemStack offer : condensedOffered) {
				if(req.func_190916_E() <= offer.func_190916_E() && req.func_77969_a(offer) &&
						(!req.func_77942_o() || ItemStack.areItemStackShareTagsEqual(req, offer))) {
					continue f;
				}
			}
			return false;
		}
		if(simulate) {
			return true;
		}
		for(ItemStack req : condensedRequired) {
			int count = req.func_190916_E();
			for(ItemStack offer : offered) {
				if(!offer.func_190926_b()) {
					if(req.func_77969_a(offer) && (!req.func_77942_o() || ItemStack.areItemStackShareTagsEqual(req, offer))) {
						int toRemove = Math.min(count, offer.func_190916_E());
						offer.func_190918_g(toRemove);
						count -= toRemove;
						if(count == 0) {
							continue;
						}
					}
				}
			}
		}
		return true;
	}

	public static boolean arePatternsDisjoint(List<IPackagePattern> patternList) {
		ObjectRBTreeSet<Triple<Item, Integer, NBTTagCompound>> set = new ObjectRBTreeSet<>(
				Comparator.comparing(triple->Triple.of(triple.getLeft().getRegistryName(), triple.getMiddle(), ""+triple.getRight())));
		for(IPackagePattern pattern : patternList) {
			List<ItemStack> condensedInputs = condenseStacks(pattern.getInputs(), true);
			for(ItemStack stack : condensedInputs) {
				Triple<Item, Integer, NBTTagCompound> toAdd = Triple.of(stack.func_77973_b(), stack.func_77952_i(), stack.func_77978_p());
				if(set.contains(toAdd)) {
					return false;
				}
				set.add(toAdd);
			}
		}
		set.clear();
		return true;
	}

	public static ItemStack insertItem(IItemHandler itemHandler, ItemStack stack, boolean requireEmptySlot, boolean simulate) {
		if(itemHandler == null || stack.func_190926_b()) {
			return stack;
		}
		if(!requireEmptySlot) {
			return ItemHandlerHelper.insertItem(itemHandler, stack, simulate);
		}
		for(int slot = 0; slot < itemHandler.getSlots(); ++slot) {
			if(itemHandler.getStackInSlot(slot).func_190926_b()) {
				stack = itemHandler.insertItem(slot, stack, simulate);
				if(stack.func_190926_b()) {
					return ItemStack.field_190927_a;
				}
			}
		}
		return stack;
	}

	public static Runnable conditionalRunnable(BooleanSupplier conditionSupplier, Supplier<Runnable> trueRunnable, Supplier<Runnable> falseRunnable) {
		return ()->(conditionSupplier.getAsBoolean() ? trueRunnable : falseRunnable).get().run();
	}

	public static <T> Supplier<T> conditionalSupplier(BooleanSupplier conditionSupplier, Supplier<Supplier<T>> trueSupplier, Supplier<Supplier<T>> falseSupplier) {
		return ()->(conditionSupplier.getAsBoolean() ? trueSupplier : falseSupplier).get().get();
	}
}
