package thelm.packagedauto.tile;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import appeng.api.networking.IGridHost;
import appeng.api.networking.IGridNode;
import appeng.api.networking.security.IActionHost;
import appeng.api.util.AECableType;
import appeng.api.util.AEPartLocation;
import it.unimi.dsi.fastutil.ints.Int2ObjectArrayMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.entity.player.InventoryPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ITickable;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3d;
import net.minecraft.util.text.translation.I18n;
import net.minecraftforge.fml.common.Loader;
import net.minecraftforge.fml.common.Optional;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.ItemHandlerHelper;
import thelm.packagedauto.api.DirectionalGlobalPos;
import thelm.packagedauto.api.IPackageCraftingMachine;
import thelm.packagedauto.api.IRecipeInfo;
import thelm.packagedauto.api.ISettingsCloneable;
import thelm.packagedauto.api.MiscUtil;
import thelm.packagedauto.client.gui.GuiDistributor;
import thelm.packagedauto.container.ContainerDistributor;
import thelm.packagedauto.integration.appeng.networking.HostHelperTileDistributor;
import thelm.packagedauto.inventory.InventoryDistributor;
import thelm.packagedauto.item.ItemDistributorMarker;
import thelm.packagedauto.network.packet.PacketBeam;
import thelm.packagedauto.network.packet.PacketDirectionalMarker;
import thelm.packagedauto.network.packet.PacketSizedMarker;
import thelm.packagedauto.recipe.IRecipeInfoProcessingPositioned;

@Optional.InterfaceList({
	@Optional.Interface(iface="appeng.api.networking.IGridHost", modid="appliedenergistics2"),
	@Optional.Interface(iface="appeng.api.networking.security.IActionHost", modid="appliedenergistics2"),
})
public class TileDistributor extends TileBase implements ITickable, IPackageCraftingMachine, ISettingsCloneable, IGridHost, IActionHost {

	public static int range = 16;
	public static int refreshInterval = 4;

	public final Int2ObjectMap<DirectionalGlobalPos> positions = new Int2ObjectArrayMap<>(81);
	public final Int2ObjectMap<ItemStack> pending = new Int2ObjectArrayMap<>(81);
	public final Cache<UUID, Long> previewTimes = CacheBuilder.newBuilder().initialCapacity(2).expireAfterWrite(60, TimeUnit.SECONDS).build();

	public boolean firstTick = true;

	public TileDistributor() {
		setInventory(new InventoryDistributor(this));
		if(Loader.isModLoaded("appliedenergistics2")) {
			hostHelper = new HostHelperTileDistributor(this);
		}
	}

	@Override
	protected String getLocalizedName() {
		return I18n.func_74838_a("tile.packagedauto.distributor.name");
	}

	@Override
	public String getConfigTypeName() {
		return "tile.packagedauto.distributor.name";
	}

	@Override
	public void func_73660_a() {
		if(firstTick) {
			firstTick = false;
			if(!field_145850_b.field_72995_K && hostHelper != null) {
				hostHelper.isActive();
			}
		}
		if(!field_145850_b.field_72995_K) {
			if(field_145850_b.func_82737_E() % refreshInterval == 0 && !pending.isEmpty()) {
				distributeItems();
			}
		}
	}

	@Override
	public boolean acceptPackage(IRecipeInfo recipeInfo, List<ItemStack> stacks, EnumFacing facing) {
		if(!isBusy() && recipeInfo instanceof IRecipeInfoProcessingPositioned) {
			IRecipeInfoProcessingPositioned recipe = (IRecipeInfoProcessingPositioned)recipeInfo;
			boolean blocking = false;
			TileEntity unpackager = field_145850_b.func_175625_s(field_174879_c.func_177972_a(facing));
			if(unpackager instanceof TileUnpackager) {
				blocking = ((TileUnpackager)unpackager).blocking;
			}
			Int2ObjectMap<ItemStack> matrix = recipe.getMatrix();
			if(!positions.keySet().containsAll(matrix.keySet())) {
				return false;
			}
			for(Int2ObjectMap.Entry<ItemStack> entry : matrix.int2ObjectEntrySet()) {
				BlockPos pos = positions.get(entry.getIntKey()).blockPos();
				if(!field_145850_b.func_175667_e(pos)) {
					return false;
				}
				TileEntity tile = field_145850_b.func_175625_s(pos);
				if(tile == null) {
					return false;
				}
				ItemStack stack = entry.getValue().func_77946_l();
				EnumFacing dir = positions.get(entry.getIntKey()).direction();
				IItemHandler itemHandler = tile.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, dir);
				if(itemHandler != null) {
					if(blocking && !MiscUtil.isEmpty(itemHandler)) {
						return false;
					}
					if(!ItemHandlerHelper.insertItem(itemHandler, stack, true).func_190926_b()) {
						return false;
					}
				}
				else {
					return false;
				}
			}
			for(Int2ObjectMap.Entry<ItemStack> entry : matrix.int2ObjectEntrySet()) {
				pending.put(entry.getIntKey(), entry.getValue().func_77946_l());
			}
			distributeItems();
			return true;
		}
		return false;
	}

	@Override
	public boolean isBusy() {
		return !pending.isEmpty();
	}

	protected void distributeItems() {
		List<Vec3d> deltas = new ArrayList<>();
		for(int i : pending.keySet().toIntArray()) {
			if(!positions.containsKey(i)) {
				ejectItems();
				break;
			}
			BlockPos pos = positions.get(i).blockPos();
			if(!field_145850_b.func_175667_e(pos)) {
				continue;
			}
			TileEntity tile = field_145850_b.func_175625_s(pos);
			if(tile == null) {
				ejectItems();
				break;
			}
			ItemStack stack = pending.get(i);
			EnumFacing dir = positions.get(i).direction();
			IItemHandler itemHandler = tile.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, dir);
			ItemStack stackRem = stack;
			if(itemHandler != null) {
				stackRem = ItemHandlerHelper.insertItem(itemHandler, stack, false);
			}
			else {
				ejectItems();
				break;
			}
			if(stackRem.func_190916_E() < stack.func_190916_E()) {
				Vec3d delta = new Vec3d(pos.func_177973_b(this.field_174879_c)).func_178787_e(new Vec3d(dir.func_176730_m()).func_186678_a(0.5));
				deltas.add(delta);
			}
			if(stackRem.func_190926_b()) {
				pending.remove(i);
			}
			else {
				pending.put(i, stackRem);
			}
		}
		if(!deltas.isEmpty()) {
			Vec3d source = new Vec3d(field_174879_c).func_72441_c(0.5, 0.5, 0.5);
			PacketBeam.sendBeams(source, deltas, 0x00FFFF, 6, true, field_145850_b.field_73011_w.getDimension(), 32);
			func_70296_d();
		}
	}

	protected void ejectItems() {
		for(int i = 0; i < 81; ++i) {
			if(pending.containsKey(i)) {
				ItemStack stack = pending.remove(i);
				if(!stack.func_190926_b()) {
					double dx = field_145850_b.field_73012_v.nextFloat()/2+0.25;
					double dy = field_145850_b.field_73012_v.nextFloat()/2+0.75;
					double dz = field_145850_b.field_73012_v.nextFloat()/2+0.25;
					EntityItem entityitem = new EntityItem(field_145850_b, field_174879_c.func_177958_n()+dx, field_174879_c.func_177956_o()+dy, field_174879_c.func_177952_p()+dz, stack);
					entityitem.func_174869_p();
					field_145850_b.func_72838_d(entityitem);
				}
			}
		}
		func_70296_d();
	}

	public void sendPreview(EntityPlayerMP player) {
		long currentTime = field_145850_b.func_82737_E();
		Long cachedTime = previewTimes.getIfPresent(player.func_110124_au());
		if(cachedTime == null || currentTime-cachedTime > 180) {
			previewTimes.put(player.func_110124_au(), currentTime);
			if(!positions.isEmpty()) {
				List<Vec3d> deltas = positions.values().stream().map(globalPos->{
					BlockPos pos = globalPos.blockPos();
					EnumFacing dir = globalPos.direction();
					return new Vec3d(pos.func_177973_b(this.field_174879_c)).func_72441_c(dir.func_82601_c()*0.5, dir.func_96559_d()*0.5, dir.func_82599_e()*0.5);
				}).collect(Collectors.toList());
				Vec3d source = new Vec3d(field_174879_c).func_72441_c(0.5, 0.5, 0.5);
				PacketDirectionalMarker.sendDirectionalMarkers(player, new ArrayList<>(positions.values()), 0x00FF7F, 200);
				PacketBeam.sendBeams(player, source, deltas, 0x00FF7F, 200, false);
			}
			Vec3d lowerCorner = new Vec3d(field_174879_c).func_178786_a(range, range, range);
			Vec3d size = new Vec3d(range*2+1, range*2+1, range*2+1);
			PacketSizedMarker.sendSizedMarker(player, lowerCorner, size, 0x00FFFF, 200);
		}
	}

	@Override
	public int getComparatorSignal() {
		if(!pending.isEmpty()) {
			return 15;
		}
		return 0;
	}

	public HostHelperTileDistributor hostHelper;

	@Override
	public void func_145843_s() {
		super.func_145843_s();
		if(hostHelper != null) {
			hostHelper.invalidate();
		}
	}

	@Override
	public void onChunkUnload() {
		super.onChunkUnload();
		if(hostHelper != null) {
			hostHelper.invalidate();
		}
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public IGridNode getGridNode(AEPartLocation dir) {
		return getActionableNode();
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public AECableType getCableConnectionType(AEPartLocation dir) {
		return AECableType.SMART;
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public void securityBreak() {
		field_145850_b.func_175655_b(field_174879_c, true);
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public IGridNode getActionableNode() {
		return hostHelper.getNode();
	}

	@Override
	public boolean loadConfig(NBTTagCompound nbt, EntityPlayer player) {
		NBTTagList positionsTag = nbt.func_150295_c("Positions", 10);
		if(positionsTag.func_82582_d()) {
			return false;
		}
		int requiredCount = positionsTag.func_74745_c();
		int availableCount = 0;
		InventoryPlayer playerInventory = player.field_71071_by;
		for(int i = 0; i < inventory.func_70302_i_(); ++i) {
			ItemStack stack = inventory.func_70301_a(i);
			if(!stack.func_190926_b()) {
				if(stack.func_77973_b() == ItemDistributorMarker.INSTANCE) {
					availableCount += stack.func_190916_E();
				}
				else {
					return false;
				}
			}
		}
		if(availableCount < requiredCount) {
			for(int i = 0; i < playerInventory.func_70302_i_(); ++i) {
				ItemStack stack = playerInventory.func_70301_a(i);
				if(!stack.func_190926_b() && stack.func_77973_b() == ItemDistributorMarker.INSTANCE && !stack.func_77942_o()) {
					availableCount += stack.func_190916_E();
				}
				if(availableCount >= requiredCount) {
					break;
				}
			}
		}
		if(availableCount < requiredCount) {
			return false;
		}
		int removedCount = 0;
		for(int i = 0; i < inventory.func_70302_i_(); ++i) {
			removedCount += inventory.func_70301_a(i).func_190916_E();
			inventory.func_70299_a(i, ItemStack.field_190927_a);
		}
		if(removedCount < requiredCount) {
			for(int i = 0; i < playerInventory.func_70302_i_(); ++i) {
				ItemStack stack = playerInventory.func_70301_a(i);
				if(!stack.func_190926_b() && stack.func_77973_b() == ItemDistributorMarker.INSTANCE && !stack.func_77942_o()) {
					removedCount += stack.func_77979_a(requiredCount - removedCount).func_190916_E();
				}
				if(removedCount >= requiredCount) {
					break;
				}
			}
		}
		if(removedCount > requiredCount) {
			ItemStack stack = new ItemStack(ItemDistributorMarker.INSTANCE, removedCount-requiredCount);
			if(!playerInventory.func_70441_a(stack)) {
				EntityItem item = new EntityItem(field_145850_b, player.field_70165_t, player.field_70163_u, player.field_70161_v, stack);
				item.func_145799_b(player.func_70005_c_());
				field_145850_b.func_72838_d(item);
			}
		}
		for(int i = 0; i < requiredCount; ++i) {
			NBTTagCompound positionTag = positionsTag.func_150305_b(i);
			int index = positionTag.func_74771_c("Index");
			int dimension = positionTag.func_74762_e("Dimension");
			int[] posArray = positionTag.func_74759_k("Position");
			BlockPos blockPos = new BlockPos(posArray[0], posArray[1], posArray[2]);
			EnumFacing direction = EnumFacing.func_82600_a(positionTag.func_74771_c("Direction"));
			DirectionalGlobalPos globalPos = new DirectionalGlobalPos(dimension, blockPos, direction);
			ItemStack stack = new ItemStack(ItemDistributorMarker.INSTANCE);
			ItemDistributorMarker.INSTANCE.setDirectionalGlobalPos(stack, globalPos);
			inventory.func_70299_a(index, stack);
		}
		return true;
	}

	@Override
	public boolean saveConfig(NBTTagCompound nbt, EntityPlayer player) {
		if(positions.isEmpty()) {
			return false;
		}
		NBTTagList positionsTag = new NBTTagList();
		for(Int2ObjectMap.Entry<DirectionalGlobalPos> entry : positions.int2ObjectEntrySet()) {
			DirectionalGlobalPos pos = entry.getValue();
			NBTTagCompound positionTag = new NBTTagCompound();
			positionTag.func_74774_a("Index", (byte)entry.getIntKey());
			positionTag.func_74768_a("Dimension", pos.dimension());
			positionTag.func_74783_a("Position", new int[] {pos.x(), pos.y(), pos.z()});
			positionTag.func_74774_a("Direction", (byte)pos.direction().func_176745_a());
			positionsTag.func_74742_a(positionTag);
		}
		nbt.func_74782_a("Positions", positionsTag);
		return true;
	}

	@Override
	public void func_145839_a(NBTTagCompound nbt) {
		if(hostHelper != null) {
			hostHelper.readFromNBT(nbt);
		}
		super.func_145839_a(nbt);
		pending.clear();
		List<ItemStack> pendingList = new ArrayList<>();
		MiscUtil.loadAllItems(nbt.func_150295_c("Pending", 10), pendingList);
		for(int i = 0; i < 81 && i < pendingList.size(); ++i) {
			ItemStack stack = pendingList.get(i);
			if(!stack.func_190926_b()) {
				pending.put(i, stack);
			}
		}
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbt) {
		super.func_189515_b(nbt);
		List<ItemStack> pendingList = new ArrayList<>();
		for(int i = 0; i < 81; ++i) {
			pendingList.add(pending.getOrDefault(i, ItemStack.field_190927_a));
		}
		NBTTagList pendingTag = MiscUtil.saveAllItems(new NBTTagList(), pendingList);
		nbt.func_74782_a("Pending", pendingTag);
		if(hostHelper != null) {
			hostHelper.writeToNBT(nbt);
		}
		return nbt;
	}

	@SideOnly(Side.CLIENT)
	@Override
	public GuiContainer getClientGuiElement(EntityPlayer player, Object... args) {
		return new GuiDistributor(new ContainerDistributor(player.field_71071_by, this));
	}

	@Override
	public Container getServerGuiElement(EntityPlayer player, Object... args) {
		return new ContainerDistributor(player.field_71071_by, this);
	}
}
