package thelm.packagedauto.tile;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;

import appeng.api.networking.IGridHost;
import appeng.api.networking.IGridNode;
import appeng.api.networking.crafting.ICraftingPatternDetails;
import appeng.api.networking.crafting.ICraftingProvider;
import appeng.api.networking.crafting.ICraftingProviderHelper;
import appeng.api.networking.events.MENetworkChannelsChanged;
import appeng.api.networking.events.MENetworkEventSubscribe;
import appeng.api.networking.events.MENetworkPowerStatusChange;
import appeng.api.networking.security.IActionHost;
import appeng.api.util.AECableType;
import appeng.api.util.AEPartLocation;
import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.InventoryPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.InventoryCrafting;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.Ingredient;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ITickable;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.text.translation.I18n;
import net.minecraftforge.common.crafting.IngredientNBT;
import net.minecraftforge.common.util.RecipeMatcher;
import net.minecraftforge.energy.CapabilityEnergy;
import net.minecraftforge.fml.common.Loader;
import net.minecraftforge.fml.common.Optional;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.ItemHandlerHelper;
import thelm.packagedauto.api.IPackageItem;
import thelm.packagedauto.api.IPackagePattern;
import thelm.packagedauto.api.IRecipeInfo;
import thelm.packagedauto.api.IRecipeList;
import thelm.packagedauto.api.IRecipeListItem;
import thelm.packagedauto.api.ISettingsCloneable;
import thelm.packagedauto.api.MiscUtil;
import thelm.packagedauto.client.gui.GuiPackager;
import thelm.packagedauto.container.ContainerPackager;
import thelm.packagedauto.energy.EnergyStorage;
import thelm.packagedauto.integration.appeng.networking.HostHelperTilePackager;
import thelm.packagedauto.integration.appeng.recipe.PackageCraftingPatternHelper;
import thelm.packagedauto.inventory.InventoryPackager;
import thelm.packagedauto.item.ItemRecipeHolder;

@Optional.InterfaceList({
	@Optional.Interface(iface="appeng.api.networking.IGridHost", modid="appliedenergistics2"),
	@Optional.Interface(iface="appeng.api.networking.security.IActionHost", modid="appliedenergistics2"),
	@Optional.Interface(iface="appeng.api.networking.crafting.ICraftingProvider", modid="appliedenergistics2")
})
public class TilePackager extends TileBase implements ITickable, ISettingsCloneable, IGridHost, IActionHost, ICraftingProvider {

	public static int energyCapacity = 5000;
	public static int energyReq = 500;
	public static int energyUsage = 100;
	public static int refreshInterval = 4;
	public static boolean drawMEEnergy = true;

	public boolean isWorking = false;
	public int remainingProgress = 0;
	public List<IPackagePattern> patternList = new ArrayList<>();
	public IPackagePattern currentPattern;
	public boolean lockPattern = false;
	public Mode mode = Mode.EXACT;
	public boolean disjoint = false;
	public boolean powered = false;

	public TilePackager() {
		setInventory(new InventoryPackager(this));
		setEnergyStorage(new EnergyStorage(this, energyCapacity));
		if(Loader.isModLoaded("appliedenergistics2")) {
			hostHelper = new HostHelperTilePackager(this);
		}
	}

	@Override
	protected String getLocalizedName() {
		return I18n.func_74838_a("tile.packagedauto.packager.name");
	}

	@Override
	public String getConfigTypeName() {
		return "tile.packagedauto.packager.name";
	}

	@Override
	public void onLoad() {
		updatePowered();
	}

	@Override
	public void func_73660_a() {
		if(!field_145850_b.field_72995_K) {
			if(isWorking) {
				tickProcess();
				if(remainingProgress <= 0 && isInputValid()) {
					finishProcess();
					if(hostHelper != null && hostHelper.isActive() && !inventory.func_70301_a(9).func_190926_b()) {
						hostHelper.ejectItem();
					}
					else if(!inventory.func_70301_a(9).func_190926_b()) {
						ejectItem();
					}
					if(!canStart()) {
						endProcess();
					}
					else {
						startProcess();
					}
				}
			}
			else if(field_145850_b.func_82737_E() % refreshInterval == 0) {
				if(canStart()) {
					startProcess();
					tickProcess();
					isWorking = true;
				}
			}
			chargeEnergy();
			if(field_145850_b.func_82737_E() % refreshInterval == 0) {
				if(hostHelper != null && hostHelper.isActive()) {
					if(!inventory.func_70301_a(9).func_190926_b()) {
						hostHelper.ejectItem();
					}
					if(drawMEEnergy) {
						hostHelper.chargeEnergy();
					}
				}
				else if(!inventory.func_70301_a(9).func_190926_b()) {
					ejectItem();
				}
			}
		}
	}

	protected static Ingredient getIngredient(ItemStack stack) {
		return stack.func_77942_o() ? new IngredientNBT(stack) {} : Ingredient.func_193369_a(stack);
	}

	public boolean isInputValid() {
		if(currentPattern == null) {
			getPattern();
		}
		if(currentPattern == null) {
			return false;
		}
		List<ItemStack> input = inventory.stacks.subList(0, 9).stream().filter(stack->!stack.func_190926_b()).collect(Collectors.toList());
		if(input.isEmpty()) {
			return false;
		}
		if(!lockPattern && disjoint) {
			return MiscUtil.removeExactSet(input, currentPattern.getInputs(), true);
		}
		List<Ingredient> matchers = Lists.transform(currentPattern.getInputs(), TilePackager::getIngredient);
		int[] matches = RecipeMatcher.findMatches(input, matchers);
		if(matches == null) {
			return false;
		}
		for(int i = 0; i < matches.length; ++i) {
			if(input.get(i).func_190916_E() < currentPattern.getInputs().get(matches[i]).func_190916_E()) {
				return false;
			}
		}
		return true;
	}

	protected boolean canStart() {
		getPattern();
		if(currentPattern == null) {
			return false;
		}
		if(!isInputValid()) {
			return false;
		}
		ItemStack slotStack = inventory.func_70301_a(9);
		ItemStack outputStack = currentPattern.getOutput();
		return slotStack.func_190926_b() || slotStack.func_77969_a(outputStack) && ItemStack.areItemStackShareTagsEqual(slotStack, outputStack) && slotStack.func_190916_E()+1 <= outputStack.func_77976_d();
	}

	protected boolean canFinish() {
		return remainingProgress <= 0 && isInputValid();
	}

	protected void getPattern() {
		if(currentPattern != null && lockPattern) {
			return;
		}
		lockPattern = false;
		currentPattern = null;
		if(powered) {
			return;
		}
		List<ItemStack> input = inventory.stacks.subList(0, 9).stream().filter(stack->!stack.func_190926_b()).collect(Collectors.toList());
		if(input.isEmpty()) {
			return;
		}
		for(IPackagePattern pattern : patternList) {
			if(disjoint) {
				if(MiscUtil.removeExactSet(input, pattern.getInputs(), true)) {
					currentPattern = pattern;
					return;
				}
			}
			else {
				List<Ingredient> matchers = Lists.transform(pattern.getInputs(), TilePackager::getIngredient);
				int[] matches = RecipeMatcher.findMatches(input, matchers);
				if(matches != null) {
					currentPattern = pattern;
					return;
				}
			}
		}
	}

	protected void tickProcess() {
		int energy = energyStorage.extractEnergy(Math.min(energyUsage, remainingProgress), false);
		remainingProgress -= energy;
	}

	protected void finishProcess() {
		if(currentPattern == null) {
			getPattern();
		}
		if(currentPattern == null) {
			endProcess();
			return;
		}
		List<ItemStack> input = inventory.stacks.subList(0, 9).stream().filter(stack->!stack.func_190926_b()).collect(Collectors.toList());
		if(input.isEmpty()) {
			endProcess();
			return;
		}
		if(!lockPattern && disjoint) {
			if(!MiscUtil.removeExactSet(input, currentPattern.getInputs(), true)) {
				endProcess();
				return;
			}
			if(inventory.func_70301_a(9).func_190926_b()) {
				inventory.func_70299_a(9, currentPattern.getOutput());
			}
			else if(inventory.func_70301_a(9).func_77973_b() instanceof IPackageItem) {
				inventory.func_70301_a(9).func_190917_f(1);
			}
			else {
				endProcess();
				return;
			}
			MiscUtil.removeExactSet(input, currentPattern.getInputs(), false);
		}
		else {
			List<Ingredient> matchers = Lists.transform(currentPattern.getInputs(), TilePackager::getIngredient);
			int[] matches = RecipeMatcher.findMatches(input, matchers);
			if(matches == null) {
				endProcess();
				return;
			}
			if(inventory.func_70301_a(9).func_190926_b()) {
				inventory.func_70299_a(9, currentPattern.getOutput());
			}
			else if(inventory.func_70301_a(9).func_77973_b() instanceof IPackageItem) {
				inventory.func_70301_a(9).func_190917_f(1);
			}
			else {
				endProcess();
				return;
			}
			for(int i = 0; i < matches.length; ++i) {
				input.get(i).func_190918_g(currentPattern.getInputs().get(matches[i]).func_190916_E());
			}
		}
		for(int i = 0; i < 9; ++i) {
			if(inventory.func_70301_a(i).func_190926_b()) {
				inventory.func_70299_a(i, ItemStack.field_190927_a);
			}
		}
	}

	protected void startProcess() {
		remainingProgress = energyReq;
		func_70296_d();
	}

	public void endProcess() {
		remainingProgress = 0;
		isWorking = false;
		lockPattern = false;
		func_70296_d();
	}

	protected void ejectItem() {
		for(EnumFacing facing : EnumFacing.field_82609_l) {
			TileEntity te = field_145850_b.func_175625_s(field_174879_c.func_177972_a(facing));
			if(te instanceof TileUnpackager) {
				TileUnpackager tile = (TileUnpackager)te;
				ItemStack stack = inventory.func_70301_a(9);
				if(!stack.func_190926_b()) {
					IItemHandler itemHandler = tile.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, facing.func_176734_d());
					ItemStack stackRem = ItemHandlerHelper.insertItem(itemHandler, stack, false);
					inventory.func_70299_a(9, stackRem);
				}
			}
		}
	}

	protected void chargeEnergy() {
		ItemStack energyStack = inventory.func_70301_a(11);
		if(energyStack.hasCapability(CapabilityEnergy.ENERGY, null)) {
			int energyRequest = Math.min(energyStorage.getMaxReceive(), energyStorage.getMaxEnergyStored() - energyStorage.getEnergyStored());
			energyStorage.receiveEnergy(energyStack.getCapability(CapabilityEnergy.ENERGY, null).extractEnergy(energyRequest, false), false);
			if(energyStack.func_190916_E() <= 0) {
				inventory.func_70299_a(11, ItemStack.field_190927_a);
			}
		}
	}

	public void updatePowered() {
		if(field_145850_b.func_175687_A(field_174879_c) > 0 != powered) {
			powered = !powered;
			func_70296_d();
		}
	}

	@Override
	public int getComparatorSignal() {
		if(isWorking) {
			return 1;
		}
		if(!inventory.func_70301_a(9).func_190926_b()) {
			return 15;
		}
		return 0;
	}

	public HostHelperTilePackager hostHelper;

	@Override
	public void func_145843_s() {
		super.func_145843_s();
		if(hostHelper != null) {
			hostHelper.invalidate();
		}
	}

	@Override
	public void onChunkUnload() {
		super.onChunkUnload();
		if(hostHelper != null) {
			hostHelper.invalidate();
		}
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public IGridNode getGridNode(AEPartLocation dir) {
		return getActionableNode();
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public AECableType getCableConnectionType(AEPartLocation dir) {
		return AECableType.SMART;
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public void securityBreak() {
		field_145850_b.func_175655_b(field_174879_c, true);
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public IGridNode getActionableNode() {
		return hostHelper.getNode();
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public boolean pushPattern(ICraftingPatternDetails patternDetails, InventoryCrafting table) {
		if(hostHelper.isActive() && patternDetails instanceof PackageCraftingPatternHelper) {
			PackageCraftingPatternHelper pattern = (PackageCraftingPatternHelper)patternDetails;
			ItemStack outputStack = pattern.pattern.getOutput();
			if(canPushPattern()) {
				ItemStack slotStack = inventory.func_70301_a(9);
				if(slotStack.func_190926_b() || slotStack.func_77969_a(outputStack) && ItemStack.areItemStackShareTagsEqual(slotStack, outputStack) && slotStack.func_190916_E()+1 <= outputStack.func_77976_d()) {
					currentPattern = pattern.pattern;
					lockPattern = true;
					for(int i = 0; i < table.func_70302_i_() && i < 9; ++i) {
						inventory.func_70299_a(i, table.func_70301_a(i).func_77946_l());
					}
					return true;
				}
			}
			for(BlockPos posP : BlockPos.func_177975_b(field_174879_c.func_177982_a(-1, -1, -1), field_174879_c.func_177982_a(1, 1, 1))) {
				TileEntity te = field_145850_b.func_175625_s(posP);
				if(te instanceof TilePackagerExtension) {
					TilePackagerExtension extension = (TilePackagerExtension)te;
					if(extension.packager == this && extension.hostHelper.isActive() && getActionableNode().getGrid() == extension.getActionableNode().getGrid() && extension.canPushPattern()) {
						ItemStack slotStack = extension.inventory.func_70301_a(9);
						if(slotStack.func_190926_b() || slotStack.func_77969_a(outputStack) && ItemStack.areItemStackShareTagsEqual(slotStack, outputStack) && slotStack.func_190916_E()+1 <= outputStack.func_77976_d()) {
							extension.currentPattern = pattern.pattern;
							extension.lockPattern = true;
							for(int i = 0; i < table.func_70302_i_() && i < 9; ++i) {
								extension.inventory.func_70299_a(i, table.func_70301_a(i).func_77946_l());
							}
							return true;
						}
					}
				}
			}
		}
		return false;
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public boolean isBusy() {
		if(canPushPattern()) {
			return false;
		}
		for(BlockPos posP : BlockPos.func_177975_b(field_174879_c.func_177982_a(-1, -1, -1), field_174879_c.func_177982_a(1, 1, 1))) {
			TileEntity te = field_145850_b.func_175625_s(posP);
			if(te instanceof TilePackagerExtension) {
				TilePackagerExtension extension = (TilePackagerExtension)te;
				if(extension.packager == this && getActionableNode().getGrid() == extension.getActionableNode().getGrid() && extension.canPushPattern()) {
					return false;
				}
			}
		}
		return true;
	}

	public boolean canPushPattern() {
		return !isWorking && inventory.stacks.subList(0, 9).stream().allMatch(ItemStack::func_190926_b);
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public void provideCrafting(ICraftingProviderHelper craftingTracker) {
		if(hostHelper.isActive()) {
			for(IPackagePattern pattern : patternList) {
				craftingTracker.addCraftingOption(this, new PackageCraftingPatternHelper(pattern));
			}
		}
	}

	@Optional.Method(modid="appliedenergistics2")
	@MENetworkEventSubscribe
	public void onChannelsChanged(MENetworkChannelsChanged event) {
		hostHelper.postPatternChange();
	}

	@Optional.Method(modid="appliedenergistics2")
	@MENetworkEventSubscribe
	public void onPowerStatusChange(MENetworkPowerStatusChange event) {
		hostHelper.postPatternChange();
	}

	@Override
	public boolean loadConfig(NBTTagCompound nbt, EntityPlayer player) {
		mode = Mode.values()[nbt.func_74771_c("Mode")];
		if(nbt.func_74764_b("Recipes") && inventory.func_70301_a(10).func_190926_b()) {
			InventoryPlayer playerInventory = player.field_71071_by;
			for(int i = 0; i < playerInventory.func_70302_i_(); ++i) {
				ItemStack stack = playerInventory.func_70301_a(i);
				if(!stack.func_190926_b() && stack.func_77973_b() == ItemRecipeHolder.INSTANCE && !stack.func_77942_o()) {
					ItemStack stackCopy = stack.func_77979_a(1);
					IRecipeList recipeListObj = ItemRecipeHolder.INSTANCE.getRecipeList(stackCopy);
					List<IRecipeInfo> recipeList = MiscUtil.readRecipeListFromNBT(nbt.func_150295_c("Recipes", 10));
					recipeListObj.setRecipeList(recipeList);
					ItemRecipeHolder.INSTANCE.setRecipeList(stackCopy, recipeListObj);
					inventory.func_70299_a(10, stackCopy);
					break;
				}
			}
		}
		return true;
	}

	@Override
	public boolean saveConfig(NBTTagCompound nbt, EntityPlayer player) {
		nbt.func_74774_a("Mode", (byte)mode.ordinal());
		ItemStack listStack = inventory.func_70301_a(9);
		if(listStack.func_77973_b() instanceof IRecipeListItem) {
			List<IRecipeInfo> recipeList = ((IRecipeListItem)listStack.func_77973_b()).getRecipeList(listStack).getRecipeList();
			if(!recipeList.isEmpty()) {
				nbt.func_74782_a("Recipes", MiscUtil.writeRecipeListToNBT(new NBTTagList(), recipeList));
			}
		}
		return true;
	}

	@Override
	public void func_145839_a(NBTTagCompound nbt) {
		if(hostHelper != null) {
			hostHelper.readFromNBT(nbt);
		}
		mode = Mode.values()[nbt.func_74771_c("Mode")];
		super.func_145839_a(nbt);
		isWorking = nbt.func_74767_n("Working");
		remainingProgress = nbt.func_74762_e("Progress");
		powered = nbt.func_74767_n("Powered");
		lockPattern = false;
		currentPattern = null;
		if(nbt.func_74764_b("Pattern")) {
			NBTTagCompound tag = nbt.func_74775_l("Pattern");
			IRecipeInfo recipe = MiscUtil.readRecipeFromNBT(tag);
			if(recipe != null) {
				List<IPackagePattern> patterns = recipe.getPatterns();
				byte index = tag.func_74771_c("Index");
				if(index >= 0 && index < patterns.size()) {
					currentPattern = patterns.get(index);
					lockPattern = true;
				}
			}
		}
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbt) {
		super.func_189515_b(nbt);
		nbt.func_74774_a("Mode", (byte)mode.ordinal());
		nbt.func_74757_a("Working", isWorking);
		nbt.func_74768_a("Progress", remainingProgress);
		nbt.func_74757_a("Powered", powered);
		if(lockPattern) {
			NBTTagCompound tag = MiscUtil.writeRecipeToNBT(new NBTTagCompound(), currentPattern.getRecipeInfo());
			tag.func_74774_a("Index", (byte)currentPattern.getIndex());
			nbt.func_74782_a("Pattern", tag);
		}
		if(hostHelper != null) {
			hostHelper.writeToNBT(nbt);
		}
		return nbt;
	}

	public void changePackagingMode() {
		mode = Mode.values()[((mode.ordinal()+1) % 3)];
		((InventoryPackager)inventory).updatePatternList();
		func_70296_d();
	}

	@Override
	public void func_70296_d() {
		if(isWorking && !isInputValid()) {
			endProcess();
		}
		super.func_70296_d();
	}

	public int getScaledEnergy(int scale) {
		if(energyStorage.getMaxEnergyStored() <= 0) {
			return 0;
		}
		return scale * energyStorage.getEnergyStored() / energyStorage.getMaxEnergyStored();
	}

	public int getScaledProgress(int scale) {
		if(remainingProgress <= 0) {
			return 0;
		}
		return scale * (energyReq-remainingProgress) / energyReq;
	}

	@SideOnly(Side.CLIENT)
	@Override
	public GuiContainer getClientGuiElement(EntityPlayer player, Object... args) {
		return new GuiPackager(new ContainerPackager(player.field_71071_by, this));
	}

	@Override
	public Container getServerGuiElement(EntityPlayer player, Object... args) {
		return new ContainerPackager(player.field_71071_by, this);
	}

	public static enum Mode {
		EXACT, DISJOINT, FIRST;

		public String getTooltip() {
			return I18n.func_74838_a("tile.packagedauto.packager.mode."+name().toLowerCase(Locale.US));
		}
	}
}
