package thelm.packagedauto.tile;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;

import appeng.api.networking.IGridHost;
import appeng.api.networking.IGridNode;
import appeng.api.networking.crafting.ICraftingPatternDetails;
import appeng.api.networking.crafting.ICraftingProvider;
import appeng.api.networking.crafting.ICraftingProviderHelper;
import appeng.api.networking.events.MENetworkChannelsChanged;
import appeng.api.networking.events.MENetworkEventSubscribe;
import appeng.api.networking.events.MENetworkPowerStatusChange;
import appeng.api.networking.security.IActionHost;
import appeng.api.util.AECableType;
import appeng.api.util.AEPartLocation;
import it.unimi.dsi.fastutil.booleans.BooleanArrayList;
import it.unimi.dsi.fastutil.booleans.BooleanList;
import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.InventoryPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.InventoryCrafting;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ITickable;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.text.translation.I18n;
import net.minecraftforge.energy.CapabilityEnergy;
import net.minecraftforge.fml.common.Loader;
import net.minecraftforge.fml.common.Optional;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.IItemHandler;
import thelm.packagedauto.api.IPackageCraftingMachine;
import thelm.packagedauto.api.IPackageItem;
import thelm.packagedauto.api.IRecipeInfo;
import thelm.packagedauto.api.IRecipeList;
import thelm.packagedauto.api.ISettingsCloneable;
import thelm.packagedauto.api.MiscUtil;
import thelm.packagedauto.client.gui.GuiUnpackager;
import thelm.packagedauto.container.ContainerUnpackager;
import thelm.packagedauto.energy.EnergyStorage;
import thelm.packagedauto.integration.appeng.AppEngUtil;
import thelm.packagedauto.integration.appeng.networking.HostHelperTileUnpackager;
import thelm.packagedauto.integration.appeng.recipe.RecipeCraftingPatternHelper;
import thelm.packagedauto.inventory.InventoryUnpackager;
import thelm.packagedauto.item.ItemRecipeHolder;

@Optional.InterfaceList({
	@Optional.Interface(iface="appeng.api.networking.IGridHost", modid="appliedenergistics2"),
	@Optional.Interface(iface="appeng.api.networking.security.IActionHost", modid="appliedenergistics2"),
	@Optional.Interface(iface="appeng.api.networking.crafting.ICraftingProvider", modid="appliedenergistics2")
})
public class TileUnpackager extends TileBase implements ITickable, ISettingsCloneable, IGridHost, IActionHost, ICraftingProvider {

	public static int energyCapacity = 5000;
	public static int energyUsage = 50;
	public static int refreshInterval = 4;
	public static boolean drawMEEnergy = true;

	public final PackageTracker[] trackers = new PackageTracker[10];
	public List<IRecipeInfo> recipeList = new ArrayList<>();
	public boolean powered = false;
	public boolean blocking = false;
	public int trackerCount = 6;
	public int roundRobinIndex = 0;

	public TileUnpackager() {
		setInventory(new InventoryUnpackager(this));
		setEnergyStorage(new EnergyStorage(this, energyCapacity));
		for(int i = 0; i < trackers.length; ++i) {
			trackers[i] = new PackageTracker();
		}
		if(Loader.isModLoaded("appliedenergistics2")) {
			hostHelper = new HostHelperTileUnpackager(this);
		}
	}

	@Override
	protected String getLocalizedName() {
		return I18n.func_74838_a("tile.packagedauto.unpackager.name");
	}

	@Override
	public String getConfigTypeName() {
		return "tile.packagedauto.unpackager.name";
	}

	@Override
	public void onLoad() {
		updatePowered();
	}

	@Override
	public void func_73660_a() {
		if(!field_145850_b.field_72995_K) {
			chargeEnergy();
			if(field_145850_b.func_82737_E() % refreshInterval == 0) {
				fillTrackers();
				emptyTrackers();
				if(drawMEEnergy && hostHelper != null && hostHelper.isActive()) {
					hostHelper.chargeEnergy();
				}
			}
		}
	}

	protected void fillTrackers() {
		List<PackageTracker> emptyTrackers = Arrays.stream(trackers).limit(trackerCount).filter(t->t.isEmpty()).collect(Collectors.toList());
		List<PackageTracker> nonEmptyTrackers = Arrays.stream(trackers).filter(t->!t.isEmpty()).filter(t->!t.isFilled()).collect(Collectors.toList());
		for(int i = 0; i < 9; ++i) {
			if(energyStorage.getEnergyStored() >= energyUsage) {
				ItemStack stack = inventory.func_70301_a(i);
				if(!stack.func_190926_b() && stack.func_77973_b() instanceof IPackageItem) {
					IPackageItem packageItem = (IPackageItem)stack.func_77973_b();
					boolean flag = false;
					for(PackageTracker tracker : nonEmptyTrackers) {
						if(tracker.tryAcceptPackage(packageItem, stack, i)) {
							flag = true;
							stack.func_190918_g(1);
							if(stack.func_190926_b()) {
								inventory.func_70299_a(i, ItemStack.field_190927_a);
							}
							else {
								tracker.setRejectedIndex(i, true);
							}
							energyStorage.extractEnergy(energyUsage, false);
							break;
						}
						else {
							tracker.setRejectedIndex(i, true);
						}
					}
					if(!flag) {
						for(PackageTracker tracker : emptyTrackers) {
							if(tracker.tryAcceptPackage(packageItem, stack, i)) {
								stack.func_190918_g(1);
								if(stack.func_190926_b()) {
									inventory.func_70299_a(i, ItemStack.field_190927_a);
								}
								else {
									tracker.setRejectedIndex(i, true);
								}
								energyStorage.extractEnergy(energyUsage, false);
								break;
							}
							else {
								tracker.setRejectedIndex(i, true);
							}
						}
					}
				}
			}
		}
	}

	protected void emptyTrackers() {
		List<EnumFacing> directions = Lists.newArrayList(EnumFacing.field_82609_l);
		Collections.rotate(directions, roundRobinIndex);
		for(EnumFacing facing : directions) {
			TileEntity tile = field_145850_b.func_175625_s(field_174879_c.func_177972_a(facing));
			if(tile instanceof IPackageCraftingMachine) {
				IPackageCraftingMachine machine = (IPackageCraftingMachine)tile;
				for(PackageTracker tracker : trackers) {
					if(tracker.isFilled() && tracker.recipe != null && tracker.recipe.getRecipeType().hasMachine()) {
						if(!machine.isBusy() && machine.acceptPackage(tracker.recipe, Lists.transform(tracker.recipe.getInputs(), ItemStack::func_77946_l), facing.func_176734_d())) {
							tracker.clearRecipe();
							roundRobinIndex = (roundRobinIndex+1) % 6;
							func_70296_d();
							break;
						}
					}
				}
				continue;
			}
		}
		if(!powered) {
			directions = Lists.newArrayList(EnumFacing.field_82609_l);
			Collections.rotate(directions, roundRobinIndex);
			for(EnumFacing facing : directions) {
				TileEntity tile = field_145850_b.func_175625_s(field_174879_c.func_177972_a(facing));
				if(!validSendTarget(tile, facing.func_176734_d())) {
					continue;
				}
				if(!tile.hasCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, facing.func_176734_d())) {
					continue;
				}
				IItemHandler itemHandler = tile.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, facing.func_176734_d());
				if(blocking && !MiscUtil.isEmpty(itemHandler)) {
					continue;
				}
				PackageTracker trackerToEmpty = Arrays.stream(trackers).filter(t->t.isFilled() && t.facing == null && t.recipe != null && !t.recipe.getRecipeType().hasMachine()).findFirst().orElse(null);
				if(trackerToEmpty == null) {
					continue;
				}
				if(trackerToEmpty.toSend.isEmpty()) {
					trackerToEmpty.setupToSend();
				}
				boolean acceptsAll = true;
				for(int i = 0; i < trackerToEmpty.toSend.size(); ++i) {
					ItemStack stack = trackerToEmpty.toSend.get(i);
					ItemStack stackRem = MiscUtil.insertItem(itemHandler, stack, false, true);
					acceptsAll &= stackRem.func_190916_E() < stack.func_190916_E();
				}
				trackerToEmpty.toSend.removeIf(ItemStack::func_190926_b);
				if(acceptsAll) {
					trackerToEmpty.facing = facing;
					roundRobinIndex = (roundRobinIndex+1) % 6;
				}
				func_70296_d();
			}
		}
		for(EnumFacing facing : EnumFacing.field_82609_l) {
			TileEntity tile = field_145850_b.func_175625_s(field_174879_c.func_177972_a(facing));
			PackageTracker trackerToEmpty = Arrays.stream(trackers).filter(t->t.facing == facing).findFirst().orElse(null);
			if(trackerToEmpty == null) {
				continue;
			}
			if(trackerToEmpty.toSend.isEmpty()) {
				trackerToEmpty.setupToSend();
			}
			boolean ordered = false;
			if(trackerToEmpty.recipe != null) {
				ordered = trackerToEmpty.recipe.getRecipeType().isOrdered();
			}
			if(!validSendTarget(tile, facing.func_176734_d())) {
				trackerToEmpty.facing = null;
				continue;
			}
			if(!tile.hasCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, facing.func_176734_d())) {
				trackerToEmpty.facing = null;
				continue;
			}
			IItemHandler itemHandler = tile.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, facing.func_176734_d());
			for(int i = 0; i < trackerToEmpty.toSend.size(); ++i) {
				ItemStack stack = trackerToEmpty.toSend.get(i);
				ItemStack stackRem = MiscUtil.insertItem(itemHandler, stack, ordered, false);
				trackerToEmpty.toSend.set(i, stackRem);
			}
			trackerToEmpty.toSend.removeIf(ItemStack::func_190926_b);
			if(trackerToEmpty.toSend.isEmpty()) {
				trackerToEmpty.clearRecipe();
			}
			func_70296_d();
		}
	}

	protected boolean validSendTarget(TileEntity tile, EnumFacing facing) {
		return tile != null &&
				!(tile instanceof IPackageCraftingMachine) &&
				!(tile instanceof TilePackager) &&
				!(tile instanceof TilePackagerExtension) &&
				!(tile instanceof TileUnpackager) &&
				!isInterface(tile, facing.func_176734_d());
	}

	protected void chargeEnergy() {
		ItemStack energyStack = inventory.func_70301_a(10);
		if(energyStack.hasCapability(CapabilityEnergy.ENERGY, null)) {
			int energyRequest = Math.min(energyStorage.getMaxReceive(), energyStorage.getMaxEnergyStored() - energyStorage.getEnergyStored());
			energyStorage.receiveEnergy(energyStack.getCapability(CapabilityEnergy.ENERGY, null).extractEnergy(energyRequest, false), false);
			if(energyStack.func_190916_E() <= 0) {
				inventory.func_70299_a(10, ItemStack.field_190927_a);
			}
		}
	}

	public void updatePowered() {
		if(field_145850_b.func_175687_A(field_174879_c) > 0 != powered) {
			powered = !powered;
			func_70296_d();
		}
	}

	@Override
	public int getComparatorSignal() {
		return Math.min((int)Arrays.stream(trackers).filter(t->t.isFilled()).count(), 15);
	}

	public HostHelperTileUnpackager hostHelper;

	@Override
	public void func_145843_s() {
		super.func_145843_s();
		if(hostHelper != null) {
			hostHelper.invalidate();
		}
	}

	@Override
	public void onChunkUnload() {
		super.onChunkUnload();
		if(hostHelper != null) {
			hostHelper.invalidate();
		}
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public IGridNode getGridNode(AEPartLocation dir) {
		return getActionableNode();
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public AECableType getCableConnectionType(AEPartLocation dir) {
		return AECableType.SMART;
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public void securityBreak() {
		field_145850_b.func_175655_b(field_174879_c, true);
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public IGridNode getActionableNode() {
		return hostHelper.getNode();
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public boolean pushPattern(ICraftingPatternDetails patternDetails, InventoryCrafting table) {
		if(hostHelper.isActive() && !isBusy() && patternDetails instanceof RecipeCraftingPatternHelper) {
			RecipeCraftingPatternHelper pattern = (RecipeCraftingPatternHelper)patternDetails;
			int energyReq = energyUsage*pattern.recipe.getPatterns().size();
			if(energyStorage.getEnergyStored() >= energyReq) {
				PackageTracker tracker = Arrays.stream(trackers).limit(trackerCount).filter(PackageTracker::isEmpty).findFirst().get();
				tracker.fillRecipe(pattern.recipe);
				energyStorage.extractEnergy(energyReq, false);
				return true;
			}
		}
		return false;
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public boolean isBusy() {
		return Arrays.stream(trackers).limit(trackerCount).noneMatch(PackageTracker::isEmpty);
	}

	@Optional.Method(modid="appliedenergistics2")
	@Override
	public void provideCrafting(ICraftingProviderHelper craftingTracker) {
		if(hostHelper.isActive()) {
			for(IRecipeInfo pattern : recipeList) {
				if(!pattern.getOutputs().isEmpty()) {
					craftingTracker.addCraftingOption(this, new RecipeCraftingPatternHelper(pattern));
				}
			}
		}
	}

	@Optional.Method(modid="appliedenergistics2")
	@MENetworkEventSubscribe
	public void onChannelsChanged(MENetworkChannelsChanged event) {
		hostHelper.postPatternChange();
	}

	@Optional.Method(modid="appliedenergistics2")
	@MENetworkEventSubscribe
	public void onPowerStatusChange(MENetworkPowerStatusChange event) {
		hostHelper.postPatternChange();
	}

	protected boolean isInterface(TileEntity tile, EnumFacing facing) {
		if(Loader.isModLoaded("appliedenergistics2")) {
			return AppEngUtil.isInterface(tile, facing);
		}
		return false;
	}

	public int getScaledEnergy(int scale) {
		if(energyStorage.getMaxEnergyStored() <= 0) {
			return 0;
		}
		return scale * energyStorage.getEnergyStored() / energyStorage.getMaxEnergyStored();
	}

	@Override
	public boolean loadConfig(NBTTagCompound nbt, EntityPlayer player) {
		blocking = nbt.func_74767_n("Blocking");
		trackerCount = nbt.func_74771_c("Trackers");
		if(nbt.func_74764_b("Recipes") && inventory.func_70301_a(9).func_190926_b()) {
			InventoryPlayer playerInventory = player.field_71071_by;
			for(int i = 0; i < playerInventory.func_70302_i_(); ++i) {
				ItemStack stack = playerInventory.func_70301_a(i);
				if(!stack.func_190926_b() && stack.func_77973_b() == ItemRecipeHolder.INSTANCE && !stack.func_77942_o()) {
					ItemStack stackCopy = stack.func_77979_a(1);
					IRecipeList recipeListObj = ItemRecipeHolder.INSTANCE.getRecipeList(stackCopy);
					List<IRecipeInfo> recipeList = MiscUtil.readRecipeListFromNBT(nbt.func_150295_c("Recipes", 10));
					recipeListObj.setRecipeList(recipeList);
					ItemRecipeHolder.INSTANCE.setRecipeList(stackCopy, recipeListObj);
					inventory.func_70299_a(9, stackCopy);
					break;
				}
			}
		}
		return true;
	}

	@Override
	public boolean saveConfig(NBTTagCompound nbt, EntityPlayer player) {
		nbt.func_74757_a("Blocking", blocking);
		nbt.func_74774_a("Trackers", (byte)trackerCount);
		if(!recipeList.isEmpty()) {
			nbt.func_74782_a("Recipes", MiscUtil.writeRecipeListToNBT(new NBTTagList(), recipeList));
		}
		return true;
	}

	@Override
	public void func_145839_a(NBTTagCompound nbt) {
		if(hostHelper != null) {
			hostHelper.readFromNBT(nbt);
		}
		super.func_145839_a(nbt);
		blocking = nbt.func_74767_n("Blocking");
		trackerCount = nbt.func_74764_b("Trackers") ? nbt.func_74771_c("Trackers") : 6;
		powered = nbt.func_74767_n("Powered");
		for(int i = 0; i < trackers.length; ++i) {
			trackers[i].readFromNBT(nbt.func_74775_l(String.format("Tracker%02d", i)));
		}
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbt) {
		super.func_189515_b(nbt);
		nbt.func_74757_a("Blocking", blocking);
		nbt.func_74774_a("Trackers", (byte)trackerCount);
		nbt.func_74757_a("Powered", powered);
		for(int i = 0; i < trackers.length; ++i) {
			NBTTagCompound subNBT = new NBTTagCompound();
			trackers[i].writeToNBT(subNBT);
			nbt.func_74782_a(String.format("Tracker%02d", i), subNBT);
		}
		if(hostHelper != null) {
			hostHelper.writeToNBT(nbt);
		}
		return nbt;
	}

	public void changeBlockingMode() {
		blocking = !blocking;
		func_70296_d();
	}

	public void changeTrackerCount(boolean decrease) {
		trackerCount = MathHelper.func_76125_a(trackerCount + (decrease ? -1 : 1), 1, 10);
		func_70296_d();
	}

	@SideOnly(Side.CLIENT)
	@Override
	public GuiContainer getClientGuiElement(EntityPlayer player, Object... args) {
		return new GuiUnpackager(new ContainerUnpackager(player.field_71071_by, this));
	}

	@Override
	public Container getServerGuiElement(EntityPlayer player, Object... args) {
		return new ContainerUnpackager(player.field_71071_by, this);
	}

	public class PackageTracker {

		public boolean[] rejectedIndexes = new boolean[9];
		public IRecipeInfo recipe;
		public int amount;
		public BooleanList received = new BooleanArrayList();
		public List<ItemStack> toSend = new ArrayList<>();
		public EnumFacing facing;

		public void clearRecipe() {
			clearRejectedIndexes();
			recipe = null;
			amount = 0;
			received.clear();
			facing = null;
			if(field_145850_b != null && !field_145850_b.field_72995_K) {
				func_70296_d();
			}
		}

		public void fillRecipe(IRecipeInfo recipe) {
			this.recipe = recipe;
			amount = recipe.getPatterns().size();
			received.clear();
			received.size(amount);
			for(int i = 0; i < received.size(); ++i) {
				received.set(i, true);
			}
			if(field_145850_b != null && !field_145850_b.field_72995_K) {
				func_70296_d();
			}
		}

		public boolean tryAcceptPackage(IPackageItem packageItem, ItemStack stack, int invIndex) {
			if(rejectedIndexes[invIndex]) {
				return false;
			}
			IRecipeInfo recipe = packageItem.getRecipeInfo(stack);
			int index = packageItem.getIndex(stack);
			if(recipe != null && recipe.isValid() && recipe.validPatternIndex(index)) {
				if(this.recipe == null) {
					this.recipe = recipe;
					amount = recipe.getPatterns().size();
					received.size(amount);
					received.set(index, true);
					func_70296_d();
					return true;
				}
				else if(this.recipe.equals(recipe)) {
					if(!received.get(index)) {
						received.set(index, true);
						func_70296_d();
						return true;
					}
				}
			}
			return false;
		}

		public void setRejectedIndex(int index, boolean rejected) {
			rejectedIndexes[index] = rejected;
		}

		public void clearRejectedIndexes() {
			Arrays.fill(rejectedIndexes, false);
		}

		public boolean isFilled() {
			if(!toSend.isEmpty()) {
				return true;
			}
			if(received.isEmpty()) {
				return false;
			}
			for(boolean b : received) {
				if(!b) {
					return false;
				}
			}
			return true;
		}

		public boolean isEmpty() {
			return recipe == null || !recipe.isValid();
		}

		public void setupToSend() {
			if(isEmpty() || recipe.getRecipeType().hasMachine() || !toSend.isEmpty()) {
				return;
			}
			toSend.addAll(Lists.transform(recipe.getInputs(), ItemStack::func_77946_l));
		}

		public void readFromNBT(NBTTagCompound nbt) {
			clearRecipe();
			NBTTagCompound tag = nbt.func_74775_l("Recipe");
			IRecipeInfo recipe = MiscUtil.readRecipeFromNBT(tag);
			if(recipe != null) {
				this.recipe = recipe;
				amount = nbt.func_74771_c("Amount");
				received.size(amount);
				byte[] receivedArray = nbt.func_74770_j("Received");
				for(int i = 0; i < received.size(); ++i) {
					received.set(i, receivedArray[i] != 0);
				}
			}
			MiscUtil.loadAllItems(nbt.func_150295_c("ToSend", 10), toSend);
			if(nbt.func_74764_b("Facing")) {
				facing = EnumFacing.func_82600_a(nbt.func_74771_c("Facing"));
			}
		}

		public NBTTagCompound writeToNBT(NBTTagCompound nbt) {
			if(recipe != null) {
				NBTTagCompound tag = MiscUtil.writeRecipeToNBT(new NBTTagCompound(), recipe);
				nbt.func_74782_a("Recipe", tag);
				nbt.func_74774_a("Amount", (byte)amount);
				byte[] receivedArray = new byte[received.size()];
				for(int i = 0; i < received.size(); ++i) {
					receivedArray[i] = (byte)(received.getBoolean(i) ? 1 : 0);
				}
				nbt.func_74773_a("Received", receivedArray);
			}
			nbt.func_74782_a("ToSend", MiscUtil.saveAllItems(new NBTTagList(), toSend));
			if(facing != null) {
				nbt.func_74774_a("Facing", (byte)facing.func_176745_a());
			}
			return nbt;
		}

		public int getSyncValue() {
			int val = 0;
			for(int i = 0; i < received.size(); ++i) {
				if(received.getBoolean(i)) {
					val |= 1 << i;
				}
			}
			val <<= 4;
			val |= amount;
			return val;
		}

		public void setSyncValue(int val) {
			amount = val & 15;
			received.size(amount);
			val >>>= 4;
			for(int i = 0; i < received.size(); ++i) {
				received.set(i, ((val >>> i) & 1) != 0);
			}
		}
	}
}
