package thelm.packagedauto.network.packet;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;

import net.minecraft.entity.player.ServerPlayerEntity;
import net.minecraft.network.PacketBuffer;
import net.minecraft.util.Direction;
import net.minecraft.util.RegistryKey;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.registry.Registry;
import net.minecraft.world.World;
import net.minecraftforge.fml.network.NetworkEvent;
import net.minecraftforge.fml.network.PacketDistributor;
import thelm.packagedauto.api.DirectionalGlobalPos;
import thelm.packagedauto.client.WorldOverlayRenderer;
import thelm.packagedauto.network.PacketHandler;

public class DirectionalMarkerPacket {

	private final List<DirectionalGlobalPos> positions;
	private final int color;
	private final int lifetime;

	public DirectionalMarkerPacket(List<DirectionalGlobalPos> positions, int color, int lifetime) {
		this.positions = positions;
		this.color = color;
		this.lifetime = lifetime;
	}

	public void encode(PacketBuffer buf) {
		buf.writeByte(positions.size());
		for(DirectionalGlobalPos globalPos : positions) {
			buf.writeResourceLocation(globalPos.dimension().location());
			buf.writeInt(globalPos.x());
			buf.writeInt(globalPos.y());
			buf.writeInt(globalPos.z());
			buf.writeByte(globalPos.direction().get3DDataValue());
		}
		buf.writeMedium(color);
		buf.writeShort(lifetime);
	}

	public static DirectionalMarkerPacket decode(PacketBuffer buf) {
		int size = buf.readByte();
		List<DirectionalGlobalPos> positions = new ArrayList<>(size);
		for(int i = 0; i < size; ++i) {
			RegistryKey<World> dimension = RegistryKey.create(Registry.DIMENSION_REGISTRY, buf.readResourceLocation());
			BlockPos pos = new BlockPos(buf.readInt(), buf.readInt(), buf.readInt());
			Direction direction = Direction.from3DDataValue(buf.readByte());
			positions.add(new DirectionalGlobalPos(dimension, pos, direction));
		}
		return new DirectionalMarkerPacket(positions, buf.readUnsignedMedium(), buf.readUnsignedShort());
	}

	public void handle(Supplier<NetworkEvent.Context> ctx) {
		ctx.get().enqueueWork(()->{
			WorldOverlayRenderer.INSTANCE.addDirectionalMarkers(positions, color, lifetime);
		});
		ctx.get().setPacketHandled(true);
	}

	public static void sendDirectionalMarkers(ServerPlayerEntity player, List<DirectionalGlobalPos> positions, int color, int lifetime) {
		PacketHandler.INSTANCE.send(PacketDistributor.PLAYER.with(()->player), new DirectionalMarkerPacket(positions, color, lifetime));
	}
}
