/**
 * World In a Jar
 * Copyright (C) 2024  VulpixelMC
 * <p>
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * <p>
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package gay.sylv.wij.impl.block.entity;

import com.mojang.authlib.GameProfile;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.serialization.MapCodec;
import gay.sylv.wij.api.block.JarContainmentBlock;
import gay.sylv.wij.api.block.WorldJar;
import gay.sylv.wij.impl.Main;
import gay.sylv.wij.impl.block.Blocks;
import gay.sylv.wij.impl.block.tag.BlockTags;
import gay.sylv.wij.impl.client.render.*;
import gay.sylv.wij.impl.component.Components;
import gay.sylv.wij.impl.dimension.Dimensions;
import gay.sylv.wij.impl.duck.PlayerWithReturn;
import gay.sylv.wij.impl.network.JarChunkUpdatePayload;
import gay.sylv.wij.impl.network.JarLoadedAckPayload;
import gay.sylv.wij.impl.network.Networking;
import gay.sylv.wij.impl.network.client.JarEnterPayload;
import gay.sylv.wij.impl.network.client.JarLoadedPayload;
import gay.sylv.wij.impl.util.Constants;
import gay.sylv.wij.impl.util.WeakReferenceList;
import gay.sylv.wij.impl.util.jar.JarEntry;
import gay.sylv.wij.mixin.Accessor_BaseContainerBlockEntity;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayNetworking;
import net.fabricmc.fabric.api.entity.FakePlayer;
import net.fabricmc.fabric.api.event.player.PlayerBlockBreakEvents;
import net.fabricmc.fabric.api.networking.v1.PlayerLookup;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_1268;
import net.minecraft.class_1269;
import net.minecraft.class_1273;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import net.minecraft.class_181;
import net.minecraft.class_1922;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_2237;
import net.minecraft.class_2338;
import net.minecraft.class_2371;
import net.minecraft.class_243;
import net.minecraft.class_2464;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2624;
import net.minecraft.class_2680;
import net.minecraft.class_2823;
import net.minecraft.class_2841;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3610;
import net.minecraft.class_3965;
import net.minecraft.class_4076;
import net.minecraft.class_5321;
import net.minecraft.class_5454;
import net.minecraft.class_7225;
import net.minecraft.class_7923;
import net.minecraft.class_8527;
import net.minecraft.class_8567;
import net.minecraft.class_9062;
import net.minecraft.client.renderer.*;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joml.Matrix4f;
import org.joml.Matrix4fStack;

import java.nio.charset.StandardCharsets;
import java.util.*;

public class WorldJarBlockEntity extends class_2624 implements class_2823, WorldJar {
	private int scale = DEFAULT_SCALE;
	private class_2338 internalSpawnPos = DEFAULT_SPAWN_POS;
	
	private static final int DEFAULT_SCALE = 64;
	private static final class_2338 DEFAULT_SPAWN_POS = new class_2338(0, -64, 0);
	
	public static final WeakReferenceList<WorldJarBlockEntity> INSTANCES = new WeakReferenceList<>();
	public static final class_1273 CONTAINER_LOCK = new class_1273("glowcase");
	
	/**
	 * {@link JarLevelChunkSection}s that are loaded in the {@link WorldJarBlockEntity}.
	 */
	private final Long2ObjectMap<JarLevelChunkSection> chunkSections = new Long2ObjectOpenHashMap<>();
	
	/**
	 * The full versions of chunks that are loaded in the {@link WorldJarBlockEntity}. This is used in lighting.
	 */
	private final Long2ObjectMap<JarChunk> chunks = new Long2ObjectOpenHashMap<>();
	
	@Environment(EnvType.CLIENT)
	public JarRenderChunkRegion renderChunkRegion;
	
	/**
	 * The location of the target jar.
	 */
	@Nullable
	private Networking.JarLocation targetJarLocation;
	
	/**
	 * The {@link JarEntry} linking to the jar dimension.
	 */
	private JarEntry jarEntry = new JarEntry(-1);
	
	/**
	 * If the {@link class_2680}s in the jar have changed.
	 * <p>
	 * This is used in rendering to determine whether we need to rebuild the VBOs.
	 */
	public boolean statesChanged = false;
	
	private final Object2ObjectMap<UUID, FakePlayer> fakePlayers = new Object2ObjectOpenHashMap<>();
	
	/**
	 * True if loaded rather than placed.
	 */
	private boolean loadedNotPlaced = false;
	
	public WorldJarBlockEntity(class_2338 pos, class_2680 blockState) {
		super(Blocks.WORLD_JAR.type(), pos, blockState);
		INSTANCES.addAuto(this);
	}
	
	public float getVisualScale() {
		return 1.0f / scale;
	}
	
	public int getScale() {
		return scale;
	}
	
	public boolean isLocked() {
		return ((Accessor_BaseContainerBlockEntity) this).getLockKey() != class_1273.field_5817;
	}
	
	public void setLocked(boolean locked) {
		((Accessor_BaseContainerBlockEntity) this).setLockKey(locked ? CONTAINER_LOCK : class_1273.field_5817);
	}
	
	/**
	 * Sets a {@link class_2680} at the specified position.
	 * @author sylv
	 */
	public void setBlockState(class_2338 pos, class_2680 state) {
		var sectionPos = class_4076.method_18682(pos);
		var section = chunkSections.get(sectionPos.method_18694());
		if (section == null) return;
		section.setBlockState(pos.method_10263() & 15, pos.method_10264() & 15, pos.method_10260() & 15, state);
	}
	
	/**
	 * Gets a {@link class_2680} from the specified position.
	 * @return {@link class_2680}
	 * @author sylv
	 */
	public class_2680 getBlockState(class_2338 pos) {
		var sectionPos = class_4076.method_18682(pos);
		var section = chunkSections.get(sectionPos.method_18694());
		if (section == null) {
			return net.minecraft.class_2246.field_10124.method_9564();
		}
		return section.getBlockState(pos.method_10263() & 15, pos.method_10264() & 15, pos.method_10260() & 15);
	}
	
	/**
	 * Gets a {@link class_3610} from the specified position.
	 * @return a {@link class_3610} at the {@link class_2680} at the specified position.
	 */
	public class_3610 getFluidState(class_2338 pos) {
		return getBlockState(pos).method_26227();
	}
	
	public void updateBlockStates(MinecraftServer server) {
		class_1937 level = server.method_3847(Dimensions.JAR);
		int max = scale;
		for (int x = 0; x < max; x++) {
			for (int y = 0; y < max; y++) {
				for (int z = 0; z < max; z++) {
					class_2338 pos = new class_2338(x, y, z);
					assert level != null;
					class_2680 state = level.method_8320(pos.method_10081(getInternalPos()));
					setBlockState(pos, state);
				}
			}
		}
	}
	
	public void updateSectionStates(MinecraftServer server, class_4076 sectionPos) {
		class_1937 level = server.method_3847(Dimensions.JAR);
		int min = sectionPos.method_19527();
		int max = sectionPos.method_19530();
		for (int x = min; x < max; x++) {
			for (int y = min; y < max; y++) {
				for (int z = min; z < max; z++) {
					class_2338 pos = new class_2338(x, y, z);
					assert level != null;
					class_2680 state = level.method_8320(pos.method_10081(getInternalPos()));
					setBlockState(pos, state);
				}
			}
		}
	}
	
	/**
	 * Initializes the chunks server-side.
	 * @author sylv
	 */
	private void initializeServerChunks() {
		// initialize chunks
		chunkSections.clear();
		chunks.clear();
		int max = getChunkDiameter() - 1;
		for (int x = 0; x < max; x++) {
			for (int y = 0; y < max; y++) {
				for (int z = 0; z < max; z++) {
					var sectionPos = class_4076.method_18676(x, y, z);
					var chunkSection = new JarLevelChunkSection(sectionPos, false);
					var chunkPos = new class_1923(sectionPos.method_10263(), sectionPos.method_10260());
					var chunk = new JarChunk(chunkPos, this);
					
					// put chunk
					chunkSections.put(sectionPos.method_18694(), chunkSection);
					chunks.put(chunkPos.method_8324(), chunk);
				}
			}
		}
	}
	
	public void sendJarChunks(class_3222 player) {
		Networking.JarLocation jarLocation = getJarLocation();
		getChunkSections().forEach((pos, section) -> {
			class_4076 sectionPos = class_4076.method_18677(pos);
			class_2841<class_2680> blockStates = section.getBlockStates().method_39957();
			JarChunkUpdatePayload payload = new JarChunkUpdatePayload(jarLocation, sectionPos, blockStates);
			ServerPlayNetworking.send(player, payload);
		});
	}
	
	public void sendJarChunk(class_3222 player, class_4076 sectionPos) {
		Networking.JarLocation jarLocation = getJarLocation();
		JarLevelChunkSection section = getChunkSections().get(sectionPos.method_18694());
		if (section == null) return;
		class_2841<class_2680> blockStates = section.getBlockStates().method_39957();
		JarChunkUpdatePayload payload = new JarChunkUpdatePayload(jarLocation, sectionPos, blockStates);
		ServerPlayNetworking.send(player, payload);
	}
	
	private Networking.JarLocation getJarLocation() {
		assert this.field_11863 != null;
		return new Networking.JarLocation(this.method_11016(), this.field_11863.method_27983());
	}
	
	public class_8527 getChunk(int chunkX, int chunkZ) {
		long chunkPos = class_1923.method_8331(chunkX, chunkZ);
		return chunks.get(chunkPos);
	}
	
	@Override
	protected void method_11014(class_2487 tag, class_7225.class_7874 registries) {
		super.method_11014(tag, registries);
		loadedNotPlaced = true;
		class_2487 modTag = tag.method_10562(Constants.COMPAT_MOD_ID);
		scale = modTag.method_10550("scale");
		jarEntry = new JarEntry(modTag.method_10550("id"));
		internalSpawnPos = jarEntry.chunkPos().method_8323().method_10084();
		targetJarLocation = Networking.JarLocation.CODEC.parse(class_2509.field_11560, modTag.method_10580("target_jar_location")).result().orElse(null);
	}
	
	@Override
	protected void method_11007(class_2487 tag, class_7225.class_7874 registries) {
		super.method_11007(tag, registries);
		class_2487 modTag = new class_2487();
		if (scale != 0) modTag.method_10569("scale", scale);
		modTag.method_10569("id", jarEntry.id());
		if (targetJarLocation != null) Networking.JarLocation.CODEC.encodeStart(class_2509.field_11560, targetJarLocation).result().ifPresent(target -> modTag.method_10566("target_jar_location", target));
		tag.method_10566(Constants.COMPAT_MOD_ID, modTag);
	}
	
	@Override
	protected void method_57568(class_9473 componentInput) {
		class_1273 lockKey = ((Accessor_BaseContainerBlockEntity) this).getLockKey();
		super.method_57568(componentInput);
		((Accessor_BaseContainerBlockEntity) this).setLockKey(lockKey);
	}
	
	@Override
	protected @NotNull class_2561 method_17823() {
		return class_2561.method_43471("block.worldinajar.world_jar");
	}
	
	@Override
	protected @NotNull class_2371<class_1799> method_11282() {
		return class_2371.method_10211();
	}
	
	@Override
	protected void method_11281(class_2371<class_1799> items) {
	}
	
	@SuppressWarnings("DataFlowIssue") // this is actually nullable lol
	@Override
	protected @NotNull class_1703 method_5465(int containerId, class_1661 inventory) {
		return null;
	}
	
	@Override
	public void method_11012() {
		super.method_11012();
		assert field_11863 != null;
		if (!this.field_11863.method_8608()) {
			this.fakePlayers.forEach((uuid, fakePlayer) -> {
				Main.removeKnownFakePlayerWithJar(this, (class_3218) this.field_11863, fakePlayer);
			});
		}
	}
	
	@Override
	public void method_31662(class_1937 level) {
		super.method_31662(level);
		if (level.method_27983() == Dimensions.JAR) return;
		if (level.field_9236) {
			JarLevelLightEngine lightEngine = new JarLevelLightEngine(this, true, true);
			renderChunkRegion = new JarRenderChunkRegion(this, lightEngine);
			ClientPlayNetworking.send(new JarLoadedPayload(getJarLocation()));
		} else {
			initializeServerChunks();
			// Send to tracking players when server loads placed jar.
			if (!loadedNotPlaced) {
				for (class_3222 player : PlayerLookup.tracking(this)) {
					ServerPlayNetworking.send(player, new JarLoadedAckPayload(getJarLocation()));
				}
			}
		}
	}
	
	@SuppressWarnings("NullableProblems")
	@Nullable
	@Override
	public class_1937 method_10997() {
		return super.method_10997();
	}
	
	/**
	 * This method is called upon updating a chunk on the clientside. It first remaps {@link class_2680}s to the given {@link class_2841}&lt;{@link class_2680}&gt;, then recreates the {@link JarLevelChunkSection}s, and finally marks {@code statesChanged} as {@code true}.
	 * @author sylv
	 */
	@Environment(EnvType.CLIENT)
	public void onChunkUpdate(class_310 client, class_4076 sectionPos, class_2841<class_2680> blockStateContainer) {
		client.execute(() -> {
			// put chunk
			JarLevelChunkSection chunkSection = chunkSections.get(sectionPos.method_18694());
			class_1923 chunkPos = new class_1923(sectionPos.method_10263(), sectionPos.method_10260());
			JarChunk chunk = chunks.get(chunkPos.method_8324());
			
			// remap block states
			if (chunkSection == null) {
				chunkSection = new JarLevelChunkSection(sectionPos, true, blockStateContainer);
			} else {
				chunkSection.setBlockStates(blockStateContainer);
			}
			
			chunkSections.put(sectionPos.method_18694(), chunkSection);
			chunks.put(chunkPos.method_8324(), chunk);
			
			statesChanged = true;
		});
	}
	
	/**
	 * This method is called upon updating a block on the clientside.
	 * @author sylv
	 */
	@Environment(EnvType.CLIENT)
	public void onBlockUpdate(class_310 client, class_2338 blockPos, class_2680 blockState) {
		client.execute(() -> {
			// set block state
			this.setBlockState(blockPos, blockState);
			
			statesChanged = true;
		});
	}
	
	/**
	 * Returns how many chunks high/wide the {@link WorldJarBlockEntity} is. This always rounds up to include partial chunks.
	 * @return how many chunks high/wide the {@link WorldJarBlockEntity} is.
	 * @author sylv
	 */
	public int getChunkDiameter() {
		return class_4076.method_32204(scale) + 1;
	}
	
	public Long2ObjectMap<JarLevelChunkSection> getChunkSections() {
		return chunkSections;
	}
	
	@Nullable
	@Override
	public class_8527 method_12246(int chunkX, int chunkZ) {
		return getChunk(chunkX, chunkZ);
	}
	
	public class_2338 getInternalSpawnPos() {
		return internalSpawnPos;
	}
	
	public class_2338 getInternalPos() {
		return internalSpawnPos.method_10074();
	}
	
	@Override
	public boolean hasBlockPos(class_2338 pos) {
		return pos.method_19769(getCenterPos(), scale);
	}
	
	public @NotNull class_243 getCenterPos() {
		return getInternalPos().method_46558().method_1031(scale / 2.0d, scale / 2.0d, scale / 2.0d);
	}
	
	@Override
	public int method_5439() {
		return 0;
	}
	
	public Object2ObjectMap<UUID, FakePlayer> getFakePlayers() {
		return fakePlayers;
	}
	
	public FakePlayer getOrCreateFakePlayer(class_3218 outsideJarLevel, class_3222 player) {
		UUID uuid = UUID.nameUUIDFromBytes(player.method_5477().getString().getBytes(StandardCharsets.UTF_8));
		return fakePlayers.computeIfAbsent(uuid, ignored -> FakePlayer.get(outsideJarLevel, new GameProfile(uuid, player.method_5477().getString())));
	}
	
	public static class WorldJarBlock extends class_2237 {
		private static final MapCodec<WorldJarBlock> CODEC = method_54094(WorldJarBlock::new);
		
		public WorldJarBlock(class_2251 properties) {
			super(properties);
			PlayerBlockBreakEvents.BEFORE.register((class_1937 level, class_1657 player, class_2338 pos, class_2680 state, @Nullable class_2586 blockEntity) -> {
				// Prevent breaking/destruction in jar dimension
				return !(level.method_27983().equals(Dimensions.JAR) && state.method_27852(Blocks.WORLD_JAR.block()));
			});
		}
		
		@Override
		protected @NotNull MapCodec<? extends class_2237> method_53969() {
			return field_46280;
		}
		
		@Nullable
		@Override
		public class_2586 method_10123(class_2338 pos, class_2680 state) {
			return new WorldJarBlockEntity(pos, state);
		}
		
		@Override
		protected void method_9615(class_2680 state, class_1937 level, class_2338 pos, class_2680 oldState, boolean movedByPiston) {
			super.method_9615(state, level, pos, oldState, movedByPiston);
		}
		
		@Override
		protected @NotNull List<class_1799> method_9560(class_2680 state, class_8567.class_8568 params) {
			List<class_1799> drops = super.method_9560(state, params);
			class_2586 blockEntity = params.method_51876(class_181.field_1228);
			if (blockEntity instanceof WorldJarBlockEntity jar) {
				drops.getFirst().method_57379(Components.JAR_ENTRY_TYPE, jar.jarEntry);
			}
			return drops;
		}
		
		@Override
		protected @NotNull class_2464 method_9604(class_2680 state) {
			return class_2464.field_11458;
		}
		
		@Override
		protected float method_9594(class_2680 state, class_1657 player, class_1922 level, class_2338 pos) {
			// Prevent breaking in jar dimension
			if (!player.method_37908().method_27983().equals(Dimensions.JAR)) {
				return super.method_9594(state, player, level, pos);
			} else {
				return 0.0F;
			}
		}
		
		@Override
		protected @NotNull class_9062 method_55765(class_1799 stack, class_2680 state, class_1937 level, class_2338 pos, class_1657 player, class_1268 hand, class_3965 hitResult) {
			if (class_7923.field_41178.method_10221(stack.method_7909()).equals(class_2960.method_60655("glowcase", "lock")) && player.method_7337()) {
				return class_9062.field_47729;
			} else {
				return super.method_55765(stack, state, level, pos, player, hand, hitResult);
			}
		}
		
		@Override
		protected @NotNull class_1269 method_55766(class_2680 state, class_1937 level, class_2338 pos, class_1657 player, class_3965 hitResult) {
			boolean isReturnJar = level.method_27983().equals(Dimensions.JAR);
			
			if (level.method_8608()) {
				if (!isReturnJar) {
					ClientPlayNetworking.send(new JarEnterPayload(new Networking.JarLocation(pos, level.method_27983())));
				}
				
				return class_1269.field_21466;
			}
			if (!isReturnJar) return class_1269.field_5812;
			
			MinecraftServer server = level.method_8503();
			assert server != null;
			
			class_243 returnPos = ((PlayerWithReturn) player).worldinajar$getReturnPos();
			class_5321<class_1937> returnDim = ((PlayerWithReturn) player).worldinajar$getReturnDimension();
			
			Main.removeFakePlayer((class_3218) level, (class_3222) player);
			
			class_3218 returnLevel = server.method_3847(returnDim);
			class_5454 transition = new class_5454(returnLevel, returnPos, class_243.field_1353, 0.0f, 0.0f, class_5454.field_52245);
			player.method_5731(transition);
			return class_1269.field_5812;
		}
	}
}
