/**
 * World In a Jar
 * Copyright (C) 2024  VulpixelMC
 * <p>
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * <p>
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package gay.sylv.wij.impl.datagen;

import gay.sylv.wij.api.block.BarkType;
import gay.sylv.wij.api.datagen.RuntimeResourcePack;
import gay.sylv.wij.impl.item.BarkItem;
import gay.sylv.wij.impl.item.Items;
import gay.sylv.wij.impl.util.Constants;
import gay.sylv.wij.impl.util.Initializable;
import gay.sylv.wij.impl.util.Pair;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1011;
import net.minecraft.class_1792;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3262;
import net.minecraft.class_3264;
import net.minecraft.class_5253;
import net.minecraft.class_7367;
import net.minecraft.class_7923;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;

import static gay.sylv.wij.impl.datagen.RuntimeResourcePackImpl.generatedTag;
import static gay.sylv.wij.impl.util.Constants.modId;

public final class DynamicDataGenerator implements Initializable {
	public static final DynamicDataGenerator INSTANCE = new DynamicDataGenerator();
	
	private DynamicDataGenerator() {}
	
	@Override
	public void initialize() {
		BarkType.registerAll(BarkType.class);
	}
	
	public static final class ItemGenerator {
		private ItemGenerator() {}
		
		public static void registerBark(BarkType type) {
			class_1792 barkItem = class_2378.method_10230(
					class_7923.field_41178,
					type.getIdentifier(),
					new BarkItem(
							new class_1792.class_1793(),
							type
					)
			);
			Items.BARK.put(type, barkItem);
		}
		
		public static void generateBarkTag(BarkType type) {
			String generatedTag = generatedTag(List.of(type.getIdentifier()), false);
			RuntimeResourcePack.getInstance().addItemTag(modId("bark"), generatedTag);
		}
		
		public static void generateModel(BarkType type) {
			String model;
			if (type == BarkType.SPRUCE) {
				model = String.format(
						"""
						{
							"parent": "item/generated",
							"textures": {
								"layer0": "%1$s"
							},
							"overrides": [
						 		{
						 			"predicate": {
						 				"custom_model_data": 1
						 			},
						 			"model": "minecraft:item/cooked_beef"
						 		}
						 	]
						}
						""", type.getResourceIdentifier("item"));
			} else {
				model = generatedModel(type.getResourceIdentifier("item"));
			}
			RuntimeResourcePack.getInstance().addModel(type.getResourceIdentifier("item"), model);
		}
		
		private static String generatedModel(class_2960 id) {
			return String.format("""
              {
              	"parent": "item/generated",
              	"textures": {
              		"layer0": "%1$s"
              	}
              }
              """, id.toString());
		}
	}
	
	@Environment(EnvType.CLIENT)
	public static final class TextureGenerator {
		private TextureGenerator() {}
		
		public static void generate(RuntimeResourcePack rrp, List<class_3262> resources) {
			class_1011 barkMask = getTexture(rrp, "bark.png");
			
			Map<class_2960, class_7367<InputStream>> data = new HashMap<>();
			
			BarkType.getTypes().stream()
					.map(type -> {
						resources
								.forEach(packResources -> {
									packResources.method_14408(
											class_3264.field_14188,
											type.toFilePath().method_12836(),
											type.toFilePath().method_12832(),
											data::put
									);
									packResources.method_14408(
											class_3264.field_14188,
											type.toFilePath().method_12836(),
											type.toFilePath().method_48331(".mcmeta").method_12832(),
											data::put
									);
								});
						
						class_1011 log;
						try {
							log = class_1011.method_4309(data.get(type.toFilePath()).get());
						} catch (IOException e) {
							throw new RuntimeException(e);
						}
						class_1011 darkenedLog = multiplyBrightness(log, 0.85f);
						
						if (type.isAnimated()) {
							try (var inputStream = data.get(type.toFilePath().method_48331(".mcmeta")).get()) {
								rrp.addItemMcmeta(type.getIdentifier(), new String(inputStream.readAllBytes(), StandardCharsets.UTF_8));
							} catch (IOException e) {
								throw new RuntimeException(e);
							}
						}
						
						class_1011 expandedMask = expand(barkMask, log.method_4307(), log.method_4323());
						int shadowMask = class_5253.class_8045.method_48344(255, 0, 0, 255);
						maskImage(expandedMask, log);
						maskImage(log, darkenedLog, shadowMask);
						
						return new Pair<>(darkenedLog, type);
					})
					.forEach((pair) -> {
						class_1011 barkImage = pair.first();
						BarkType type = pair.second();
						rrp.addItemTexture(type.getIdentifier(), barkImage);
					});
		}
		
		private static @NotNull class_1011 getTexture(RuntimeResourcePack rrp, String texture) {
			class_7367<InputStream> inputStream = Objects.requireNonNull(rrp.method_14410(texture));
			class_1011 image;
			try {
				image = class_1011.method_4309(inputStream.get());
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
			
			if (image.method_4318() != class_1011.class_1012.field_4997) {
				throwError(texture);
			}
			return image;
		}
		
		private static void throwError(String maskName) {
			throw new RuntimeException("World In a Jar's runtime resource generation has failed due to an incorrect image format! Please check ensure that `src/main/resources/rrp/" + maskName + "` is in RGBA format.");
		}
		
		public static class_1011 expand(class_1011 mask, int width, int height) {
			if (width == mask.method_4307() && height == mask.method_4323()) {
				return mask;
			}
			
			class_1011 newMask = new class_1011(mask.method_4318(), width, height, false);
			int[] pixels = mask.method_48463();
			AtomicInteger index = new AtomicInteger();
			newMask.method_51596(pixel -> {
				int i = index.getAndIncrement();
				if (i + 1 > mask.method_4307() * mask.method_4323()) {
					index.set(1);
					i = 0;
				}
				
				return pixels[i];
			});
			
			return newMask;
		}
		
		public static class_1011 multiplyBrightness(class_1011 texture, float multiply) {
			return texture.method_48462(pixel -> {
				int red = class_5253.class_8045.method_48345(pixel);
				int green = class_5253.class_8045.method_48346(pixel);
				int blue = class_5253.class_8045.method_48347(pixel);
				int alpha = class_5253.class_8045.method_48342(pixel);
				return class_5253.class_8045.method_48344(alpha, (int) (blue * multiply), (int) (green * multiply), (int) (red * multiply));
			});
		}
		
		public static void maskImage(class_1011 mask, class_1011 texture) {
			maskImage(mask, texture, Constants.TEXTURE_MASK);
		}
		
		public static void maskImage(class_1011 mask, class_1011 texture, int maskColor) {
			int maskRed = class_5253.class_8045.method_48345(maskColor);
			int maskGreen = class_5253.class_8045.method_48346(maskColor);
			int maskBlue = class_5253.class_8045.method_48347(maskColor);
			
			int[] pixels = mask.method_48463();
			AtomicInteger index = new AtomicInteger();
			texture.method_51596(pixel -> {
				int i = index.getAndIncrement();
				
				int newRed = class_5253.class_8045.method_48345(pixel);
				int newGreen = class_5253.class_8045.method_48346(pixel);
				int newBlue = class_5253.class_8045.method_48347(pixel);
				
				int color = pixels[i];
				int red = class_5253.class_8045.method_48345(color);
				int green = class_5253.class_8045.method_48346(color);
				int blue = class_5253.class_8045.method_48347(color);
				int alpha = class_5253.class_8045.method_48342(color);
				
				if (red == maskRed && green == maskGreen && blue == maskBlue) {
					return class_5253.class_8045.method_48344(alpha, newBlue, newGreen, newRed);
				} else {
					return color;
				}
			});
		}
	}
}
