/*
 * Decompiled with CFR 0.152.
 */
package dev.vankka.dependencydownload;

import dev.vankka.dependencydownload.classpath.ClasspathAppender;
import dev.vankka.dependencydownload.common.util.HashUtil;
import dev.vankka.dependencydownload.dependency.Dependency;
import dev.vankka.dependencydownload.path.CleanupPathProvider;
import dev.vankka.dependencydownload.path.DependencyPathProvider;
import dev.vankka.dependencydownload.path.DirectoryDependencyPathProvider;
import dev.vankka.dependencydownload.relocation.Relocation;
import dev.vankka.dependencydownload.repository.Repository;
import dev.vankka.dependencydownload.resource.DependencyDownloadResource;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class DependencyManager {
    private final DependencyPathProvider dependencyPathProvider;
    private final List<Dependency> dependencies = new CopyOnWriteArrayList<Dependency>();
    private final Set<Relocation> relocations = new CopyOnWriteArraySet<Relocation>();
    private final AtomicInteger step = new AtomicInteger(0);

    public DependencyManager(@NotNull Path cacheDirectory) {
        this(new DirectoryDependencyPathProvider(cacheDirectory));
    }

    public DependencyManager(@NotNull DependencyPathProvider dependencyPathProvider) {
        this.dependencyPathProvider = dependencyPathProvider;
    }

    public void addDependency(@NotNull Dependency dependency) {
        this.addDependencies(Collections.singleton(dependency));
    }

    public void addDependencies(@NotNull Collection<Dependency> dependencies) {
        if (this.step.get() > 0) {
            throw new IllegalStateException("Cannot add dependencies after downloading");
        }
        this.dependencies.addAll(dependencies);
    }

    @NotNull
    public List<Dependency> getDependencies() {
        return Collections.unmodifiableList(this.dependencies);
    }

    public void addRelocation(@NotNull Relocation relocation) {
        this.addRelocations(Collections.singleton(relocation));
    }

    public void addRelocations(@NotNull Collection<Relocation> relocations) {
        if (this.step.get() > 2) {
            throw new IllegalStateException("Cannot add relocations after relocating");
        }
        this.relocations.addAll(relocations);
    }

    @NotNull
    public Set<Relocation> getRelocations() {
        return Collections.unmodifiableSet(this.relocations);
    }

    @NotNull
    public DependencyPathProvider getDependencyPathProvider() {
        return this.dependencyPathProvider;
    }

    public void loadFromResource(@NotNull URL resourceURL) throws IOException {
        DependencyDownloadResource resource = new DependencyDownloadResource(resourceURL);
        this.loadFromResource(resource);
    }

    public void loadFromResource(@NotNull String fileContents) {
        DependencyDownloadResource resource = new DependencyDownloadResource(fileContents);
        this.loadFromResource(resource);
    }

    public void loadFromResource(@NotNull List<String> fileLines) {
        DependencyDownloadResource resource = new DependencyDownloadResource(fileLines);
        this.loadFromResource(resource);
    }

    public void loadFromResource(@NotNull DependencyDownloadResource resource) {
        this.dependencies.addAll(resource.getDependencies());
        this.relocations.addAll(resource.getRelocations());
    }

    public CompletableFuture<Void> downloadAll(@Nullable Executor executor, @NotNull List<Repository> repositories) {
        return CompletableFuture.allOf(this.download(executor, repositories));
    }

    public CompletableFuture<Void>[] download(@Nullable Executor executor, @NotNull List<Repository> repositories) {
        if (!this.step.compareAndSet(0, 1)) {
            throw new IllegalStateException("Download has already been executed");
        }
        return this.forEachDependency(executor, dependency -> this.downloadDependency((Dependency)dependency, repositories), (dependency, cause) -> new RuntimeException("Failed to download dependency " + dependency.getMavenArtifact(), (Throwable)cause));
    }

    public CompletableFuture<Void> relocateAll(@Nullable Executor executor) {
        return CompletableFuture.allOf(this.relocate(executor, this.getClass().getClassLoader()));
    }

    public CompletableFuture<Void> relocateAll(@Nullable Executor executor, @Nullable ClassLoader jarRelocatorLoader) {
        return CompletableFuture.allOf(this.relocate(executor, jarRelocatorLoader));
    }

    public CompletableFuture<Void>[] relocate(@Nullable Executor executor) {
        return this.relocate(executor, this.getClass().getClassLoader());
    }

    public CompletableFuture<Void>[] relocate(@Nullable Executor executor, @Nullable ClassLoader jarRelocatorLoader) {
        int currentStep = this.step.get();
        if (currentStep == 0) {
            throw new IllegalArgumentException("Download hasn't been executed");
        }
        if (currentStep != 1) {
            throw new IllegalArgumentException("Relocate has already been executed");
        }
        this.step.set(2);
        JarRelocatorHelper helper = new JarRelocatorHelper(jarRelocatorLoader != null ? jarRelocatorLoader : this.getClass().getClassLoader());
        return this.forEachDependency(executor, dependency -> this.relocateDependency((Dependency)dependency, helper), (dependency, cause) -> new RuntimeException("Failed to relocate dependency " + dependency.getMavenArtifact(), (Throwable)cause));
    }

    public CompletableFuture<Void> loadAll(@Nullable Executor executor, @NotNull ClasspathAppender classpathAppender) {
        return CompletableFuture.allOf(this.load(executor, classpathAppender));
    }

    public CompletableFuture<Void>[] load(@Nullable Executor executor, @NotNull ClasspathAppender classpathAppender) {
        int currentStep = this.step.get();
        if (currentStep == 0) {
            throw new IllegalArgumentException("Download hasn't been executed");
        }
        this.step.set(3);
        return this.forEachDependency(executor, dependency -> this.loadDependency((Dependency)dependency, classpathAppender, currentStep == 2), (dependency, cause) -> new RuntimeException("Failed to load dependency " + dependency.getMavenArtifact(), (Throwable)cause));
    }

    @NotNull
    public Path getPathForDependency(@NotNull Dependency dependency, boolean relocated) {
        return this.getDependencyPathProvider().getDependencyPath(dependency, relocated);
    }

    @NotNull
    public Set<Path> getAllPaths(boolean includeRelocated) {
        HashSet<Path> paths = new HashSet<Path>();
        for (Dependency dependency : this.dependencies) {
            paths.add(this.getPathForDependency(dependency, false));
            if (!includeRelocated) continue;
            paths.add(this.getPathForDependency(dependency, true));
        }
        return paths;
    }

    public void cleanupCacheDirectory() throws IOException, IllegalStateException {
        Set filesToDelete;
        if (!(this.dependencyPathProvider instanceof CleanupPathProvider)) {
            throw new IllegalStateException("Cache directory cleanup is only available when dependencyPathProvider is a instance of CleanupPathProvider");
        }
        Path cacheDirectory = ((CleanupPathProvider)this.dependencyPathProvider).getCleanupPath();
        Set<Path> paths = this.getAllPaths(true);
        try (Stream<Path> stream = Files.list(cacheDirectory);){
            filesToDelete = stream.filter(path -> !Files.isDirectory(path, new LinkOption[0])).filter(path -> !paths.contains(path)).collect(Collectors.toSet());
        }
        for (Path path2 : filesToDelete) {
            Files.delete(path2);
        }
    }

    private CompletableFuture<Void>[] forEachDependency(Executor executor, ExceptionalConsumer<Dependency> runnable, BiFunction<Dependency, Throwable, Throwable> dependencyException) {
        int size = this.dependencies.size();
        CompletableFuture[] futures = new CompletableFuture[size];
        for (int index = 0; index < size; ++index) {
            Dependency dependency = this.dependencies.get(index);
            CompletableFuture future = new CompletableFuture();
            Runnable run = () -> {
                try {
                    runnable.run(dependency);
                    future.complete(null);
                }
                catch (Throwable t) {
                    future.completeExceptionally((Throwable)dependencyException.apply(dependency, t));
                }
            };
            if (executor != null) {
                executor.execute(run);
            } else {
                run.run();
            }
            futures[index] = future;
            if (future.isCompletedExceptionally()) break;
        }
        return futures;
    }

    private void downloadDependency(Dependency dependency, List<Repository> repositories) throws IOException, NoSuchAlgorithmException {
        if (repositories.isEmpty()) {
            throw new RuntimeException("No repositories provided");
        }
        Path dependencyPath = this.getPathForDependency(dependency, false);
        if (!Files.exists(dependencyPath.getParent(), new LinkOption[0])) {
            Files.createDirectories(dependencyPath.getParent(), new FileAttribute[0]);
        }
        if (Files.exists(dependencyPath, new LinkOption[0])) {
            String fileHash = HashUtil.getFileHash(dependencyPath.toFile(), dependency.getHashingAlgorithm());
            if (fileHash.equals(dependency.getHash())) {
                return;
            }
            Files.delete(dependencyPath);
        }
        Files.createFile(dependencyPath, new FileAttribute[0]);
        RuntimeException failure = new RuntimeException("All provided repositories failed to download dependency");
        boolean anyFailures = false;
        for (Repository repository : repositories) {
            try {
                MessageDigest digest = MessageDigest.getInstance(dependency.getHashingAlgorithm());
                this.downloadFromRepository(dependency, repository, dependencyPath, digest);
                String hash = HashUtil.getHash(digest);
                String dependencyHash = dependency.getHash();
                if (!hash.equals(dependencyHash)) {
                    throw new RuntimeException("Failed to verify file hash: " + hash + " should've been: " + dependencyHash);
                }
                return;
            }
            catch (Throwable e) {
                Files.deleteIfExists(dependencyPath);
                failure.addSuppressed(e);
                anyFailures = true;
            }
        }
        if (!anyFailures) {
            throw new RuntimeException("Nothing failed yet nothing passed");
        }
        throw failure;
    }

    private void downloadFromRepository(Dependency dependency, Repository repository, Path dependencyPath, MessageDigest digest) throws Throwable {
        URLConnection connection = repository.openConnection(dependency);
        byte[] buffer = new byte[repository.getBufferSize()];
        try (BufferedInputStream inputStream = new BufferedInputStream(connection.getInputStream());
             BufferedOutputStream outputStream = new BufferedOutputStream(Files.newOutputStream(dependencyPath, new OpenOption[0]));){
            int total;
            while ((total = inputStream.read(buffer)) != -1) {
                outputStream.write(buffer, 0, total);
                digest.update(buffer, 0, total);
            }
        }
    }

    private void relocateDependency(Dependency dependency, JarRelocatorHelper helper) {
        Path dependencyFile = this.getPathForDependency(dependency, false);
        Path relocatedFile = this.getPathForDependency(dependency, true);
        try {
            helper.run(dependencyFile, relocatedFile, this.relocations);
        }
        catch (InvocationTargetException e) {
            throw new RuntimeException("Failed to run relocation", e.getCause());
        }
        catch (ReflectiveOperationException e) {
            throw new RuntimeException("Failed to initialize relocator", e);
        }
    }

    private void loadDependency(Dependency dependency, ClasspathAppender classpathAppender, boolean relocated) throws MalformedURLException {
        Path fileToLoad = relocated ? this.getPathForDependency(dependency, true) : this.getPathForDependency(dependency, false);
        classpathAppender.appendFileToClasspath(fileToLoad);
    }

    private static class JarRelocatorHelper {
        private final Constructor<?> relocatorConstructor;
        private final Method relocatorRunMethod;
        private final Constructor<?> relocationConstructor;

        public JarRelocatorHelper(ClassLoader classLoader) {
            try {
                Class<?> relocatorClass = classLoader.loadClass("forcepack.libs.relocator.JarRelocator");
                this.relocatorConstructor = relocatorClass.getConstructor(File.class, File.class, Collection.class);
                this.relocatorRunMethod = relocatorClass.getMethod("run", new Class[0]);
                Class<?> relocationClass = classLoader.loadClass("forcepack.libs.relocator.Relocation");
                this.relocationConstructor = relocationClass.getConstructor(String.class, String.class, Collection.class, Collection.class);
            }
            catch (ClassNotFoundException | NoSuchMethodException e) {
                throw new RuntimeException("Failed to load jar-relocator from the provided ClassLoader", e);
            }
        }

        public void run(Path from, Path to, Set<Relocation> relocations) throws ReflectiveOperationException {
            HashSet mappedRelocations = new HashSet();
            for (Relocation relocation : relocations) {
                Object mapped = this.relocationConstructor.newInstance(relocation.getPattern(), relocation.getShadedPattern(), relocation.getIncludes(), relocation.getExcludes());
                mappedRelocations.add(mapped);
            }
            Object relocator = this.relocatorConstructor.newInstance(from.toFile(), to.toFile(), mappedRelocations);
            this.relocatorRunMethod.invoke(relocator, new Object[0]);
        }
    }

    @FunctionalInterface
    private static interface ExceptionalConsumer<T> {
        public void run(T var1) throws Throwable;
    }
}

