package io.github.dennisochulor.tickrate.api;

import io.github.dennisochulor.tickrate.api_impl.TickRateAPIImpl;
import java.util.Collection;
import net.minecraft.class_1297;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_3194;

/**
 * API v1 for TickRate v0.4.0 <p>
 * This class represents the sole entrypoint for TickRate's API. This API should only be used on the logical server. <p>
 *
 * @see TickRateEvents
 */
public interface TickRateAPI {

    /**
     * Obtain the API instance. This method should only be called after the server has been fully initialised. <p>
     * In singleplayer, a new API instance is used each time a new save is loaded.
     * @throws IllegalStateException if the server has not been fully intialised yet.
     */
    static TickRateAPI getInstance() {
        return TickRateAPIImpl.getInstance();
    }

    /**
     * Returns the server's tick rate.
     */
    float queryServer();

    /**
     * Set the tick rate of the server. <code>rate</code> is rounded using {@link Math#round(float)}
     *
     * @throws IllegalArgumentException if <code>rate</code> is less than 1.
     */
    void rateServer(float rate);

    /**
     * Freezes or unfreezes the server depending on <code>freeze</code>.
     *
     * @param freeze <code>true</code> to freeze, <code>false</code> to unfreeze.
     */
    void freezeServer(boolean freeze);

    /**
     * Steps the server. The server will step according to its current TPS. <p>
     * If <code>stepTicks</code> is 0, the server will stop stepping.
     *
     * @throws IllegalArgumentException if <code>stepTicks</code> is less than 0.
     * @throws IllegalStateException if the server is not frozen.
     */
    void stepServer(int stepTicks);

    /**
     * Sprints the server. <p>
     * If <code>sprintTicks</code> is 0, the server will stop sprinting.
     *
     * @throws IllegalArgumentException if <code>sprintTicks</code> is less than 0.
     */
    void sprintServer(int sprintTicks);



    /**
     * Returns the tick rate of the entity. <p>
     * If the entity has no specific tick rate, the tick rate of the chunk it is in will be returned.
     * If the chunk it is in also has no specific tick rate, the server's tick rate will be returned. <p>
     *
     * If the entity is a passenger, the tick rate of its {@link class_1297#method_5668() root vehicle} will be returned.
     * If the server is stepping, the server's tick rate is returned.
     *
     * @throws IllegalArgumentException if {@link class_1297#method_31481()} is true.
     */
    float queryEntity(class_1297 entity);

    /**
     * Sets the tick rate of the specified entities. <code>rate</code> is rounded using {@link Math#round(float)} <p>
     * If <code>rate</code> is exactly <code>0.0f</code>, then the entities' tick rate will be reset.
     *
     * @throws IllegalArgumentException if {@link class_1297#method_31481()} is true for any of the specified entities OR if <code>rate</code> is less than 1 and not 0.
     */
    void rateEntity(Collection<? extends class_1297> entities, float rate);

    /**
     * Sets the tick rate of the specified entity. <code>rate</code> is rounded using {@link Math#round(float)} <p>
     * If <code>rate</code> is exactly <code>0.0f</code>, then the entity's tick rate will be reset.
     *
     * @throws IllegalArgumentException if {@link class_1297#method_31481()} is true OR if <code>rate</code> is less than 1 and not 0.
     */
    void rateEntity(class_1297 entity, float rate);

    /**
     * Freezes or unfreezes the specified entities depending on <code>freeze</code>.
     *
     * @param freeze <code>true</code> to freeze, <code>false</code> to unfreeze
     * @throws IllegalArgumentException if {@link class_1297#method_31481()} is true for any of the specified entities.
     */
    void freezeEntity(Collection<? extends class_1297> entities, boolean freeze);

    /**
     * Freezes or unfreezes the specified entity depending on <code>freeze</code>.
     *
     * @param freeze <code>true</code> to freeze, <code>false</code> to unfreeze
     * @throws IllegalArgumentException if {@link class_1297#method_31481()} is true the specified entity.
     */
    void freezeEntity(class_1297 entity, boolean freeze);

    /**
     * Steps the specified entities for <code>stepTicks</code> ticks. The entities will step according to their current TPS. <p>
     * If <code>stepTicks</code> is 0, the entities will stop stepping.
     *
     * @throws IllegalArgumentException if any of the following conditions are met: <p>
     * <ul>
     *          <li> {@link class_1297#method_31481()} is true for any of the specified entities.
     *          <li> Any of the specified entities are currently NOT frozen or ARE sprinting.
     *          <li> <code>stepTicks</code> is less than 0.
     * </ul>
     */
    void stepEntity(Collection<? extends class_1297> entities, int stepTicks);

    /**
     * Steps the specified entity for <code>stepTicks</code> ticks. The entity will step according to its current TPS. <p>
     * If <code>stepTicks</code> is 0, the entitiy will stop stepping.
     *
     * @throws IllegalArgumentException if any of the following conditions are met: <p>
     * <ul>
     *          <li> {@link class_1297#method_31481()} is true.
     *          <li> The specified entity is currently NOT frozen or IS sprinting.
     *          <li> <code>stepTicks</code> is less than 0.
     * </ul>
     */
    void stepEntity(class_1297 entity, int stepTicks);

    /**
     * Sprints the specified entities for <code>sprintTicks</code> ticks. <p>
     * If <code>sprintTicks</code> is 0, the entities will stop sprinting.
     *
     * @throws IllegalArgumentException if any of the following conditions are met: <p>
     * <ul>
     *          <li> {@link class_1297#method_31481()} is true for any of the specified entities.
     *          <li> Any of the specified entities are currently stepping.
     *          <li> <code>sprintTicks</code> is less than 0.
     * </ul>
     */
    void sprintEntity(Collection<? extends class_1297> entities, int sprintTicks);

    /**
     * Sprints the specified entity for <code>sprintTicks</code> ticks. <p>
     * If <code>sprintTicks</code> is 0, the entity will stop sprinting.
     *
     * @throws IllegalArgumentException if any of the following conditions are met: <p>
     * <ul>
     *          <li> {@link class_1297#method_31481()} is true.
     *          <li> The specified entity is currently stepping.
     *          <li> <code>sprintTicks</code> is less than 0.
     * </ul>
     */
    void sprintEntity(class_1297 entity, int sprintTicks);



    /**
     * Returns the tick rate of the chunk. <p>
     * If the chunk has no specific tick rate, the server's tick rate will be returned.
     * <p>
     * If the server is stepping, the server's tick rate is returned.
     *
     * @throws IllegalArgumentException if the chunk is {@link class_3194#field_19334 INACCESSIBLE} (not loaded).
     */
    float queryChunk(class_1937 world, class_1923 chunk);

    /**
     * Sets the tick rate of the specified chunks. <code>rate</code> is rounded using {@link Math#round(float)} <p>
     * If <code>rate</code> is exactly <code>0.0f</code>, then the chunks' tick rate will be reset.
     *
     * @throws IllegalArgumentException if any of the chunks are {@link class_3194#field_19334 INACCESSIBLE} (not loaded) OR if <code>rate</code> is less than 1 and not 0.
     */
    void rateChunk(class_1937 world, Collection<class_1923> chunks, float rate);

    /**
     * Sets the tick rate of the specified chunk. <code>rate</code> is rounded using {@link Math#round(float)} <p>
     * If <code>rate</code> is exactly <code>0.0f</code>, then the chunk's tick rate will be reset.
     *
     * @throws IllegalArgumentException if the chunk is {@link class_3194#field_19334 INACCESSIBLE} (not loaded) OR if <code>rate</code> is less than 1 and not 0.
     */
    void rateChunk(class_1937 world, class_1923 chunk, float rate);

    /**
     * Freezes or unfreezes the specified chunks depending on <code>freeze</code>.
     *
     * @param freeze <code>true</code> to freeze, <code>false</code> to unfreeze
     * @throws IllegalArgumentException if any of the chunks are {@link class_3194#field_19334 INACCESSIBLE} (not loaded).
     */
    void freezeChunk(class_1937 world, Collection<class_1923> chunks, boolean freeze);

    /**
     * Freezes or unfreezes the specified chunk depending on <code>freeze</code>.
     *
     * @param freeze <code>true</code> to freeze, <code>false</code> to unfreeze
     * @throws IllegalArgumentException if the chunk is {@link class_3194#field_19334 INACCESSIBLE} (not loaded).
     */
    void freezeChunk(class_1937 world, class_1923 chunk, boolean freeze);

    /**
     * Steps the specified chunks for <code>stepTicks</code> ticks. The chunks will step according to their current TPS. <p>
     * If <code>stepTicks</code> is 0, the chunks will stop stepping.
     *
     * @throws IllegalArgumentException if any of the following conditions are met: <p>
     * <ul>
     *          <li> Any of the chunks are {@link class_3194#field_19334 INACCESSIBLE} (not loaded).
     *          <li> Any of the specified chunks are currently NOT frozen or ARE sprinting.
     *          <li> <code>stepTicks</code> is less than 0.
     * </ul>
     */
    void stepChunk(class_1937 world, Collection<class_1923> chunks, int stepTicks);

    /**
     * Steps the specified chunk for <code>stepTicks</code> ticks. The chunk will step according to its current TPS. <p>
     * If <code>stepTicks</code> is 0, the chunk will stop stepping.
     *
     * @throws IllegalArgumentException if any of the following conditions are met: <p>
     * <ul>
     *          <li> The chunk is {@link class_3194#field_19334 INACCESSIBLE} (not loaded).
     *          <li> The specified chunk is currently NOT frozen or IS sprinting.
     *          <li> <code>stepTicks</code> is less than 0.
     * </ul>
     */
    void stepChunk(class_1937 world, class_1923 chunk, int stepTicks);

    /**
     * Sprints the specified chunks for <code>sprintTicks</code> ticks. <p>
     * If <code>sprintTicks</code> is 0, the chunks will stop sprinting.
     *
     * @throws IllegalArgumentException if any of the following conditions are met: <p>
     * <ul>
     *          <li> Any of the chunks are {@link class_3194#field_19334 INACCESSIBLE} (not loaded).
     *          <li> Any of the specified chunks are currently stepping.
     *          <li> <code>sprintTicks</code> is less than 0.
     * </ul>
     */
    void sprintChunk(class_1937 world, Collection<class_1923> chunks, int sprintTicks);

    /**
     * Sprints the specified chunk for <code>sprintTicks</code> ticks. <p>
     * If <code>sprintTicks</code> is 0, the chunk will stop sprinting.
     *
     * @throws IllegalArgumentException if any of the following conditions are met: <p>
     * <ul>
     *          <li> The chunk is {@link class_3194#field_19334 INACCESSIBLE} (not loaded).
     *          <li> The specified chunk is currently stepping.
     *          <li> <code>sprintTicks</code> is less than 0.
     * </ul>
     */
    void sprintChunk(class_1937 world, class_1923 chunk, int sprintTicks);

}
