package io.github.dennisochulor.tickrate.api_impl;

import io.github.dennisochulor.tickrate.api.TickRateAPI;
import io.github.dennisochulor.tickrate.api.TickRateEvents;
import net.minecraft.class_1297;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_2806;
import net.minecraft.class_2818;
import net.minecraft.class_3194;
import net.minecraft.class_3222;
import net.minecraft.class_8915;
import net.minecraft.server.MinecraftServer;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

public final class TickRateAPIImpl implements TickRateAPI {

    private static TickRateAPI INSTANCE;

    public static TickRateAPI getInstance() {
        if(INSTANCE == null) throw new IllegalStateException("The MinecraftServer must be fully initialised first before using the TickRateAPI!");
        return INSTANCE;
    }

    /** Only ever called when the logical server is fully initialised */
    public static void init(MinecraftServer server) {
        if(INSTANCE != null) throw new IllegalStateException("Only one instance can be present at any given time!");
        INSTANCE = new TickRateAPIImpl(server);
    }

    /** Only ever called when the logical server starts shutting down */
    public static void uninit() {
        INSTANCE = null;
    }


    private final MinecraftServer server;
    private final class_8915 tickManager;

    private TickRateAPIImpl(MinecraftServer server) {
        this.server = server;
        this.tickManager = this.server.method_54833();
    }


    private void entityCheck(Collection<? extends class_1297> entities) throws IllegalArgumentException {
        Objects.requireNonNull(entities, "entities cannot be null!");
        entities.forEach(entity -> {
            if(entity instanceof class_3222 player && !tickManager.tickRate$hasClientMod(player))
                throw new IllegalArgumentException("Some of the specified entities are players that do not have TickRate mod installed on their client, so their tick rate cannot be manipulated.");
            if(entity.method_31481()) throw new IllegalArgumentException("Entity must not be removed!");
        });
    }

    private void chunkCheck(class_1937 world, Collection<class_1923> chunks) throws IllegalArgumentException {
        Objects.requireNonNull(world, "world cannot be null!");
        Objects.requireNonNull(chunks, "chunks cannot be null!");
        chunks.forEach(chunkPos -> {
            class_2818 worldChunk = (class_2818) world.method_8402(chunkPos.field_9181,chunkPos.field_9180,class_2806.field_12803,false);
            if(worldChunk==null || worldChunk.method_12225() == class_3194.field_19334)
                throw new IllegalArgumentException("Some of the specified chunks are not loaded!");
        });
    }

    private void send(Runnable runnable) {
        server.method_63588(server.method_16209(runnable));
    }


    // API Implementation

    @Override
    public float queryServer() {
        return tickManager.tickRate$getServerRate();
    }

    @Override
    public void rateServer(float rate) {
        if(rate < 1) throw new IllegalArgumentException("rate must be >= 1");
        int roundRate = Math.round(rate);
        tickManager.tickRate$setServerRate(roundRate);
        tickManager.tickRate$sendUpdatePacket();
        send(() -> TickRateEvents.SERVER_RATE.invoker().onServerRate(roundRate));
    }

    @Override
    public void freezeServer(boolean freeze) {
        if(freeze) {
            if(tickManager.tickRate$isServerSprint()) tickManager.method_54678();
            if(tickManager.method_54752()) tickManager.method_54676();
        }
        tickManager.method_54675(freeze);
        tickManager.tickRate$sendUpdatePacket();
        send(() -> TickRateEvents.SERVER_FREEZE.invoker().onServerFreeze(freeze));
    }

    @Override
    public void stepServer(int stepTicks) {
        if(stepTicks < 0) throw new IllegalArgumentException("stepTicks must be >= 0");

        if(stepTicks == 0) tickManager.method_54676();
        else {
            if(!tickManager.method_54672(stepTicks)) throw new IllegalStateException("server must be frozen to step!");
            send(() -> TickRateEvents.SERVER_STEP.invoker().onServerStep(stepTicks));
        }

        tickManager.tickRate$sendUpdatePacket();
    }

    @Override
    public void sprintServer(int sprintTicks) {
        if(sprintTicks < 0) throw new IllegalArgumentException("sprintTicks must be >= 0");

        if(sprintTicks == 0) tickManager.method_54678();
        else {
            tickManager.method_54677(sprintTicks);
            send(() -> TickRateEvents.SERVER_SPRINT.invoker().onServerSprint(sprintTicks));
        }

        tickManager.tickRate$sendUpdatePacket();
    }



    @Override
    public float queryEntity(class_1297 entity) {
        entityCheck(List.of(entity));
        return tickManager.tickRate$getEntityRate(entity);
    }

    @Override
    public void rateEntity(Collection<? extends class_1297> entities, float rate) {
        if(rate < 1.0f && rate != 0.0f) throw new IllegalArgumentException("rate must be >= 1 or exactly 0");
        entityCheck(entities);

        int roundRate = Math.round(rate);
        tickManager.tickRate$setEntityRate(roundRate, entities);
        tickManager.tickRate$sendUpdatePacket();
        send(() -> entities.forEach(entity -> TickRateEvents.ENTITY_RATE.invoker().onEntityRate(entity, roundRate)));
    }

    @Override
    public void rateEntity(class_1297 entity, float rate) {
        rateEntity(List.of(entity), rate);
    }

    @Override
    public void freezeEntity(Collection<? extends class_1297> entities, boolean freeze) {
        entityCheck(entities);

        tickManager.tickRate$setEntityFrozen(freeze, entities);
        tickManager.tickRate$sendUpdatePacket();
        send(() -> entities.forEach(entity -> TickRateEvents.ENTITY_FREEZE.invoker().onEntityFreeze(entity, freeze)));
    }

    @Override
    public void freezeEntity(class_1297 entity, boolean freeze) {
        freezeEntity(List.of(entity), freeze);
    }

    @Override
    public void stepEntity(Collection<? extends class_1297> entities, int stepTicks) {
        if(stepTicks < 0) throw new IllegalArgumentException("stepTicks must be >= 0");
        entityCheck(entities);

        if(tickManager.tickRate$stepEntity(stepTicks, entities)) {
            tickManager.tickRate$sendUpdatePacket();
            if(stepTicks != 0) send(() -> entities.forEach(entity -> TickRateEvents.ENTITY_STEP.invoker().onEntityStep(entity, stepTicks)));
        }
        else throw new IllegalArgumentException("All of the specified entities must be frozen first and cannot be sprinting!");
    }

    @Override
    public void stepEntity(class_1297 entity, int stepTicks) {
        stepEntity(List.of(entity), stepTicks);
    }

    @Override
    public void sprintEntity(Collection<? extends class_1297> entities, int sprintTicks) {
        if(sprintTicks < 0) throw new IllegalArgumentException("sprintTicks must be >= 0");
        entityCheck(entities);

        if(tickManager.tickRate$sprintEntity(sprintTicks, entities)) {
            tickManager.tickRate$sendUpdatePacket();
            if(sprintTicks != 0) send(() -> entities.forEach(entity -> TickRateEvents.ENTITY_SPRINT.invoker().onEntitySprint(entity, sprintTicks)));
        }
        else throw new IllegalArgumentException("All of the specified entities must not be stepping!");
    }

    @Override
    public void sprintEntity(class_1297 entity, int sprintTicks) {
        sprintEntity(List.of(entity), sprintTicks);
    }



    @Override
    public float queryChunk(class_1937 world, class_1923 chunk) {
        chunkCheck(world, List.of(chunk));
        return tickManager.tickRate$getChunkRate(world, chunk.method_8324());
    }

    @Override
    public void rateChunk(class_1937 world, Collection<class_1923> chunks, float rate) {
        if(rate < 1.0f && rate != 0.0f) throw new IllegalArgumentException("rate must be >= 1 or exactly 0");
        chunkCheck(world, chunks);

        int roundRate = Math.round(rate);
        tickManager.tickRate$setChunkRate(rate, world, chunks);
        tickManager.tickRate$sendUpdatePacket();
        send(() -> chunks.forEach(chunkPos -> TickRateEvents.CHUNK_RATE.invoker().onChunkRate(world, chunkPos, roundRate)));
    }

    @Override
    public void rateChunk(class_1937 world, class_1923 chunk, float rate) {
        rateChunk(world, List.of(chunk), rate);
    }

    @Override
    public void freezeChunk(class_1937 world, Collection<class_1923> chunks, boolean freeze) {
        chunkCheck(world, chunks);

        tickManager.tickRate$setChunkFrozen(freeze, world, chunks);
        tickManager.tickRate$sendUpdatePacket();
        send(() -> chunks.forEach(chunkPos -> TickRateEvents.CHUNK_FREEZE.invoker().onChunkFreeze(world, chunkPos, freeze)));
    }

    @Override
    public void freezeChunk(class_1937 world, class_1923 chunk, boolean freeze) {
        freezeChunk(world, List.of(chunk), freeze);
    }

    @Override
    public void stepChunk(class_1937 world, Collection<class_1923> chunks, int stepTicks) {
        if(stepTicks < 0) throw new IllegalArgumentException("stepTicks must be >= 0");
        chunkCheck(world, chunks);

        if(tickManager.tickRate$stepChunk(stepTicks, world, chunks)) {
            tickManager.tickRate$sendUpdatePacket();
            if(stepTicks != 0) send(() -> chunks.forEach(chunkPos -> TickRateEvents.CHUNK_STEP.invoker().onChunkStep(world, chunkPos, stepTicks)));
        }
        else throw new IllegalArgumentException("All of the specified chunks must be frozen first and cannot be sprinting!");
    }

    @Override
    public void stepChunk(class_1937 world, class_1923 chunk, int stepTicks) {
        stepChunk(world, List.of(chunk), stepTicks);
    }

    @Override
    public void sprintChunk(class_1937 world, Collection<class_1923> chunks, int sprintTicks) {
        if(sprintTicks < 0) throw new IllegalArgumentException("sprintTicks must be >= 0");
        chunkCheck(world, chunks);

        if(tickManager.tickRate$sprintChunk(sprintTicks, world, chunks)) {
            tickManager.tickRate$sendUpdatePacket();
            if(sprintTicks != 0) send(() -> chunks.forEach(chunkPos -> TickRateEvents.CHUNK_SPRINT.invoker().onChunkSprint(world, chunkPos, sprintTicks)));
        }
        else throw new IllegalArgumentException("All of the specified chunks must not be stepping!");
    }

    @Override
    public void sprintChunk(class_1937 world, class_1923 chunk, int sprintTicks) {
        sprintChunk(world, List.of(chunk), sprintTicks);
    }

}
