#version 330

layout(std140) uniform LightmapInfo {
    float AmbientLightFactor;
    float SkyFactor;
    float BlockFactor;
    float NightVisionFactor;
    float DarknessScale;
    float DarkenWorldFactor;
    float BrightnessFactor;
    vec3 SkyLightColor;
    vec3 AmbientColor;
} lightmapInfo;

in vec2 texCoord;

out vec4 fragColor;

float get_brightness(float level) {
    return level / (4.0 - 3.0 * level);
}

vec3 notGamma(vec3 x) {
    vec3 nx = 1.0 - x;
    return 1.0 - nx * nx * nx * nx;
}

void main() {
    // GUI-Detektion: keine Beleuchtung angefordert oder volle Helligkeit (15/16)
    bool isGuiPass = (lightmapInfo.BlockFactor == 0.0 && lightmapInfo.SkyFactor == 0.0)
                  || (texCoord.x >= 15.0/16.0 && texCoord.y >= 15.0/16.0);

    if (isGuiPass) {
        fragColor = vec4(1.0);
        return;
    }

    float quantized_block_level = floor(texCoord.x * 16.0) / 15.0;
    float quantized_sky_level   = floor(texCoord.y * 16.0) / 15.0;

    float block_brightness = get_brightness(quantized_block_level) * lightmapInfo.BlockFactor;

    float sky_brightness;
    if (lightmapInfo.BrightnessFactor >= 0.10) {
        sky_brightness = get_brightness(quantized_sky_level) * lightmapInfo.SkyFactor * 0.5;
    } else {
        sky_brightness = get_brightness(quantized_sky_level) * lightmapInfo.SkyFactor;
    }

    vec3 color;
    if (lightmapInfo.BrightnessFactor >= 0.10) {
        color = vec3(
            block_brightness / 3.0,
            block_brightness * ((block_brightness * 0.6 + 0.4) * 0.6 + 0.4) / 3.0,
            block_brightness * (block_brightness * block_brightness * 0.6 + 0.4) / 3.0
        );
    } else {
        color = vec3(
            block_brightness,
            block_brightness * ((block_brightness * 0.6 + 0.4) * 0.6 + 0.4),
            block_brightness * (block_brightness * block_brightness * 0.6 + 0.4)
        );
    }

    color = mix(color, lightmapInfo.AmbientColor, lightmapInfo.AmbientLightFactor);

    if (lightmapInfo.AmbientLightFactor > 0.0) {
        color = mix(color, vec3(0.99, 1.12, 1.0), 0.25);
        color = clamp(color, 0.0, 1.0);
    } else {
        color += lightmapInfo.SkyLightColor * sky_brightness;
        color = mix(color, vec3(0.75), 0.04);

        vec3 darkened_color = color * vec3(0.7, 0.6, 0.6);
        color = mix(color, darkened_color, lightmapInfo.DarkenWorldFactor);
    }

    if (lightmapInfo.NightVisionFactor > 0.0) {
        float max_component = max(color.r, max(color.g, color.b));
        if (max_component < 1.0) {
            vec3 bright_color = color / max_component;
            color = mix(color, bright_color, lightmapInfo.NightVisionFactor);
        }
    }

    if (lightmapInfo.AmbientLightFactor == 0.0) {
        color = color - vec3(lightmapInfo.DarknessScale);
    }

    float brightness;
    if (lightmapInfo.BrightnessFactor >= 0.80 && lightmapInfo.BrightnessFactor != 1.00) {
        brightness = 1.0;
    } else {
        if (lightmapInfo.BrightnessFactor >= 0.10) {
            brightness = 0.3;
        } else {
            brightness = 0.05;
        }
    }

    vec3 notGammaColor = notGamma(color);
    color = mix(color, notGammaColor, brightness * 10.0);
    color = mix(color, vec3(0.75), 0.04);

    color = clamp(color, 0.0, 1.0);
    fragColor = vec4(color, 1.0);
}
