package toni.lib.animation;

import F;
import I;
import io.netty.buffer.ByteBuf;
import lombok.Getter;
import lombok.Setter;
import net.minecraft.class_2540;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import toni.lib.utils.ColorUtils;
import toni.lib.animation.easing.EasingType;
import toni.lib.animation.effects.IAnimationEffect;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

public class AnimationTimeline {
    public final float duration;

    @Getter @Setter
    private float current;
    private AnimationKeyframe effected_keyframe = new AnimationKeyframe();
    private AnimationKeyframe keyframe = new AnimationKeyframe();

    private boolean hasSortedTransitions = false;
    private final HashMap<Binding, List<Transition>> transitions = new HashMap<>();
    private final HashMap<Binding, List<AnimationEffect>> effects = new HashMap<>();
    private final Binding[] bindings = Binding.values();

    public static AnimationTimeline builder(float duration) {
        return new AnimationTimeline(duration);
    }

    private AnimationTimeline(float duration) {
        this.duration = duration;
    }

    public void encode(class_2540 buffer) {
        buffer.method_52941(duration);
        buffer.method_52941(current);
        effected_keyframe.encode(buffer);
        keyframe.encode(buffer);

        buffer.method_34063(transitions, class_2540::method_10817, (buf, val) -> {
            buf.method_10804(val.size());
            val.forEach(i -> i.encode(buf));
        });

        buffer.method_34063(effects, class_2540::method_10817, (buf, val) -> {
            buf.method_10804(val.size());
            val.forEach(i -> i.encode(buf));
        });
    }

    public static AnimationTimeline decode(class_2540 buffer) {
        var duration = buffer.readFloat();
        var current = buffer.readFloat();
        var effected_keyframe = AnimationKeyframe.decode(buffer);
        var keyframe = AnimationKeyframe.decode(buffer);

        var transitions = buffer.method_34067(enumClass -> buffer.method_10818(Binding.class), (buf) -> {
            var size = buffer.method_10816();
            var list = new ArrayList<Transition>();
            for (int i = 0; i < size; i++) {
                list.add(Transition.decode(buf));
            }
            return list;
        });

        var effects = buffer.method_34067(enumClass -> buffer.method_10818(Binding.class), (buf) -> {
            var size = buffer.method_10816();
            var list = new ArrayList<AnimationEffect>();
            for (int i = 0; i < size; i++) {
                list.add(AnimationEffect.decode(buf));
            }
            return list;
        });

        var ths = new AnimationTimeline(duration);
        ths.current = current;
        ths.effected_keyframe = effected_keyframe;
        ths.keyframe = keyframe;
        ths.transitions.putAll(transitions);
        ths.effects.putAll(effects);

        return ths;
    }

    public AnimationKeyframe getKeyframe() {
        if (!hasSortedTransitions)
        {
            hasSortedTransitions = true;
            for (var kvp : transitions.entrySet())
                kvp.getValue().sort((a, b) -> Float.compare(a.getOut(), b.getOut()));
        }

        // loop over the list of transitions on the timeline
        for (var kvp : transitions.entrySet()) {
            // they should be sorted, so it's safe to just grab the first start value, and treat them all as if they're not overlapping
            float value = kvp.getValue().get(0).getStartValue();
            for (var transition : kvp.getValue()) {
                switch (transition.getPosition(current)) {
                    case DURING -> value = transition.eval(kvp.getKey(), current);
                    case AFTER -> value = transition.getEndValue();
                }
            }

            keyframe.setValue(kvp.getKey(), value);
        }

        for (var key : bindings) {
            if (!effects.containsKey(key)) {
                effected_keyframe.setValue(key, keyframe.getValue(key));
                continue;
            }

            for (var effect : effects.get(key)) {
                if (effect.getPosition(current) == IAnimationEffect.Position.DURING)
                    effected_keyframe.setValue(key, keyframe.getValue(key) + effect.type.calculate(effect, current));
            }
        }

        return effected_keyframe;
    }

    public AnimationKeyframe applyPose(class_332 context, float objectWidth, float objectHeight) {
        var key = getKeyframe();

        PoseUtils.applyScale(context, key.size);
        PoseUtils.applyPosition(context, key.size, objectWidth, objectHeight, key.posX, key.posY, key.posZ);
        PoseUtils.applyYRotation(context, key.size, objectWidth, objectHeight, key.rotY);
        PoseUtils.applyXRotation(context, key.size, objectWidth, objectHeight, key.rotX);
        PoseUtils.applyZRotation(context, key.size, objectWidth, objectHeight, key.rotZ);

        return key;
    }


    public int getColor() {
        var keyframe = getKeyframe();
        return ColorUtils.color(
                class_3532.method_15340((int) (keyframe.alpha * 255 + 5), 0, 255),
                ColorUtils.red((int) keyframe.color),
                ColorUtils.green((int) keyframe.color),
                ColorUtils.blue((int) keyframe.color));
    }



    public AnimationTimeline advancePlayhead(float delta) {
        current = Math.max(0, Math.min(current + delta, duration));
        return this;
    }

    public AnimationTimeline resetPlayhead(float newPosition) {
        current = Math.max(0, Math.min(newPosition, duration));
        return this;
    }

    public AnimationTimeline withColor(int alpha, int red, int green, int blue) {
        keyframe.color = ColorUtils.color(alpha, red, green, blue);
        return this;
    }

    public AnimationTimeline withXPosition(float x) {
        keyframe.posX = x;
        return this;
    }

    public AnimationTimeline withYPosition(float y) {
        keyframe.posY = y;
        return this;
    }

    public AnimationTimeline withZPosition(float z) {
        keyframe.posZ = z;
        return this;
    }

    public AnimationTimeline withXRotation(float x) {
        keyframe.rotX = x;
        return this;
    }

    public AnimationTimeline withYRotation(float y) {
        keyframe.rotY = y;
        return this;
    }

    public AnimationTimeline withZRotation(float z) {
        keyframe.rotZ = z;
        return this;
    }

    public AnimationTimeline withSize(float size) {
        keyframe.size = size;
        return this;
    }

    public AnimationTimeline transition(Binding binding, float in, float out, float start, float end, EasingType easing) {
        var transition = new Transition(in, out, easing, start, end);

        if (!transitions.containsKey(binding))
            transitions.put(binding, new ArrayList<>());

        transitions.get(binding).add(transition);
        return this;
    }

    public AnimationTimeline fadeout(float time) {
        return transition(Binding.Alpha, duration - time, duration, 1f, 0f, EasingType.EaseOutSine);
    }

    public AnimationTimeline fadeout(float time, EasingType easing) {
        return transition(Binding.Alpha, duration - time, duration, 1f, 0f, easing);
    }

    public AnimationTimeline fadein(float time) {
        return transition(Binding.Alpha, 0f, time, 0f, 1f, EasingType.EaseOutSine);
    }

    public AnimationTimeline fadein(float time, EasingType easing) {
        return transition(Binding.Alpha, 0f, time, 0f, 1f, easing);
    }


    public AnimationTimeline shake(float time) {
        transition(Binding.zRot, time, time + 0.1f, 0f, 5f, EasingType.EaseInBounce);
        transition(Binding.Size, time, time + 0.1f, 1f, 1.1f, EasingType.EaseInCubic);
        transition(Binding.Size, time + 0.1f, time + 0.2f, 1.f, 1f, EasingType.EaseOutCubic);
        return transition(Binding.zRot, time, time + 0.2f, 5f, 0, EasingType.EaseOutBounce);
    }

    public AnimationTimeline waveEffect(Binding binding, float intensity, float speed, float in, float out) {
        if (!effects.containsKey(binding))
            effects.put(binding, new ArrayList<>());

        effects.get(binding).add(new AnimationEffect(in, out, intensity, speed, AnimationEffect.Type.WAVE));
        return this;
    }

    public AnimationTimeline waveEffect(Binding binding, float intensity, float speed) {
        return waveEffect(binding, intensity, speed, 0f, duration);
    }

    public AnimationTimeline waveEffect() {
        return waveEffect(Binding.zRot, 2.5f, 5f, 0f, duration);
    }
}


