package com.ranull.graves.manager;

import com.ranull.graves.Graves;
import com.ranull.graves.integration.MiniMessage;
import com.ranull.graves.util.StringUtil;
import org.bukkit.Bukkit;
import org.bukkit.ChatColor;
import org.bukkit.Material;
import org.bukkit.NamespacedKey;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.inventory.ItemFlag;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.Recipe;
import org.bukkit.inventory.ShapedRecipe;
import org.bukkit.inventory.meta.ItemMeta;
import org.bukkit.inventory.meta.components.CustomModelDataComponent;
import org.bukkit.persistence.PersistentDataType;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**6
 * Manages custom recipes for the Graves plugin.
 */
public class RecipeManager {
    /**
     * The main plugin instance associated with Graves.
     * <p>
     * This {@link Graves} instance represents the core plugin that Graves is part of. It provides access
     * to the plugin's functionality, configuration, and other services.
     * </p>
     */
    private final Graves plugin;

    /**
     * A list of {@link NamespacedKey} objects used for managing and accessing custom data.
     * <p>
     * This {@link List} holds {@link NamespacedKey} instances which are used to uniquely identify and manage custom data
     * keys within Graves. Namespaced keys are crucial for avoiding key collisions in persistent data containers.
     * </p>
     */
    private final List<NamespacedKey> namespacedKeyList;

    /**
     * Initializes a new instance of the RecipeManager class.
     *
     * @param plugin The plugin instance.
     */
    public RecipeManager(Graves plugin) {
        this.plugin = plugin;
        this.namespacedKeyList = new ArrayList<>();
        reload();
    }

    /**
     * Reloads the recipes.
     */
    public void reload() {
        unload();
        load();
    }

    /**
     * Loads the recipes from the configuration.
     */
    public void load() {
        ConfigurationSection configurationSection = plugin.getConfig().getConfigurationSection("settings.token");

        if (configurationSection != null) {
            for (String key : configurationSection.getKeys(false)) {
                if (plugin.getConfig().getBoolean("settings.token." + key + ".craft")) {
                    addTokenRecipe(key, getToken(key));
                    plugin.debugMessage("Added recipe " + key, 1);
                }
            }
        }
    }

    /**
     * Unloads the custom recipes.
     */
    public void unload() {
        try {
            Iterator<Recipe> iterator = Bukkit.recipeIterator();
            while (iterator.hasNext()) {
                Recipe recipe = iterator.next();

                if (recipe != null) {
                    ItemStack itemStack = recipe.getResult();

                    if (itemStack.hasItemMeta() && isToken(itemStack)) {
                        iterator.remove();
                    }
                }
            }
        } catch (Exception ignored) {
            // End the loop if the exception occurs
        }
    }

    /**
     * Retrieves the token item based on the configuration.
     *
     * @param token The token identifier.
     * @return The token item.
     */
    public ItemStack getToken(String token) {
        if (plugin.getConfig().isConfigurationSection("settings.token." + token)) {
            Material material = Material.matchMaterial(
                    plugin.getConfig().getString("settings.token." + token + ".material", "SUNFLOWER")
            );
            ItemStack itemStack = new ItemStack(material != null ? material : Material.CHEST);

            setRecipeData(token, itemStack);

            if (itemStack.hasItemMeta()) {
                ItemMeta itemMeta = itemStack.getItemMeta();

                if (itemMeta != null) {
                    String name;
                    if (plugin.getIntegrationManager().hasMiniMessage()) {
                        String newName = StringUtil.parseString(
                                "&f" + plugin.getConfig().getString("settings.token." + token + ".name"),
                                plugin
                        );
                        name = MiniMessage.parseString(newName);
                    } else {
                        name = ChatColor.WHITE + StringUtil.parseString(
                                plugin.getConfig().getString("settings.token." + token + ".name"),
                                plugin
                        );
                    }

                    List<String> loreList = new ArrayList<>();
                    int customModelData = plugin.getConfig()
                            .getInt("settings.token." + token + ".model-data", -1);

                    for (String string : plugin.getConfig().getStringList("settings.token." + token + ".lore")) {
                        if (plugin.getIntegrationManager().hasMiniMessage()) {
                            String newLine = StringUtil.parseString("&7" + string, plugin);
                            loreList.add(MiniMessage.parseString(newLine));
                        } else {
                            loreList.add(ChatColor.GRAY + StringUtil.parseString(string, plugin));
                        }
                    }

                    if (plugin.getConfig().getBoolean("settings.token." + token + ".glow")) {
                        itemMeta.addEnchant(plugin.getVersionManager().getEnchantmentForVersion("DURABILITY"), 1, true);
                        itemMeta.addItemFlags(ItemFlag.HIDE_ENCHANTS);
                    }

                    if (customModelData > -1) {
                        try {
                            CustomModelDataComponent cmdComponent = itemMeta.getCustomModelDataComponent();
                            cmdComponent.setFloats(Collections.singletonList((float) customModelData));
                            itemMeta.setCustomModelDataComponent(cmdComponent);
                        } catch (Exception e) {
                            itemMeta.setCustomModelData(customModelData);
                        }
                    }

                    itemMeta.setLore(loreList);
                    itemMeta.setDisplayName(name);
                    itemStack.setItemMeta(itemMeta);
                }
            }

            return itemStack;
        }

        return null;
    }

    /**
     * Retrieves the list of tokens from the configuration.
     *
     * @return The list of token identifiers.
     */
    public List<String> getTokenList() {
        List<String> stringList = new ArrayList<>();
        ConfigurationSection configurationSection = plugin.getConfig().getConfigurationSection("settings.token");

        if (configurationSection != null) {
            stringList.addAll(configurationSection.getKeys(false));
        }

        return stringList;
    }

    /**
     * Adds a custom token recipe.
     *
     * @param token     The token identifier.
     * @param itemStack The item stack representing the token.
     */
    public void addTokenRecipe(String token, ItemStack itemStack) {
        NamespacedKey namespacedKey = new NamespacedKey(plugin, token + "GraveToken");

        if (namespacedKeyList.contains(namespacedKey)) return;

        try {
            List<String> lineList = plugin.getConfig().getStringList("settings.token." + token + ".recipe");
            if (lineList.size() < 3) {
                plugin.getLogger().warning("Invalid recipe format for token " + token + ", expected 3 lines.");
                return;
            }

            Map<Character, Material> ingredients = new HashMap<>();
            StringBuilder[] shapeLines = new StringBuilder[]{
                    new StringBuilder(), new StringBuilder(), new StringBuilder()
            };

            int recipeKey = 1;

            for (int row = 0; row < 3; row++) {
                String[] parts = lineList.get(row).split(" ");
                for (String part : parts) {
                    char ingredientChar = getChar(recipeKey);
                    Material mat = Material.matchMaterial(part);

                    if (mat != null && mat != Material.AIR) {
                        ingredients.put(ingredientChar, mat);
                        shapeLines[row].append(ingredientChar);
                    } else {
                        shapeLines[row].append(' ');
                    }

                    recipeKey++;
                }
            }

            ShapedRecipe shapedRecipe = new ShapedRecipe(namespacedKey, itemStack);
            shapedRecipe.shape(shapeLines[0].toString(), shapeLines[1].toString(), shapeLines[2].toString());

            for (Map.Entry<Character, Material> entry : ingredients.entrySet()) {
                shapedRecipe.setIngredient(entry.getKey(), entry.getValue());
            }

            if (plugin.getServer().getRecipe(namespacedKey) == null) {
                plugin.getServer().addRecipe(shapedRecipe);
                namespacedKeyList.add(namespacedKey);
            } else {
                plugin.debugMessage("Unable to add recipe " + namespacedKey.getKey(), 1);
            }

        } catch (Exception e) {
            plugin.getLogger().severe("Unable to register token recipe for " + token + ". Check your recipe in token.yml");
            plugin.getLogger().severe("This is likely an invalid Material or a Spigot-related bug.");
            plugin.logStackTrace(e);
        }
    }

    /**
     * Retrieves a token item from a player's inventory.
     *
     * @param token         The token identifier.
     * @param itemStackList The list of item stacks in the player's inventory.
     * @return The token item stack, or null if not found.
     */
    public ItemStack getGraveTokenFromPlayer(String token, List<ItemStack> itemStackList) {
        for (ItemStack itemStack : itemStackList) {
            if (itemStack != null && isToken(token, itemStack)) {
                return itemStack;
            }
        }

        return null;
    }

    /**
     * Sets the recipe data on an item stack.
     *
     * @param token     The token identifier.
     * @param itemStack The item stack.
     */
    public void setRecipeData(String token, ItemStack itemStack) {
        if (plugin.getVersionManager().hasPersistentData()) {
            ItemMeta itemMeta = itemStack.getItemMeta();

            if (itemMeta != null) {
                itemMeta.getPersistentDataContainer().set(
                        new NamespacedKey(plugin, "token"),
                        PersistentDataType.STRING,
                        token
                );
                itemStack.setItemMeta(itemMeta);
            }
        }
    }

    /**
     * Checks if an item stack is a token of a specified type.
     *
     * @param token     The token identifier.
     * @param itemStack The item stack.
     * @return True if the item stack is a token of the specified type, otherwise false.
     */
    public boolean isToken(String token, ItemStack itemStack) {
        if (plugin.getVersionManager().hasPersistentData()) {
            if (itemStack.getItemMeta() != null && itemStack.getItemMeta().getPersistentDataContainer()
                    .has(new NamespacedKey(plugin, "token"), PersistentDataType.STRING)) {
                String string = itemStack.getItemMeta().getPersistentDataContainer()
                        .get(new NamespacedKey(plugin, "token"), PersistentDataType.STRING);

                return string != null && string.equals(token);
            }
        } else {
            ItemMeta meta = itemStack.getItemMeta();
            if (meta != null && meta.hasLore()) {
                List<String> lore = meta.getLore();
                if (lore != null) {
                    for (String loreEntry : lore) {
                        if (loreEntry.equals(token)) {
                            return true;
                        }
                    }
                }
            }
        }

        return false;
    }

    /**
     * Retrieves the token name from an item stack.
     *
     * @param itemStack The item stack.
     * @return The token name, or null if not found.
     */
    public String getTokenName(ItemStack itemStack) {
        if (plugin.getVersionManager().hasPersistentData()) {
            if (itemStack.getItemMeta() != null && itemStack.getItemMeta().getPersistentDataContainer()
                    .has(new NamespacedKey(plugin, "token"), PersistentDataType.STRING)) {
                return itemStack.getItemMeta().getPersistentDataContainer()
                        .get(new NamespacedKey(plugin, "token"), PersistentDataType.STRING);
            }
        }

        return null;
    }

    /**
     * Checks if an item stack is a token.
     *
     * @param itemStack The item stack.
     * @return True if the item stack is a token, otherwise false.
     */
    public boolean isToken(ItemStack itemStack) {
        return plugin.getVersionManager().hasPersistentData()
                && itemStack.getItemMeta() != null
                && itemStack.getItemMeta().getPersistentDataContainer()
                .has(new NamespacedKey(plugin, "token"), PersistentDataType.STRING);
    }

    /**
     * Gets the character for a recipe slot based on the count.
     *
     * @param count The count.
     * @return The character for the recipe slot.
     */
    private char getChar(int count) {
        return switch (count) {
            case 1 -> 'A';
            case 2 -> 'B';
            case 3 -> 'C';
            case 4 -> 'D';
            case 5 -> 'E';
            case 6 -> 'F';
            case 7 -> 'G';
            case 8 -> 'H';
            case 9 -> 'I';
            default -> '*';
        };
    }
}