/*
	* Copyright (C) 2002-2024 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.objects;

import it.unimi.dsi.fastutil.Hash;
import it.unimi.dsi.fastutil.HashCommon;
import static it.unimi.dsi.fastutil.HashCommon.arraySize;
import static it.unimi.dsi.fastutil.HashCommon.maxFill;
import java.util.Map;
import java.util.Arrays;
import java.util.NoSuchElementException;
import java.util.function.Consumer;
import it.unimi.dsi.fastutil.doubles.DoubleCollection;
import it.unimi.dsi.fastutil.doubles.AbstractDoubleCollection;
import it.unimi.dsi.fastutil.doubles.DoubleIterator;
import it.unimi.dsi.fastutil.doubles.DoubleSpliterator;
import it.unimi.dsi.fastutil.doubles.DoubleSpliterators;
import java.util.Comparator;
import it.unimi.dsi.fastutil.doubles.DoubleListIterator;

/**
 * A type-specific linked hash map with with a fast, small-footprint implementation.
 *
 * <p>
 * Instances of this class use a hash table to represent a map. The table is filled up to a
 * specified <em>load factor</em>, and then doubled in size to accommodate new entries. If the table
 * is emptied below <em>one fourth</em> of the load factor, it is halved in size; however, the table
 * is never reduced to a size smaller than that at creation time: this approach makes it possible to
 * create maps with a large capacity in which insertions and deletions do not cause immediately
 * rehashing. Moreover, halving is not performed when deleting entries from an iterator, as it would
 * interfere with the iteration process.
 *
 * <p>
 * Note that {@link #clear()} does not modify the hash table size. Rather, a family of
 * {@linkplain #trim() trimming methods} lets you control the size of the table; this is
 * particularly useful if you reuse instances of this class.
 *
 * <p>
 * Entries returned by the type-specific {@link #entrySet()} method implement the suitable
 * type-specific {@link it.unimi.dsi.fastutil.Pair Pair} interface; only values are mutable.
 *
 * <p>
 * Iterators generated by this map will enumerate pairs in the same order in which they have been
 * added to the map (addition of pairs whose key is already present in the map does not change the
 * iteration order). Note that this order has nothing in common with the natural order of the keys.
 * The order is kept by means of a doubly linked list, represented <i>via</i> an array of longs
 * parallel to the table.
 *
 * <p>
 * This class implements the interface of a sorted map, so to allow easy access of the iteration
 * order: for instance, you can get the first key in iteration order with {@code firstKey()} without
 * having to create an iterator; however, this class partially violates the
 * {@link java.util.SortedMap} contract because all submap methods throw an exception and
 * {@link #comparator()} returns always {@code null}.
 *
 * <p>
 * Additional methods, such as {@code getAndMoveToFirst()}, make it easy to use instances of this
 * class as a cache (e.g., with LRU policy).
 *
 * <p>
 * The iterators provided by the views of this class using are type-specific
 * {@linkplain java.util.ListIterator list iterators}, and can be started at any element <em>which
 * is a key of the map</em>, or a {@link NoSuchElementException} exception will be thrown. If,
 * however, the provided element is not the first or last key in the map, the first access to the
 * list index will require linear time, as in the worst case the entire key set must be scanned in
 * iteration order to retrieve the positional index of the starting key. If you use just the methods
 * of a type-specific {@link it.unimi.dsi.fastutil.BidirectionalIterator}, however, all operations
 * will be performed in constant time.
 *
 * @see Hash
 * @see HashCommon
 */
public class Object2DoubleLinkedOpenHashMap<K> extends AbstractObject2DoubleSortedMap<K> implements java.io.Serializable, Cloneable, Hash {
	private static final long serialVersionUID = 0L;
	private static final boolean ASSERTS = false;
	/** The array of keys. */
	protected transient K[] key;
	/** The array of values. */
	protected transient double[] value;
	/** The mask for wrapping a position counter. */
	protected transient int mask;
	/** Whether this map contains the key zero. */
	protected transient boolean containsNullKey;
	/**
	 * The index of the first entry in iteration order. It is valid iff {@link #size} is nonzero;
	 * otherwise, it contains -1.
	 */
	protected transient int first = -1;
	/**
	 * The index of the last entry in iteration order. It is valid iff {@link #size} is nonzero;
	 * otherwise, it contains -1.
	 */
	protected transient int last = -1;
	/**
	 * For each entry, the next and the previous entry in iteration order, stored as
	 * {@code ((prev & 0xFFFFFFFFL) << 32) | (next & 0xFFFFFFFFL)}. The first entry contains predecessor
	 * -1, and the last entry contains successor -1.
	 */
	protected transient long[] link;
	/** The current table size. */
	protected transient int n;
	/** Threshold after which we rehash. It must be the table size times {@link #f}. */
	protected transient int maxFill;
	/** We never resize below this threshold, which is the construction-time {#n}. */
	protected final transient int minN;
	/** Number of entries in the set (including the key zero, if present). */
	protected int size;
	/** The acceptable load factor. */
	protected final float f;
	/** Cached set of entries. */
	protected transient FastSortedEntrySet<K> entries;
	/** Cached set of keys. */
	protected transient ObjectSortedSet<K> keys;
	/** Cached collection of values. */
	protected transient DoubleCollection values;

	/**
	 * Creates a new hash map.
	 *
	 * <p>
	 * The actual table size will be the least power of two greater than {@code expected}/{@code f}.
	 *
	 * @param expected the expected number of elements in the hash map.
	 * @param f the load factor.
	 */
	@SuppressWarnings("unchecked")
	public Object2DoubleLinkedOpenHashMap(final int expected, final float f) {
		if (f <= 0 || f >= 1) throw new IllegalArgumentException("Load factor must be greater than 0 and smaller than 1");
		if (expected < 0) throw new IllegalArgumentException("The expected number of elements must be nonnegative");
		this.f = f;
		minN = n = arraySize(expected, f);
		mask = n - 1;
		maxFill = maxFill(n, f);
		key = (K[])new Object[n + 1];
		value = new double[n + 1];
		link = new long[n + 1];
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor.
	 *
	 * @param expected the expected number of elements in the hash map.
	 */
	public Object2DoubleLinkedOpenHashMap(final int expected) {
		this(expected, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map with initial expected {@link Hash#DEFAULT_INITIAL_SIZE} entries and
	 * {@link Hash#DEFAULT_LOAD_FACTOR} as load factor.
	 */
	public Object2DoubleLinkedOpenHashMap() {
		this(DEFAULT_INITIAL_SIZE, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map.
	 * @param f the load factor.
	 */
	public Object2DoubleLinkedOpenHashMap(final Map<? extends K, ? extends Double> m, final float f) {
		this(m.size(), f);
		putAll(m);
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map.
	 */
	public Object2DoubleLinkedOpenHashMap(final Map<? extends K, ? extends Double> m) {
		this(m, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map copying a given type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map.
	 * @param f the load factor.
	 */
	public Object2DoubleLinkedOpenHashMap(final Object2DoubleMap<K> m, final float f) {
		this(m.size(), f);
		putAll(m);
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given
	 * type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map.
	 */
	public Object2DoubleLinkedOpenHashMap(final Object2DoubleMap<K> m) {
		this(m, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map using the elements of two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @param f the load factor.
	 * @throws IllegalArgumentException if {@code k} and {@code v} have different lengths.
	 */
	public Object2DoubleLinkedOpenHashMap(final K[] k, final double[] v, final float f) {
		this(k.length, f);
		if (k.length != v.length) throw new IllegalArgumentException("The key array and the value array have different lengths (" + k.length + " and " + v.length + ")");
		for (int i = 0; i < k.length; i++) this.put(k[i], v[i]);
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor using the elements of
	 * two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @throws IllegalArgumentException if {@code k} and {@code v} have different lengths.
	 */
	public Object2DoubleLinkedOpenHashMap(final K[] k, final double[] v) {
		this(k, v, DEFAULT_LOAD_FACTOR);
	}

	private int realSize() {
		return containsNullKey ? size - 1 : size;
	}

	/**
	 * Ensures that this map can hold a certain number of keys without rehashing.
	 *
	 * @param capacity a number of keys; there will be no rehashing unless the map {@linkplain #size()
	 *            size} exceeds this number.
	 */
	public void ensureCapacity(final int capacity) {
		final int needed = arraySize(capacity, f);
		if (needed > n) rehash(needed);
	}

	private void tryCapacity(final long capacity) {
		final int needed = (int)Math.min(1 << 30, Math.max(2, HashCommon.nextPowerOfTwo((long)Math.ceil(capacity / f))));
		if (needed > n) rehash(needed);
	}

	private double removeEntry(final int pos) {
		final double oldValue = value[pos];
		size--;
		fixPointers(pos);
		shiftKeys(pos);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return oldValue;
	}

	private double removeNullEntry() {
		containsNullKey = false;
		key[n] = null;
		final double oldValue = value[n];
		size--;
		fixPointers(n);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return oldValue;
	}

	@Override
	public void putAll(Map<? extends K, ? extends Double> m) {
		if (f <= .5) ensureCapacity(m.size()); // The resulting map will be sized for m.size() elements
		else tryCapacity(size() + m.size()); // The resulting map will be tentatively sized for size() + m.size()
												// elements
		super.putAll(m);
	}

	@SuppressWarnings("unchecked")
	private int find(final K k) {
		if (((k) == null)) return containsNullKey ? n : -(n + 1);
		K curr;
		final K[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return -(pos + 1);
		if (((k).equals(curr))) return pos;
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == null)) return -(pos + 1);
			if (((k).equals(curr))) return pos;
		}
	}

	private void insert(final int pos, final K k, final double v) {
		if (pos == n) containsNullKey = true;
		key[pos] = k;
		value[pos] = v;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[last] ^= ((link[last] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[pos] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
			last = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size + 1, f));
		if (ASSERTS) checkTable();
	}

	@Override
	public double put(final K k, final double v) {
		final int pos = find(k);
		if (pos < 0) {
			insert(-pos - 1, k, v);
			return defRetValue;
		}
		final double oldValue = value[pos];
		value[pos] = v;
		return oldValue;
	}

	private double addToValue(final int pos, final double incr) {
		final double oldValue = value[pos];
		value[pos] = oldValue + incr;
		return oldValue;
	}

	/**
	 * Adds an increment to value currently associated with a key.
	 *
	 * <p>
	 * Note that this method respects the {@linkplain #defaultReturnValue() default return value}
	 * semantics: when called with a key that does not currently appears in the map, the key will be
	 * associated with the default return value plus the given increment.
	 *
	 * @param k the key.
	 * @param incr the increment.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value
	 *         was present for the given key.
	 */
	public double addTo(final K k, final double incr) {
		int pos;
		if (((k) == null)) {
			if (containsNullKey) return addToValue(n, incr);
			pos = n;
			containsNullKey = true;
		} else {
			K curr;
			final K[] key = this.key;
			// The starting point.
			if (!((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) {
				if (((curr).equals(k))) return addToValue(pos, incr);
				while (!((curr = key[pos = (pos + 1) & mask]) == null)) if (((curr).equals(k))) return addToValue(pos, incr);
			}
		}
		key[pos] = k;
		value[pos] = defRetValue + incr;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[last] ^= ((link[last] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[pos] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
			last = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size + 1, f));
		if (ASSERTS) checkTable();
		return defRetValue;
	}

	/**
	 * Shifts left entries with the specified hash code, starting at the specified position, and empties
	 * the resulting free entry.
	 *
	 * @param pos a starting position.
	 */
	protected final void shiftKeys(int pos) {
		// Shift entries with the same hash.
		int last, slot;
		K curr;
		final K[] key = this.key;
		final double value[] = this.value;
		for (;;) {
			pos = ((last = pos) + 1) & mask;
			for (;;) {
				if (((curr = key[pos]) == null)) {
					key[last] = (null);
					return;
				}
				slot = (it.unimi.dsi.fastutil.HashCommon.mix((curr).hashCode())) & mask;
				if (last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos) break;
				pos = (pos + 1) & mask;
			}
			key[last] = curr;
			value[last] = value[pos];
			fixPointers(pos, last);
		}
	}

	@Override
	@SuppressWarnings("unchecked")
	public double removeDouble(final Object k) {
		if ((((K)k) == null)) {
			if (containsNullKey) return removeNullEntry();
			return defRetValue;
		}
		K curr;
		final K[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return defRetValue;
		if (((k).equals(curr))) return removeEntry(pos);
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == null)) return defRetValue;
			if (((k).equals(curr))) return removeEntry(pos);
		}
	}

	private double setValue(final int pos, final double v) {
		final double oldValue = value[pos];
		value[pos] = v;
		return oldValue;
	}

	/**
	 * Removes the mapping associated with the first key in iteration order.
	 * 
	 * @return the value previously associated with the first key in iteration order.
	 * @throws NoSuchElementException is this map is empty.
	 */
	public double removeFirstDouble() {
		if (size == 0) throw new NoSuchElementException();
		final int pos = first;
		// Abbreviated version of fixPointers(pos)
		if (size == 1) first = last = -1;
		else {
			first = (int)link[pos];
			if (0 <= first) {
				// Special case of SET_PREV(link[first], -1)
				link[first] |= (-1 & 0xFFFFFFFFL) << 32;
			}
		}
		size--;
		final double v = value[pos];
		if (pos == n) {
			containsNullKey = false;
			key[n] = null;
		} else shiftKeys(pos);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return v;
	}

	/**
	 * Removes the mapping associated with the last key in iteration order.
	 * 
	 * @return the value previously associated with the last key in iteration order.
	 * @throws NoSuchElementException is this map is empty.
	 */
	public double removeLastDouble() {
		if (size == 0) throw new NoSuchElementException();
		final int pos = last;
		// Abbreviated version of fixPointers(pos)
		if (size == 1) first = last = -1;
		else {
			last = (int)(link[pos] >>> 32);
			if (0 <= last) {
				// Special case of SET_NEXT(link[last], -1)
				link[last] |= -1 & 0xFFFFFFFFL;
			}
		}
		size--;
		final double v = value[pos];
		if (pos == n) {
			containsNullKey = false;
			key[n] = null;
		} else shiftKeys(pos);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return v;
	}

	private void moveIndexToFirst(final int i) {
		if (size == 1 || first == i) return;
		if (last == i) {
			last = (int)(link[i] >>> 32);
			// Special case of SET_NEXT(link[last], -1);
			link[last] |= -1 & 0xFFFFFFFFL;
		} else {
			final long linki = link[i];
			final int prev = (int)(linki >>> 32);
			final int next = (int)linki;
			link[prev] ^= ((link[prev] ^ (linki & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[next] ^= ((link[next] ^ (linki & 0xFFFFFFFF00000000L)) & 0xFFFFFFFF00000000L);
		}
		link[first] ^= ((link[first] ^ ((i & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
		link[i] = ((-1 & 0xFFFFFFFFL) << 32) | (first & 0xFFFFFFFFL);
		first = i;
	}

	private void moveIndexToLast(final int i) {
		if (size == 1 || last == i) return;
		if (first == i) {
			first = (int)link[i];
			// Special case of SET_PREV(link[first], -1);
			link[first] |= (-1 & 0xFFFFFFFFL) << 32;
		} else {
			final long linki = link[i];
			final int prev = (int)(linki >>> 32);
			final int next = (int)linki;
			link[prev] ^= ((link[prev] ^ (linki & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[next] ^= ((link[next] ^ (linki & 0xFFFFFFFF00000000L)) & 0xFFFFFFFF00000000L);
		}
		link[last] ^= ((link[last] ^ (i & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
		link[i] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
		last = i;
	}

	/**
	 * Returns the value to which the given key is mapped; if the key is present, it is moved to the
	 * first position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value}
	 *         if no value was present for the given key.
	 */
	public double getAndMoveToFirst(final K k) {
		if (((k) == null)) {
			if (containsNullKey) {
				moveIndexToFirst(n);
				return value[n];
			}
			return defRetValue;
		}
		K curr;
		final K[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return defRetValue;
		if (((k).equals(curr))) {
			moveIndexToFirst(pos);
			return value[pos];
		}
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == null)) return defRetValue;
			if (((k).equals(curr))) {
				moveIndexToFirst(pos);
				return value[pos];
			}
		}
	}

	/**
	 * Returns the value to which the given key is mapped; if the key is present, it is moved to the
	 * last position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value}
	 *         if no value was present for the given key.
	 */
	public double getAndMoveToLast(final K k) {
		if (((k) == null)) {
			if (containsNullKey) {
				moveIndexToLast(n);
				return value[n];
			}
			return defRetValue;
		}
		K curr;
		final K[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return defRetValue;
		if (((k).equals(curr))) {
			moveIndexToLast(pos);
			return value[pos];
		}
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == null)) return defRetValue;
			if (((k).equals(curr))) {
				moveIndexToLast(pos);
				return value[pos];
			}
		}
	}

	/**
	 * Adds a pair to the map; if the key is already present, it is moved to the first position of the
	 * iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value
	 *         was present for the given key.
	 */
	public double putAndMoveToFirst(final K k, final double v) {
		int pos;
		if (((k) == null)) {
			if (containsNullKey) {
				moveIndexToFirst(n);
				return setValue(n, v);
			}
			containsNullKey = true;
			pos = n;
		} else {
			K curr;
			final K[] key = this.key;
			// The starting point.
			if (!((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) {
				if (((curr).equals(k))) {
					moveIndexToFirst(pos);
					return setValue(pos, v);
				}
				while (!((curr = key[pos = (pos + 1) & mask]) == null)) if (((curr).equals(k))) {
					moveIndexToFirst(pos);
					return setValue(pos, v);
				}
			}
		}
		key[pos] = k;
		value[pos] = v;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[first] ^= ((link[first] ^ ((pos & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
			link[pos] = ((-1 & 0xFFFFFFFFL) << 32) | (first & 0xFFFFFFFFL);
			first = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size, f));
		if (ASSERTS) checkTable();
		return defRetValue;
	}

	/**
	 * Adds a pair to the map; if the key is already present, it is moved to the last position of the
	 * iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value
	 *         was present for the given key.
	 */
	public double putAndMoveToLast(final K k, final double v) {
		int pos;
		if (((k) == null)) {
			if (containsNullKey) {
				moveIndexToLast(n);
				return setValue(n, v);
			}
			containsNullKey = true;
			pos = n;
		} else {
			K curr;
			final K[] key = this.key;
			// The starting point.
			if (!((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) {
				if (((curr).equals(k))) {
					moveIndexToLast(pos);
					return setValue(pos, v);
				}
				while (!((curr = key[pos = (pos + 1) & mask]) == null)) if (((curr).equals(k))) {
					moveIndexToLast(pos);
					return setValue(pos, v);
				}
			}
		}
		key[pos] = k;
		value[pos] = v;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[last] ^= ((link[last] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[pos] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
			last = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size, f));
		if (ASSERTS) checkTable();
		return defRetValue;
	}

	@Override
	@SuppressWarnings("unchecked")
	public double getDouble(final Object k) {
		if ((((K)k) == null)) return containsNullKey ? value[n] : defRetValue;
		K curr;
		final K[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return defRetValue;
		if (((k).equals(curr))) return value[pos];
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == null)) return defRetValue;
			if (((k).equals(curr))) return value[pos];
		}
	}

	@Override
	@SuppressWarnings("unchecked")
	public boolean containsKey(final Object k) {
		if ((((K)k) == null)) return containsNullKey;
		K curr;
		final K[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return false;
		if (((k).equals(curr))) return true;
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == null)) return false;
			if (((k).equals(curr))) return true;
		}
	}

	@Override
	public boolean containsValue(final double v) {
		final K key[] = this.key;
		final double value[] = this.value;
		if (containsNullKey && (Double.doubleToLongBits(value[n]) == Double.doubleToLongBits(v))) return true;
		for (int i = n; i-- != 0;) if (!((key[i]) == null) && (Double.doubleToLongBits(value[i]) == Double.doubleToLongBits(v))) return true;
		return false;
	}

	/** {@inheritDoc} */
	@Override
	@SuppressWarnings("unchecked")
	public double getOrDefault(final Object k, final double defaultValue) {
		if ((((K)k) == null)) return containsNullKey ? value[n] : defaultValue;
		K curr;
		final K[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return defaultValue;
		if (((k).equals(curr))) return value[pos];
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == null)) return defaultValue;
			if (((k).equals(curr))) return value[pos];
		}
	}

	/** {@inheritDoc} */
	@Override
	public double putIfAbsent(final K k, final double v) {
		final int pos = find(k);
		if (pos >= 0) return value[pos];
		insert(-pos - 1, k, v);
		return defRetValue;
	}

	/** {@inheritDoc} */
	@Override
	@SuppressWarnings("unchecked")
	public boolean remove(final Object k, final double v) {
		if ((((K)k) == null)) {
			if (containsNullKey && (Double.doubleToLongBits(v) == Double.doubleToLongBits(value[n]))) {
				removeNullEntry();
				return true;
			}
			return false;
		}
		K curr;
		final K[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return false;
		if (((k).equals(curr)) && (Double.doubleToLongBits(v) == Double.doubleToLongBits(value[pos]))) {
			removeEntry(pos);
			return true;
		}
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == null)) return false;
			if (((k).equals(curr)) && (Double.doubleToLongBits(v) == Double.doubleToLongBits(value[pos]))) {
				removeEntry(pos);
				return true;
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	public boolean replace(final K k, final double oldValue, final double v) {
		final int pos = find(k);
		if (pos < 0 || !(Double.doubleToLongBits(oldValue) == Double.doubleToLongBits(value[pos]))) return false;
		value[pos] = v;
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public double replace(final K k, final double v) {
		final int pos = find(k);
		if (pos < 0) return defRetValue;
		final double oldValue = value[pos];
		value[pos] = v;
		return oldValue;
	}

	/** {@inheritDoc} */
	@Override
	public double computeIfAbsent(final K k, final java.util.function.ToDoubleFunction<? super K> mappingFunction) {
		java.util.Objects.requireNonNull(mappingFunction);
		final int pos = find(k);
		if (pos >= 0) return value[pos];
		final double newValue = mappingFunction.applyAsDouble(k);
		insert(-pos - 1, k, newValue);
		return newValue;
	}

	/** {@inheritDoc} */
	@Override
	public double computeIfAbsent(final K key, final Object2DoubleFunction<? super K> mappingFunction) {
		java.util.Objects.requireNonNull(mappingFunction);
		final int pos = find(key);
		if (pos >= 0) return value[pos];
		if (!mappingFunction.containsKey(key)) return defRetValue;
		final double newValue = mappingFunction.getDouble(key);
		insert(-pos - 1, key, newValue);
		return newValue;
	}

	/** {@inheritDoc} */
	@Override
	public double computeDoubleIfPresent(final K k, final java.util.function.BiFunction<? super K, ? super Double, ? extends Double> remappingFunction) {
		java.util.Objects.requireNonNull(remappingFunction);
		final int pos = find(k);
		if (pos < 0) return defRetValue;
		final Double newValue = remappingFunction.apply((k), Double.valueOf(value[pos]));
		if (newValue == null) {
			if (((k) == null)) removeNullEntry();
			else removeEntry(pos);
			return defRetValue;
		}
		return value[pos] = (newValue).doubleValue();
	}

	/** {@inheritDoc} */
	@Override
	public double computeDouble(final K k, final java.util.function.BiFunction<? super K, ? super Double, ? extends Double> remappingFunction) {
		java.util.Objects.requireNonNull(remappingFunction);
		final int pos = find(k);
		final Double newValue = remappingFunction.apply((k), pos >= 0 ? Double.valueOf(value[pos]) : null);
		if (newValue == null) {
			if (pos >= 0) {
				if (((k) == null)) removeNullEntry();
				else removeEntry(pos);
			}
			return defRetValue;
		}
		double newVal = (newValue).doubleValue();
		if (pos < 0) {
			insert(-pos - 1, k, newVal);
			return newVal;
		}
		return value[pos] = newVal;
	}

	/** {@inheritDoc} */
	@Override
	public double merge(final K k, final double v, final java.util.function.BiFunction<? super Double, ? super Double, ? extends Double> remappingFunction) {
		java.util.Objects.requireNonNull(remappingFunction);

		final int pos = find(k);
		if (pos < 0) {
			if (pos < 0) insert(-pos - 1, k, v);
			else value[pos] = v;
			return v;
		}
		final Double newValue = remappingFunction.apply(Double.valueOf(value[pos]), Double.valueOf(v));
		if (newValue == null) {
			if (((k) == null)) removeNullEntry();
			else removeEntry(pos);
			return defRetValue;
		}
		return value[pos] = (newValue).doubleValue();
	}

	/* Removes all elements from this map.
	 *
	 * <p>To increase object reuse, this method does not change the table size.
	 * If you want to reduce the table size, you must use {@link #trim()}.
	 *
	 */
	@Override
	public void clear() {
		if (size == 0) return;
		size = 0;
		containsNullKey = false;
		Arrays.fill(key, (null));
		first = last = -1;
	}

	@Override
	public int size() {
		return size;
	}

	@Override
	public boolean isEmpty() {
		return size == 0;
	}

	/**
	 * The entry class for a hash map does not record key and value, but rather the position in the hash
	 * table of the corresponding entry. This is necessary so that calls to
	 * {@link java.util.Map.Entry#setValue(Object)} are reflected in the map
	 */
	final class MapEntry implements Object2DoubleMap.Entry<K>, Map.Entry<K, Double>, ObjectDoublePair<K> {
		// The table index this entry refers to, or -1 if this entry has been deleted.
		int index;

		MapEntry(final int index) {
			this.index = index;
		}

		MapEntry() {
		}

		@Override
		public K getKey() {
			return key[index];
		}

		@Override
		public K left() {
			return key[index];
		}

		@Override
		public double getDoubleValue() {
			return value[index];
		}

		@Override
		public double rightDouble() {
			return value[index];
		}

		@Override
		public double setValue(final double v) {
			final double oldValue = value[index];
			value[index] = v;
			return oldValue;
		}

		@Override
		public ObjectDoublePair<K> right(final double v) {
			value[index] = v;
			return this;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method instead.
		 */
		@Deprecated
		@Override
		public Double getValue() {
			return Double.valueOf(value[index]);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method instead.
		 */
		@Deprecated
		@Override
		public Double setValue(final Double v) {
			return Double.valueOf(setValue((v).doubleValue()));
		}

		@SuppressWarnings("unchecked")
		@Override
		public boolean equals(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			Map.Entry<K, Double> e = (Map.Entry<K, Double>)o;
			return java.util.Objects.equals(key[index], (e.getKey())) && (Double.doubleToLongBits(value[index]) == Double.doubleToLongBits((e.getValue()).doubleValue()));
		}

		@Override
		public int hashCode() {
			return ((key[index]) == null ? 0 : (key[index]).hashCode()) ^ it.unimi.dsi.fastutil.HashCommon.double2int(value[index]);
		}

		@Override
		public String toString() {
			return key[index] + "=>" + value[index];
		}
	}

	/**
	 * Modifies the {@link #link} vector so that the given entry is removed. This method will complete
	 * in constant time.
	 *
	 * @param i the index of an entry.
	 */
	protected void fixPointers(final int i) {
		if (size == 0) {
			first = last = -1;
			return;
		}
		if (first == i) {
			first = (int)link[i];
			if (0 <= first) {
				// Special case of SET_PREV(link[first], -1)
				link[first] |= (-1 & 0xFFFFFFFFL) << 32;
			}
			return;
		}
		if (last == i) {
			last = (int)(link[i] >>> 32);
			if (0 <= last) {
				// Special case of SET_NEXT(link[last], -1)
				link[last] |= -1 & 0xFFFFFFFFL;
			}
			return;
		}
		final long linki = link[i];
		final int prev = (int)(linki >>> 32);
		final int next = (int)linki;
		link[prev] ^= ((link[prev] ^ (linki & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
		link[next] ^= ((link[next] ^ (linki & 0xFFFFFFFF00000000L)) & 0xFFFFFFFF00000000L);
	}

	/**
	 * Modifies the {@link #link} vector for a shift from s to d.
	 * <p>
	 * This method will complete in constant time.
	 *
	 * @param s the source position.
	 * @param d the destination position.
	 */
	protected void fixPointers(int s, int d) {
		if (size == 1) {
			first = last = d;
			// Special case of SET_UPPER_LOWER(link[d], -1, -1)
			link[d] = -1L;
			return;
		}
		if (first == s) {
			first = d;
			link[(int)link[s]] ^= ((link[(int)link[s]] ^ ((d & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
			link[d] = link[s];
			return;
		}
		if (last == s) {
			last = d;
			link[(int)(link[s] >>> 32)] ^= ((link[(int)(link[s] >>> 32)] ^ (d & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[d] = link[s];
			return;
		}
		final long links = link[s];
		final int prev = (int)(links >>> 32);
		final int next = (int)links;
		link[prev] ^= ((link[prev] ^ (d & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
		link[next] ^= ((link[next] ^ ((d & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
		link[d] = links;
	}

	/**
	 * Returns the first key of this map in iteration order.
	 *
	 * @return the first key in iteration order.
	 */
	@Override
	public K firstKey() {
		if (size == 0) throw new NoSuchElementException();
		return key[first];
	}

	/**
	 * Returns the last key of this map in iteration order.
	 *
	 * @return the last key in iteration order.
	 */
	@Override
	public K lastKey() {
		if (size == 0) throw new NoSuchElementException();
		return key[last];
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public Object2DoubleSortedMap<K> tailMap(K from) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public Object2DoubleSortedMap<K> headMap(K to) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public Object2DoubleSortedMap<K> subMap(K from, K to) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just returns {@code null}.
	 */
	@Override
	public Comparator<? super K> comparator() {
		return null;
	}

	/**
	 * A list iterator over a linked map.
	 *
	 * <p>
	 * This class provides a list iterator over a linked hash map. The constructor runs in constant
	 * time.
	 */
	private abstract class MapIterator<ConsumerType> {
		/**
		 * The entry that will be returned by the next call to {@link java.util.ListIterator#previous()} (or
		 * {@code null} if no previous entry exists).
		 */
		int prev = -1;
		/**
		 * The entry that will be returned by the next call to {@link java.util.ListIterator#next()} (or
		 * {@code null} if no next entry exists).
		 */
		int next = -1;
		/**
		 * The last entry that was returned (or -1 if we did not iterate or used
		 * {@link java.util.Iterator#remove()}).
		 */
		int curr = -1;
		/**
		 * The current index (in the sense of a {@link java.util.ListIterator}). Note that this value is not
		 * meaningful when this iterator has been created using the nonempty constructor.
		 */
		int index = -1;

		@SuppressWarnings("unused")
		abstract void acceptOnIndex(final ConsumerType action, final int index);

		protected MapIterator() {
			next = first;
			index = 0;
		}

		private MapIterator(final K from) {
			if (((from) == null)) {
				if (Object2DoubleLinkedOpenHashMap.this.containsNullKey) {
					next = (int)link[n];
					prev = n;
					return;
				} else throw new NoSuchElementException("The key " + from + " does not belong to this map.");
			}
			if (java.util.Objects.equals(key[last], from)) {
				prev = last;
				index = size;
				return;
			}
			// The starting point.
			int pos = (it.unimi.dsi.fastutil.HashCommon.mix((from).hashCode())) & mask;
			// There's always an unused entry.
			while (!((key[pos]) == null)) {
				if (((key[pos]).equals(from))) {
					// Note: no valid index known.
					next = (int)link[pos];
					prev = pos;
					return;
				}
				pos = (pos + 1) & mask;
			}
			throw new NoSuchElementException("The key " + from + " does not belong to this map.");
		}

		public boolean hasNext() {
			return next != -1;
		}

		public boolean hasPrevious() {
			return prev != -1;
		}

		private final void ensureIndexKnown() {
			if (index >= 0) return;
			if (prev == -1) {
				index = 0;
				return;
			}
			if (next == -1) {
				index = size;
				return;
			}
			int pos = first;
			index = 1;
			while (pos != prev) {
				pos = (int)link[pos];
				index++;
			}
		}

		public int nextIndex() {
			ensureIndexKnown();
			return index;
		}

		public int previousIndex() {
			ensureIndexKnown();
			return index - 1;
		}

		public int nextEntry() {
			if (!hasNext()) throw new NoSuchElementException();
			curr = next;
			next = (int)link[curr];
			prev = curr;
			if (index >= 0) index++;
			return curr;
		}

		public int previousEntry() {
			if (!hasPrevious()) throw new NoSuchElementException();
			curr = prev;
			prev = (int)(link[curr] >>> 32);
			next = curr;
			if (index >= 0) index--;
			return curr;
		}

		public void forEachRemaining(final ConsumerType action) {
			while (hasNext()) {
				curr = next;
				next = (int)link[curr];
				prev = curr;
				if (index >= 0) index++;
				acceptOnIndex(action, curr);
			}
		}

		public void remove() {
			ensureIndexKnown();
			if (curr == -1) throw new IllegalStateException();
			if (curr == prev) {
				/* If the last operation was a next(), we are removing an entry that preceeds
						   the current index, and thus we must decrement it. */
				index--;
				prev = (int)(link[curr] >>> 32);
			} else next = (int)link[curr];
			size--;
			/* Now we manually fix the pointers. Because of our knowledge of next
				   and prev, this is going to be faster than calling fixPointers(). */
			if (prev == -1) first = next;
			else link[prev] ^= ((link[prev] ^ (next & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			if (next == -1) last = prev;
			else link[next] ^= ((link[next] ^ ((prev & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
			int last, slot, pos = curr;
			curr = -1;
			if (pos == n) {
				Object2DoubleLinkedOpenHashMap.this.containsNullKey = false;
				key[n] = null;
			} else {
				K curr;
				final K[] key = Object2DoubleLinkedOpenHashMap.this.key;
				final double value[] = Object2DoubleLinkedOpenHashMap.this.value;
				// We have to horribly duplicate the shiftKeys() code because we need to update next/prev.
				for (;;) {
					pos = ((last = pos) + 1) & mask;
					for (;;) {
						if (((curr = key[pos]) == null)) {
							key[last] = (null);
							return;
						}
						slot = (it.unimi.dsi.fastutil.HashCommon.mix((curr).hashCode())) & mask;
						if (last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos) break;
						pos = (pos + 1) & mask;
					}
					key[last] = curr;
					value[last] = value[pos];
					if (next == pos) next = last;
					if (prev == pos) prev = last;
					fixPointers(pos, last);
				}
			}
		}

		public int skip(final int n) {
			int i = n;
			while (i-- != 0 && hasNext()) nextEntry();
			return n - i - 1;
		}

		public int back(final int n) {
			int i = n;
			while (i-- != 0 && hasPrevious()) previousEntry();
			return n - i - 1;
		}

		public void set(@SuppressWarnings("unused") Object2DoubleMap.Entry<K> ok) {
			throw new UnsupportedOperationException();
		}

		public void add(@SuppressWarnings("unused") Object2DoubleMap.Entry<K> ok) {
			throw new UnsupportedOperationException();
		}
	}

	private final class EntryIterator extends MapIterator<Consumer<? super Object2DoubleMap.Entry<K>>> implements ObjectListIterator<Object2DoubleMap.Entry<K>> {
		private MapEntry entry;

		public EntryIterator() {
		}

		public EntryIterator(K from) {
			super(from);
		}

		// forEachRemaining inherited from MapIterator superclass.
		@Override
		final void acceptOnIndex(final Consumer<? super Object2DoubleMap.Entry<K>> action, final int index) {
			action.accept(new MapEntry(index));
		}

		@Override
		public MapEntry next() {
			return entry = new MapEntry(nextEntry());
		}

		@Override
		public MapEntry previous() {
			return entry = new MapEntry(previousEntry());
		}

		@Override
		public void remove() {
			super.remove();
			entry.index = -1; // You cannot use a deleted entry.
		}
	}

	private final class FastEntryIterator extends MapIterator<Consumer<? super Object2DoubleMap.Entry<K>>> implements ObjectListIterator<Object2DoubleMap.Entry<K>> {
		final MapEntry entry = new MapEntry();

		public FastEntryIterator() {
		}

		public FastEntryIterator(K from) {
			super(from);
		}

		// forEachRemaining inherited from MapIterator superclass.
		@Override
		final void acceptOnIndex(final Consumer<? super Object2DoubleMap.Entry<K>> action, final int index) {
			entry.index = index;
			action.accept(entry);
		}

		@Override
		public MapEntry next() {
			entry.index = nextEntry();
			return entry;
		}

		@Override
		public MapEntry previous() {
			entry.index = previousEntry();
			return entry;
		}
	}

	private final class MapEntrySet extends AbstractObjectSortedSet<Object2DoubleMap.Entry<K>> implements FastSortedEntrySet<K> {
		private static final int SPLITERATOR_CHARACTERISTICS = ObjectSpliterators.SET_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.ORDERED;

		@Override
		public ObjectBidirectionalIterator<Object2DoubleMap.Entry<K>> iterator() {
			return new EntryIterator();
		}

		/**
		 * {@inheritDoc}
		 *
		 * <p>
		 * There isn't a way to split efficiently while still preserving order for a linked data structure,
		 * so this implementation is just backed by the iterator. Thus, this spliterator is not well
		 * optimized for parallel streams.
		 *
		 * <p>
		 * Note, contrary to the specification of {@link java.util.SortedSet}, this spliterator does not,
		 * report {@link java.util.Spliterators.SORTED}. This is because iteration order is based on
		 * insertion order, not natural ordering.
		 */
		@Override
		public ObjectSpliterator<Object2DoubleMap.Entry<K>> spliterator() {
			return ObjectSpliterators.asSpliterator(iterator(), it.unimi.dsi.fastutil.Size64.sizeOf(Object2DoubleLinkedOpenHashMap.this), SPLITERATOR_CHARACTERISTICS);
		}

		@Override
		public Comparator<? super Object2DoubleMap.Entry<K>> comparator() {
			return null;
		}

		@Override
		public ObjectSortedSet<Object2DoubleMap.Entry<K>> subSet(Object2DoubleMap.Entry<K> fromElement, Object2DoubleMap.Entry<K> toElement) {
			throw new UnsupportedOperationException();
		}

		@Override
		public ObjectSortedSet<Object2DoubleMap.Entry<K>> headSet(Object2DoubleMap.Entry<K> toElement) {
			throw new UnsupportedOperationException();
		}

		@Override
		public ObjectSortedSet<Object2DoubleMap.Entry<K>> tailSet(Object2DoubleMap.Entry<K> fromElement) {
			throw new UnsupportedOperationException();
		}

		@Override
		public Object2DoubleMap.Entry<K> first() {
			if (size == 0) throw new NoSuchElementException();
			return new MapEntry(Object2DoubleLinkedOpenHashMap.this.first);
		}

		@Override
		public Object2DoubleMap.Entry<K> last() {
			if (size == 0) throw new NoSuchElementException();
			return new MapEntry(Object2DoubleLinkedOpenHashMap.this.last);
		}

		@Override
		@SuppressWarnings("unchecked")
		public boolean contains(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>)o;
			if (e.getValue() == null || !(e.getValue() instanceof Double)) return false;
			final K k = ((K)e.getKey());
			final double v = ((Double)(e.getValue())).doubleValue();
			if (((k) == null)) return Object2DoubleLinkedOpenHashMap.this.containsNullKey && (Double.doubleToLongBits(value[n]) == Double.doubleToLongBits(v));
			K curr;
			final K[] key = Object2DoubleLinkedOpenHashMap.this.key;
			int pos;
			// The starting point.
			if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return false;
			if (((k).equals(curr))) return (Double.doubleToLongBits(value[pos]) == Double.doubleToLongBits(v));
			// There's always an unused entry.
			while (true) {
				if (((curr = key[pos = (pos + 1) & mask]) == null)) return false;
				if (((k).equals(curr))) return (Double.doubleToLongBits(value[pos]) == Double.doubleToLongBits(v));
			}
		}

		@Override
		@SuppressWarnings("unchecked")
		public boolean remove(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>)o;
			if (e.getValue() == null || !(e.getValue() instanceof Double)) return false;
			final K k = ((K)e.getKey());
			final double v = ((Double)(e.getValue())).doubleValue();
			if (((k) == null)) {
				if (containsNullKey && (Double.doubleToLongBits(value[n]) == Double.doubleToLongBits(v))) {
					removeNullEntry();
					return true;
				}
				return false;
			}
			K curr;
			final K[] key = Object2DoubleLinkedOpenHashMap.this.key;
			int pos;
			// The starting point.
			if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask]) == null)) return false;
			if (((curr).equals(k))) {
				if ((Double.doubleToLongBits(value[pos]) == Double.doubleToLongBits(v))) {
					removeEntry(pos);
					return true;
				}
				return false;
			}
			while (true) {
				if (((curr = key[pos = (pos + 1) & mask]) == null)) return false;
				if (((curr).equals(k))) {
					if ((Double.doubleToLongBits(value[pos]) == Double.doubleToLongBits(v))) {
						removeEntry(pos);
						return true;
					}
				}
			}
		}

		@Override
		public int size() {
			return size;
		}

		@Override
		public void clear() {
			Object2DoubleLinkedOpenHashMap.this.clear();
		}

		/**
		 * Returns a type-specific list iterator on the elements in this set, starting from a given element
		 * of the set. Please see the class documentation for implementation details.
		 *
		 * @param from an element to start from.
		 * @return a type-specific list iterator starting at the given element.
		 * @throws IllegalArgumentException if {@code from} does not belong to the set.
		 */
		@Override
		public ObjectListIterator<Object2DoubleMap.Entry<K>> iterator(final Object2DoubleMap.Entry<K> from) {
			return new EntryIterator(from.getKey());
		}

		/**
		 * Returns a type-specific fast list iterator on the elements in this set, starting from the first
		 * element. Please see the class documentation for implementation details.
		 *
		 * @return a type-specific list iterator starting at the first element.
		 */
		@Override
		public ObjectListIterator<Object2DoubleMap.Entry<K>> fastIterator() {
			return new FastEntryIterator();
		}

		/**
		 * Returns a type-specific fast list iterator on the elements in this set, starting from a given
		 * element of the set. Please see the class documentation for implementation details.
		 *
		 * @param from an element to start from.
		 * @return a type-specific list iterator starting at the given element.
		 * @throws IllegalArgumentException if {@code from} does not belong to the set.
		 */
		@Override
		public ObjectListIterator<Object2DoubleMap.Entry<K>> fastIterator(final Object2DoubleMap.Entry<K> from) {
			return new FastEntryIterator(from.getKey());
		}

		/** {@inheritDoc} */
		@Override
		public void forEach(final Consumer<? super Object2DoubleMap.Entry<K>> consumer) {
			final long[] link = Object2DoubleLinkedOpenHashMap.this.link;
			for (int i = size, curr, next = first; i-- != 0;) {
				curr = next;
				next = (int)link[curr];
				consumer.accept(new MapEntry(curr));
			}
		}

		/** {@inheritDoc} */
		@Override
		public void fastForEach(final Consumer<? super Object2DoubleMap.Entry<K>> consumer) {
			final MapEntry entry = new MapEntry();
			final long[] link = Object2DoubleLinkedOpenHashMap.this.link;
			for (int i = size, next = first; i-- != 0;) {
				entry.index = next;
				next = (int)link[next];
				consumer.accept(entry);
			}
		}
	}

	@Override
	public FastSortedEntrySet<K> object2DoubleEntrySet() {
		if (entries == null) entries = new MapEntrySet();
		return entries;
	}

	/**
	 * An iterator on keys.
	 *
	 * <p>
	 * We simply override the
	 * {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods (and
	 * possibly their type-specific counterparts) so that they return keys instead of entries.
	 */
	private final class KeyIterator extends MapIterator<Consumer<? super K>> implements ObjectListIterator<K> {
		public KeyIterator(final K k) {
			super(k);
		}

		@Override
		public K previous() {
			return key[previousEntry()];
		}

		public KeyIterator() {
			super();
		}

		// forEachRemaining inherited from MapIterator superclass.
		// Despite the superclass declared with generics, the way Java inherits and generates bridge methods
		// avoids the boxing/unboxing
		@Override
		final void acceptOnIndex(final Consumer<? super K> action, final int index) {
			action.accept(key[index]);
		}

		@Override
		public K next() {
			return key[nextEntry()];
		}
	}

	private final class KeySet extends AbstractObjectSortedSet<K> {
		private static final int SPLITERATOR_CHARACTERISTICS = ObjectSpliterators.SET_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.ORDERED;

		@Override
		public ObjectListIterator<K> iterator(final K from) {
			return new KeyIterator(from);
		}

		@Override
		public ObjectListIterator<K> iterator() {
			return new KeyIterator();
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see EntrySet#spliterator()
		 */
		@Override
		public ObjectSpliterator<K> spliterator() {
			return ObjectSpliterators.asSpliterator(iterator(), it.unimi.dsi.fastutil.Size64.sizeOf(Object2DoubleLinkedOpenHashMap.this), SPLITERATOR_CHARACTERISTICS);
		}

		/** {@inheritDoc} */
		@Override
		public void forEach(final Consumer<? super K> consumer) {
			final long[] link = Object2DoubleLinkedOpenHashMap.this.link;
			final K key[] = Object2DoubleLinkedOpenHashMap.this.key;
			for (int i = size, curr, next = first; i-- != 0;) {
				curr = next;
				next = (int)link[curr];
				consumer.accept(key[curr]);
			}
		}

		@Override
		public int size() {
			return size;
		}

		@Override
		public boolean contains(Object k) {
			return containsKey(k);
		}

		@Override
		public boolean remove(Object k) {
			final int oldSize = size;
			Object2DoubleLinkedOpenHashMap.this.removeDouble(k);
			return size != oldSize;
		}

		@Override
		public void clear() {
			Object2DoubleLinkedOpenHashMap.this.clear();
		}

		@Override
		public K first() {
			if (size == 0) throw new NoSuchElementException();
			return key[first];
		}

		@Override
		public K last() {
			if (size == 0) throw new NoSuchElementException();
			return key[last];
		}

		@Override
		public Comparator<? super K> comparator() {
			return null;
		}

		@Override
		public ObjectSortedSet<K> tailSet(K from) {
			throw new UnsupportedOperationException();
		}

		@Override
		public ObjectSortedSet<K> headSet(K to) {
			throw new UnsupportedOperationException();
		}

		@Override
		public ObjectSortedSet<K> subSet(K from, K to) {
			throw new UnsupportedOperationException();
		}
	}

	@Override
	public ObjectSortedSet<K> keySet() {
		if (keys == null) keys = new KeySet();
		return keys;
	}

	/**
	 * An iterator on values.
	 *
	 * <p>
	 * We simply override the
	 * {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods (and
	 * possibly their type-specific counterparts) so that they return values instead of entries.
	 */
	private final class ValueIterator extends MapIterator<java.util.function.DoubleConsumer> implements DoubleListIterator {
		@Override
		public double previousDouble() {
			return value[previousEntry()];
		}

		public ValueIterator() {
			super();
		}

		// forEachRemaining inherited from MapIterator superclass.
		// Despite the superclass declared with generics, the way Java inherits and generates bridge methods
		// avoids the boxing/unboxing
		@Override
		final void acceptOnIndex(final java.util.function.DoubleConsumer action, final int index) {
			action.accept(value[index]);
		}

		@Override
		public double nextDouble() {
			return value[nextEntry()];
		}
	}

	@Override
	public DoubleCollection values() {
		if (values == null) values = new AbstractDoubleCollection() {
			private static final int SPLITERATOR_CHARACTERISTICS = DoubleSpliterators.COLLECTION_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.ORDERED;

			@Override
			public DoubleIterator iterator() {
				return new ValueIterator();
			}

			/**
			 * {@inheritDoc}
			 * 
			 * @see EntrySet#spliterator()
			 */
			@Override
			public DoubleSpliterator spliterator() {
				return DoubleSpliterators.asSpliterator(iterator(), it.unimi.dsi.fastutil.Size64.sizeOf(Object2DoubleLinkedOpenHashMap.this), SPLITERATOR_CHARACTERISTICS);
			}

			/** {@inheritDoc} */
			@Override
			public void forEach(final java.util.function.DoubleConsumer consumer) {
				final long[] link = Object2DoubleLinkedOpenHashMap.this.link;
				final double value[] = Object2DoubleLinkedOpenHashMap.this.value;
				for (int i = size, curr, next = first; i-- != 0;) {
					curr = next;
					next = (int)link[curr];
					consumer.accept(value[curr]);
				}
			}

			@Override
			public int size() {
				return size;
			}

			@Override
			public boolean contains(double v) {
				return containsValue(v);
			}

			@Override
			public void clear() {
				Object2DoubleLinkedOpenHashMap.this.clear();
			}
		};
		return values;
	}

	/**
	 * Rehashes the map, making the table as small as possible.
	 *
	 * <p>
	 * This method rehashes the table to the smallest size satisfying the load factor. It can be used
	 * when the set will not be changed anymore, so to optimize access speed and size.
	 *
	 * <p>
	 * If the table size is already the minimum possible, this method does nothing.
	 *
	 * @return true if there was enough memory to trim the map.
	 * @see #trim(int)
	 */
	public boolean trim() {
		return trim(size);
	}

	/**
	 * Rehashes this map if the table is too large.
	 *
	 * <p>
	 * Let <var>N</var> be the smallest table size that can hold <code>max(n,{@link #size()})</code>
	 * entries, still satisfying the load factor. If the current table size is smaller than or equal to
	 * <var>N</var>, this method does nothing. Otherwise, it rehashes this map in a table of size
	 * <var>N</var>.
	 *
	 * <p>
	 * This method is useful when reusing maps. {@linkplain #clear() Clearing a map} leaves the table
	 * size untouched. If you are reusing a map many times, you can call this method with a typical size
	 * to avoid keeping around a very large table just because of a few large transient maps.
	 *
	 * @param n the threshold for the trimming.
	 * @return true if there was enough memory to trim the map.
	 * @see #trim()
	 */
	public boolean trim(final int n) {
		final int l = HashCommon.nextPowerOfTwo((int)Math.ceil(n / f));
		if (l >= this.n || size > maxFill(l, f)) return true;
		try {
			rehash(l);
		} catch (OutOfMemoryError cantDoIt) {
			return false;
		}
		return true;
	}

	/**
	 * Rehashes the map.
	 *
	 * <p>
	 * This method implements the basic rehashing strategy, and may be overridden by subclasses
	 * implementing different rehashing strategies (e.g., disk-based rehashing). However, you should not
	 * override this method unless you understand the internal workings of this class.
	 *
	 * @param newN the new size
	 */
	@SuppressWarnings("unchecked")
	protected void rehash(final int newN) {
		final K key[] = this.key;
		final double value[] = this.value;
		final int mask = newN - 1; // Note that this is used by the hashing macro
		final K newKey[] = (K[])new Object[newN + 1];
		final double newValue[] = new double[newN + 1];
		int i = first, prev = -1, newPrev = -1, t, pos;
		final long link[] = this.link;
		final long newLink[] = new long[newN + 1];
		first = -1;
		for (int j = size; j-- != 0;) {
			if (((key[i]) == null)) pos = newN;
			else {
				pos = (it.unimi.dsi.fastutil.HashCommon.mix((key[i]).hashCode())) & mask;
				while (!((newKey[pos]) == null)) pos = (pos + 1) & mask;
			}
			newKey[pos] = key[i];
			newValue[pos] = value[i];
			if (prev != -1) {
				newLink[newPrev] ^= ((newLink[newPrev] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
				newLink[pos] ^= ((newLink[pos] ^ ((newPrev & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
				newPrev = pos;
			} else {
				newPrev = first = pos;
				// Special case of SET(newLink[pos], -1, -1);
				newLink[pos] = -1L;
			}
			t = i;
			i = (int)link[i];
			prev = t;
		}
		this.link = newLink;
		this.last = newPrev;
		if (newPrev != -1)
			// Special case of SET_NEXT(newLink[newPrev], -1);
			newLink[newPrev] |= -1 & 0xFFFFFFFFL;
		n = newN;
		this.mask = mask;
		maxFill = maxFill(n, f);
		this.key = newKey;
		this.value = newValue;
	}

	/**
	 * Returns a deep copy of this map.
	 *
	 * <p>
	 * This method performs a deep copy of this hash map; the data stored in the map, however, is not
	 * cloned. Note that this makes a difference only for object keys.
	 *
	 * @return a deep copy of this map.
	 */
	@Override
	@SuppressWarnings("unchecked")
	public Object2DoubleLinkedOpenHashMap<K> clone() {
		Object2DoubleLinkedOpenHashMap<K> c;
		try {
			c = (Object2DoubleLinkedOpenHashMap<K>)super.clone();
		} catch (CloneNotSupportedException cantHappen) {
			throw new InternalError();
		}
		c.keys = null;
		c.values = null;
		c.entries = null;
		c.containsNullKey = containsNullKey;
		c.key = key.clone();
		c.value = value.clone();
		c.link = link.clone();
		return c;
	}

	/**
	 * Returns a hash code for this map.
	 *
	 * This method overrides the generic method provided by the superclass. Since {@code equals()} is
	 * not overriden, it is important that the value returned by this method is the same value as the
	 * one returned by the overriden method.
	 *
	 * @return a hash code for this map.
	 */
	@Override
	public int hashCode() {
		int h = 0;
		final K key[] = this.key;
		final double value[] = this.value;
		for (int j = realSize(), i = 0, t = 0; j-- != 0;) {
			while (((key[i]) == null)) i++;
			if (this != key[i]) t = ((key[i]).hashCode());
			t ^= it.unimi.dsi.fastutil.HashCommon.double2int(value[i]);
			h += t;
			i++;
		}
		// Zero / null keys have hash zero.
		if (containsNullKey) h += it.unimi.dsi.fastutil.HashCommon.double2int(value[n]);
		return h;
	}

	private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException {
		final K key[] = this.key;
		final double value[] = this.value;
		final EntryIterator i = new EntryIterator();
		s.defaultWriteObject();
		for (int j = size, e; j-- != 0;) {
			e = i.nextEntry();
			s.writeObject(key[e]);
			s.writeDouble(value[e]);
		}
	}

	@SuppressWarnings("unchecked")
	private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException {
		s.defaultReadObject();
		n = arraySize(size, f);
		maxFill = maxFill(n, f);
		mask = n - 1;
		final K key[] = this.key = (K[])new Object[n + 1];
		final double value[] = this.value = new double[n + 1];
		final long link[] = this.link = new long[n + 1];
		int prev = -1;
		first = last = -1;
		K k;
		double v;
		for (int i = size, pos; i-- != 0;) {
			k = (K)s.readObject();
			v = s.readDouble();
			if (((k) == null)) {
				pos = n;
				containsNullKey = true;
			} else {
				pos = (it.unimi.dsi.fastutil.HashCommon.mix((k).hashCode())) & mask;
				while (!((key[pos]) == null)) pos = (pos + 1) & mask;
			}
			key[pos] = k;
			value[pos] = v;
			if (first != -1) {
				link[prev] ^= ((link[prev] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
				link[pos] ^= ((link[pos] ^ ((prev & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
				prev = pos;
			} else {
				prev = first = pos;
				// Special case of SET_PREV(newLink[pos], -1);
				link[pos] |= (-1L & 0xFFFFFFFFL) << 32;
			}
		}
		last = prev;
		if (prev != -1)
			// Special case of SET_NEXT(link[prev], -1);
			link[prev] |= -1 & 0xFFFFFFFFL;
		if (ASSERTS) checkTable();
	}

	private void checkTable() {
	}
}
