package com.gtnewhorizon.gtnhlib.client.renderer;

import static com.gtnewhorizon.gtnhlib.client.renderer.cel.util.ModelQuadUtil.COLOR_INDEX;
import static com.gtnewhorizon.gtnhlib.client.renderer.cel.util.ModelQuadUtil.LIGHT_INDEX;
import static com.gtnewhorizon.gtnhlib.client.renderer.cel.util.ModelQuadUtil.NORMAL_INDEX;
import static com.gtnewhorizon.gtnhlib.client.renderer.cel.util.ModelQuadUtil.POSITION_INDEX;
import static com.gtnewhorizon.gtnhlib.client.renderer.cel.util.ModelQuadUtil.TEXTURE_INDEX;
import static net.minecraft.util.MathHelper.clamp_int;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import net.minecraft.client.renderer.Tessellator;

import org.joml.Matrix3f;
import org.joml.Vector3f;
import org.lwjgl.BufferUtils;
import org.lwjgl.opengl.GL11;

import com.gtnewhorizon.gtnhlib.blockpos.BlockPos;
import com.gtnewhorizon.gtnhlib.client.renderer.cel.model.line.ModelLine;
import com.gtnewhorizon.gtnhlib.client.renderer.cel.model.primitive.ModelPrimitiveView;
import com.gtnewhorizon.gtnhlib.client.renderer.cel.model.quad.ModelQuad;
import com.gtnewhorizon.gtnhlib.client.renderer.cel.model.quad.ModelQuadViewMutable;
import com.gtnewhorizon.gtnhlib.client.renderer.cel.model.tri.ModelTriangle;
import com.gtnewhorizon.gtnhlib.client.renderer.stacks.Vector3dStack;
import com.gtnewhorizon.gtnhlib.client.renderer.vertex.VertexFormat;
import com.gtnewhorizon.gtnhlib.util.ObjectPooler;

import it.unimi.dsi.fastutil.objects.ObjectArrayList;

/// To be used in conjunction with the {@link TessellatorManager}. Used to capture the quads generated by the
/// {@link Tessellator} across multiple draw calls and make the quad list available for usage.
///
/// NOTE: This will _not_ (currently) capture, integrate, or stop any GL calls made around the tessellator draw calls.
@SuppressWarnings("unused")
public class CapturingTessellator extends Tessellator implements ITessellatorInstance {

    // Object pools reduce allocations over time by reusing primitive instances across multiple
    // capture sessions. Not meant to avoid allocations within a single call, but to amortize
    // allocation costs across the lifetime of the CapturingTessellator.
    final ObjectPooler<ModelQuad> quadPool = new ObjectPooler<>(ModelQuad::new);
    final ObjectPooler<ModelTriangle> triPool = new ObjectPooler<>(ModelTriangle::new);
    final ObjectPooler<ModelLine> linePool = new ObjectPooler<>(ModelLine::new);

    /**
     * Stores ModelQuad objects for GL_QUADS and GL_TRIANGLES draw modes. Triangles are stored as "degenerate quads" via
     * quadrangulation (v2 duplicated to v3) for backward compatibility with code expecting all geometry in this list.
     * Populated by QuadExtractor.
     */
    final List<ModelQuadViewMutable> collectedQuads = new ObjectArrayList<>();

    /**
     * Stores primitives from other draw modes: GL_LINES, GL_LINE_STRIP, GL_LINE_LOOP (as ModelLine), GL_TRIANGLE_STRIP,
     * GL_TRIANGLE_FAN (as ModelTriangle). These are converted to their base primitive types. Populated by
     * PrimitiveExtractor.
     */
    final List<ModelPrimitiveView> collectedPrimitives = new ObjectArrayList<>();

    // Reusable lists for stopCapturingToGeometry to avoid allocations
    final List<ModelLine> lineListCache = new ArrayList<>();
    final List<ModelTriangle> triangleListCache = new ArrayList<>();
    final List<ModelQuad> quadListCache = new ArrayList<>();

    int shaderBlockId = -1;

    // Any offset we need to the Tesselator's offset!
    final BlockPos offset = new BlockPos();

    // Reusable FLAGS instance to avoid allocations
    final Flags flags = new Flags(true, true, true, true);

    private final Vector3dStack storedTranslation = new Vector3dStack();

    public void setOffset(BlockPos pos) {
        this.offset.set(pos);
    }

    public void resetOffset() {
        this.offset.zero();
    }

    @Override
    public int draw() {
        // Delegate to TessellatorManager for shared logic
        return TessellatorManager.processDrawForCapturingTessellator(this);
    }

    @Override
    public void discard() {
        isDrawing = false;
        reset();
    }

    @Override
    public boolean gtnhlib$isCompiling() {
        // Check if the current state in TessellatorManager is COMPILING
        return TessellatorManager.isCurrentlyCompiling();
    }

    @Override
    public void gtnhlib$setCompiling(boolean compiling) {
        // NOOP - CapturingTessellator never needs the compiling flag (it's only for vanilla Tessellator)
    }

    public List<ModelQuadViewMutable> getQuads() {
        return collectedQuads;
    }

    public List<ModelPrimitiveView> getPrimitives() {
        return collectedPrimitives;
    }

    public void clearQuads() {
        // noinspection ForLoopReplaceableByForEach
        for (int i = 0; i < collectedQuads.size(); i++) {
            final var quad = collectedQuads.get(i);
            if (quad instanceof ModelQuad mq) quadPool.releaseInstance(mq);
        }
        collectedQuads.clear();
    }

    public void clearPrimitives() {
        // noinspection ForLoopReplaceableByForEach
        for (int i = 0; i < collectedPrimitives.size(); i++) {
            final var prim = collectedPrimitives.get(i);
            if (prim instanceof ModelQuad mq) {
                quadPool.releaseInstance(mq);
            } else if (prim instanceof ModelTriangle mt) {
                triPool.releaseInstance(mt);
            } else if (prim instanceof ModelLine ml) {
                linePool.releaseInstance(ml);
            }
        }
        collectedPrimitives.clear();
    }

    public static ByteBuffer quadsToBuffer(List<ModelQuadViewMutable> quads, VertexFormat format) {
        if (!format.canWriteQuads()) {
            throw new IllegalStateException("Vertex format has no quad writer: " + format);
        }
        final ByteBuffer byteBuffer = BufferUtils.createByteBuffer(format.getVertexSize() * quads.size() * 4);
        // noinspection ForLoopReplaceableByForEach
        for (int i = 0, quadsSize = quads.size(); i < quadsSize; i++) {
            format.writeQuad(quads.get(i), byteBuffer);
        }
        byteBuffer.rewind();
        return byteBuffer;
    }

    public void storeTranslation() {
        storedTranslation.push();

        this.storedTranslation.set(xOffset, yOffset, zOffset);
    }

    public void restoreTranslation() {

        xOffset = storedTranslation.x;
        yOffset = storedTranslation.y;
        zOffset = storedTranslation.z;
        storedTranslation.pop();
    }

    public static int createBrightness(int sky, int block) {
        return sky << 20 | block << 4;
    }

    // API from newer MC
    public CapturingTessellator pos(double x, double y, double z) {
        ensureBuffer();

        this.rawBuffer[this.rawBufferIndex + POSITION_INDEX + 0] = Float.floatToRawIntBits((float) (x + this.xOffset));
        this.rawBuffer[this.rawBufferIndex + POSITION_INDEX + 1] = Float.floatToRawIntBits((float) (y + this.yOffset));
        this.rawBuffer[this.rawBufferIndex + POSITION_INDEX + 2] = Float.floatToRawIntBits((float) (z + this.zOffset));

        return this;
    }

    public CapturingTessellator tex(double u, double v) {
        this.rawBuffer[this.rawBufferIndex + TEXTURE_INDEX] = Float.floatToRawIntBits((float) u);
        this.rawBuffer[this.rawBufferIndex + TEXTURE_INDEX + 1] = Float.floatToRawIntBits((float) v);
        this.hasTexture = true;

        return this;
    }

    public CapturingTessellator color(float red, float green, float blue, float alpha) {
        return this.color((int) (red * 255.0F), (int) (green * 255.0F), (int) (blue * 255.0F), (int) (alpha * 255.0F));
    }

    public CapturingTessellator color(int red, int green, int blue, int alpha) {
        if (this.isColorDisabled) return this;
        red = clamp_int(red, 0, 255);
        green = clamp_int(green, 0, 255);
        blue = clamp_int(blue, 0, 255);
        alpha = clamp_int(alpha, 0, 255);

        final int color;
        if (ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN) {
            color = alpha << 24 | blue << 16 | green << 8 | red;
        } else {
            color = red << 24 | green << 16 | blue << 8 | alpha;
        }
        this.rawBuffer[this.rawBufferIndex + COLOR_INDEX] = color;
        this.hasColor = true;

        return this;

    }

    public CapturingTessellator normal(float x, float y, float z) {
        final byte b0 = (byte) ((int) (x * 127.0F));
        final byte b1 = (byte) ((int) (y * 127.0F));
        final byte b2 = (byte) ((int) (z * 127.0F));

        this.rawBuffer[this.rawBufferIndex + NORMAL_INDEX] = b0 & 255 | (b1 & 255) << 8 | (b2 & 255) << 16;
        this.hasNormals = true;
        return this;
    }

    /**
     * Sets the normal based on a given normal matrix
     *
     * @param normal       The normal vector
     * @param dest         The vector that gets transformed
     * @param normalMatrix The normal matrix (typically the transpose of the inverse transformation matrix)
     */
    public CapturingTessellator setNormalTransformed(Vector3f normal, Vector3f dest, Matrix3f normalMatrix) {
        normalMatrix.transform(normal, dest).normalize();
        this.setNormal(dest.x, dest.y, dest.z);
        return this;
    }

    /**
     * Same as the method above, but this one will mutate the passed Vector3f
     */
    public CapturingTessellator setNormalTransformed(Vector3f normal, Matrix3f normalMatrix) {
        return setNormalTransformed(normal, normal, normalMatrix);
    }

    public CapturingTessellator lightmap(int skyLight, int blockLight) {
        return brightness(createBrightness(skyLight, blockLight));
    }

    public CapturingTessellator brightness(int brightness) {
        this.rawBuffer[this.rawBufferIndex + LIGHT_INDEX] = brightness;
        this.hasBrightness = true;

        return this;
    }

    public CapturingTessellator endVertex() {
        this.rawBufferIndex += 8;
        ++this.vertexCount;

        return this;
    }

    public void ensureBuffer() {
        if (rawBufferIndex >= rawBufferSize - 32) {
            if (rawBufferSize == 0) {
                rawBufferSize = 0x10000;
                rawBuffer = new int[rawBufferSize];
            } else {
                rawBufferSize *= 2;
                rawBuffer = Arrays.copyOf(rawBuffer, rawBufferSize);
            }
        }
    }

    public void setShaderBlockId(int blockId) {
        // Flush queue, so we capture the blockId in quads before we change it
        if (isDrawing) {
            draw();
            isDrawing = true;
        }

        // Now set new blockId
        shaderBlockId = blockId;
    }

    public static class Flags {

        public boolean hasTexture;
        public boolean hasBrightness;
        public boolean hasColor;
        public boolean hasNormals;
        public int drawMode = GL11.GL_QUADS;

        public Flags(boolean hasTexture, boolean hasBrightness, boolean hasColor, boolean hasNormals) {
            this.hasTexture = hasTexture;
            this.hasBrightness = hasBrightness;
            this.hasColor = hasColor;
            this.hasNormals = hasNormals;
        }

        /**
         * Copy constructor for creating immutable snapshots of flag state.
         */
        public Flags(Flags other) {
            this.hasTexture = other.hasTexture;
            this.hasBrightness = other.hasBrightness;
            this.hasColor = other.hasColor;
            this.hasNormals = other.hasNormals;
            this.drawMode = other.drawMode;
        }

        /**
         * Updates this Flags instance with new values. Used to avoid repeated field assignments.
         */
        public void copyFrom(boolean hasTexture, boolean hasBrightness, boolean hasColor, boolean hasNormals) {
            this.hasTexture = hasTexture;
            this.hasBrightness = hasBrightness;
            this.hasColor = hasColor;
            this.hasNormals = hasNormals;
        }

        /**
         * Updates this Flags instance with new values including draw mode.
         */
        public void copyFrom(boolean hasTexture, boolean hasBrightness, boolean hasColor, boolean hasNormals,
                int drawMode) {
            this.hasTexture = hasTexture;
            this.hasBrightness = hasBrightness;
            this.hasColor = hasColor;
            this.hasNormals = hasNormals;
            this.drawMode = drawMode;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (!(o instanceof Flags flags)) return false;
            return hasTexture == flags.hasTexture && hasBrightness == flags.hasBrightness
                    && hasColor == flags.hasColor
                    && hasNormals == flags.hasNormals
                    && drawMode == flags.drawMode;
        }

        @Override
        public int hashCode() {
            return Objects.hash(hasTexture, hasBrightness, hasColor, hasNormals, drawMode);
        }

    }
}
