/*
	* Copyright (C) 2002-2024 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.longs;

import java.util.Comparator;
import java.util.Objects;
import java.io.Serializable;

/**
 * A type-specific {@link Comparator}; provides methods to compare two primitive types both as
 * objects and as primitive types.
 *
 * <p>
 * Note that {@code fastutil} provides a corresponding abstract class that can be used to implement
 * this interface just by specifying the type-specific comparator.
 *
 * @see Comparator
 */
@FunctionalInterface
public interface LongComparator extends Comparator<Long> {
	/**
	 * Compares its two primitive-type arguments for order. Returns a negative integer, zero, or a
	 * positive integer as the first argument is less than, equal to, or greater than the second.
	 *
	 * @see java.util.Comparator
	 * @return a negative integer, zero, or a positive integer as the first argument is less than, equal
	 *         to, or greater than the second.
	 */
	int compare(long k1, long k2);

	@Override
	default LongComparator reversed() {
		return LongComparators.oppositeComparator(this);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	default int compare(Long ok1, Long ok2) {
		return compare(ok1.longValue(), ok2.longValue());
	}

	/**
	 * Return a new comparator that first uses this comparator, then uses the second comparator if this
	 * comparator compared the two elements as equal.
	 *
	 * @see Comparator#thenComparing(Comparator)
	 */
	default LongComparator thenComparing(LongComparator second) {
		return (LongComparator & java.io.Serializable)(k1, k2) -> {
			int comp = compare(k1, k2);
			return comp == 0 ? second.compare(k1, k2) : comp;
		};
	}

	@Override
	default Comparator<Long> thenComparing(Comparator<? super Long> second) {
		if (second instanceof LongComparator) return thenComparing((LongComparator)second);
		return Comparator.super.thenComparing(second);
	}

	/**
	 * Accepts a function that extracts a {@link java.lang.Comparable Comparable} sort key from a
	 * primitive key, and returns a comparator that compares by that sort key.
	 *
	 * <p>
	 * The returned comparator is serializable if the specified function is also serializable.
	 *
	 * @param keyExtractor the function used to extract the {@link Comparable} sort key
	 * @return a comparator that compares by an extracted key
	 * @throws NullPointerException if {@code keyExtractor} is {@code null}
	 */
	static <U extends Comparable<? super U>> LongComparator comparing(Long2ObjectFunction<? extends U> keyExtractor) {
		Objects.requireNonNull(keyExtractor);
		return (LongComparator & Serializable)(k1, k2) -> keyExtractor.get(k1).compareTo(keyExtractor.get(k2));
	}

	/**
	 * Accepts a function that extracts a sort key from a primitive key, and returns a comparator that
	 * compares by that sort key using the specified {@link Comparator}.
	 *
	 * <p>
	 * The returned comparator is serializable if the specified function and comparator are both
	 * serializable.
	 *
	 * @param keyExtractor the function used to extract the sort key
	 * @param keyComparator the {@code Comparator} used to compare the sort key
	 * @return a comparator that compares by an extracted key using the specified {@code Comparator}
	 * @throws NullPointerException if {@code keyExtractor} or {@code keyComparator} are {@code null}
	 */
	static <U extends Comparable<? super U>> LongComparator comparing(Long2ObjectFunction<? extends U> keyExtractor, Comparator<? super U> keyComparator) {
		Objects.requireNonNull(keyExtractor);
		Objects.requireNonNull(keyComparator);
		return (LongComparator & Serializable)(k1, k2) -> keyComparator.compare(keyExtractor.get(k1), keyExtractor.get(k2));
	}

	/**
	 * Accepts a function that extracts an {@code int} sort key from a primitive key, and returns a
	 * comparator that compares by that sort key.
	 *
	 * <p>
	 * The returned comparator is serializable if the specified function is also serializable.
	 *
	 * @param keyExtractor the function used to extract the integer sort key
	 * @return a comparator that compares by an extracted key
	 * @throws NullPointerException if {@code keyExtractor} is {@code null}
	 */
	static LongComparator comparingInt(Long2IntFunction keyExtractor) {
		Objects.requireNonNull(keyExtractor);
		return (LongComparator & Serializable)(k1, k2) -> Integer.compare(keyExtractor.get(k1), keyExtractor.get(k2));
	}

	/**
	 * Accepts a function that extracts an {@code long} sort key from a primitive key, and returns a
	 * comparator that compares by that sort key.
	 *
	 * <p>
	 * The returned comparator is serializable if the specified function is also serializable.
	 *
	 * @param keyExtractor the function used to extract the long sort key
	 * @return a comparator that compares by an extracted key
	 * @throws NullPointerException if {@code keyExtractor} is {@code null}
	 */
	static LongComparator comparingLong(Long2LongFunction keyExtractor) {
		Objects.requireNonNull(keyExtractor);
		return (LongComparator & Serializable)(k1, k2) -> Long.compare(keyExtractor.get(k1), keyExtractor.get(k2));
	}

	/**
	 * Accepts a function that extracts an {@code double} sort key from a primitive key, and returns a
	 * comparator that compares by that sort key.
	 *
	 * <p>
	 * The returned comparator is serializable if the specified function is also serializable.
	 *
	 * @param keyExtractor the function used to extract the double sort key
	 * @return a comparator that compares by an extracted key
	 * @throws NullPointerException if {@code keyExtractor} is {@code null}
	 */
	static LongComparator comparingDouble(Long2DoubleFunction keyExtractor) {
		Objects.requireNonNull(keyExtractor);
		return (LongComparator & Serializable)(k1, k2) -> Double.compare(keyExtractor.get(k1), keyExtractor.get(k2));
	}
}
