/*
 * Decompiled with CFR 0.152.
 */
package com.oheers.fish.api;

import com.oheers.fish.api.plugin.EMFPlugin;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarInputStream;
import java.util.logging.Level;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import org.bukkit.plugin.Plugin;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class FileUtil {
    private static final Map<File, URLClassLoader> fileClassLoaders = new HashMap<File, URLClassLoader>();

    private FileUtil() {
        throw new UnsupportedOperationException();
    }

    @NotNull
    public static <T> List<Class<? extends T>> findClasses(@NotNull File file, @NotNull Class<T> clazz) {
        if (!file.exists()) {
            return Collections.emptyList();
        }
        ArrayList<Class<T>> classes = new ArrayList<Class<T>>();
        List<String> matches = FileUtil.matchingNames(file);
        for (String match : matches) {
            try {
                URL jar = file.toURI().toURL();
                URLClassLoader loader = fileClassLoaders.computeIfAbsent(file, f -> new URLClassLoader(new URL[]{jar}, clazz.getClassLoader()));
                Class<T> addonClass = FileUtil.loadClass(loader, match, clazz);
                if (addonClass == null) continue;
                classes.add(addonClass);
            }
            catch (VerifyError ex) {
                EMFPlugin.getInstance().getLogger().severe(() -> String.format("Failed to load addon class %s", file.getName()));
                EMFPlugin.getInstance().getLogger().severe(() -> String.format("Cause: %s %s", ex.getClass().getSimpleName(), ex.getMessage()));
            }
            catch (IOException | ClassNotFoundException e) {
                throw new RuntimeException(e);
            }
        }
        return classes;
    }

    @NotNull
    private static List<String> matchingNames(File file) {
        ArrayList<String> matches = new ArrayList<String>();
        try {
            URL jar = file.toURI().toURL();
            try (JarInputStream stream = new JarInputStream(jar.openStream());){
                JarEntry entry;
                while ((entry = stream.getNextJarEntry()) != null) {
                    String name = entry.getName();
                    if (!name.endsWith(".class")) continue;
                    matches.add(name.substring(0, name.lastIndexOf(46)).replace('/', '.'));
                }
            }
        }
        catch (Exception e) {
            return Collections.emptyList();
        }
        return matches;
    }

    @Nullable
    private static <T> Class<? extends T> loadClass(@NotNull URLClassLoader loader, String match, @NotNull Class<T> clazz) throws ClassNotFoundException {
        try {
            Class<?> loaded = loader.loadClass(match);
            if (clazz.isAssignableFrom(loaded)) {
                return loaded.asSubclass(clazz);
            }
        }
        catch (NoClassDefFoundError loaded) {
        }
        catch (UnsupportedClassVersionError e) {
            EMFPlugin.getInstance().debug(e.getMessage());
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Optional<File> loadFileOrResource(@NotNull File directory, @NotNull String fileName, @NotNull String resourceName, @NotNull Plugin plugin, boolean overwrite) {
        Objects.requireNonNull(directory, "directory cannot be null");
        Objects.requireNonNull(fileName, "fileName cannot be null");
        Objects.requireNonNull(resourceName, "resourceName cannot be null");
        Objects.requireNonNull(plugin, "plugin cannot be null");
        if (!directory.exists() && !directory.mkdirs()) {
            plugin.getLogger().severe(() -> "Could not create directory: " + String.valueOf(directory));
            return Optional.empty();
        }
        File configFile = new File(directory, fileName);
        if (configFile.exists() && !overwrite) {
            return Optional.of(configFile);
        }
        try {
            if (!configFile.createNewFile()) {
                plugin.getLogger().severe(() -> "Could not create file: " + String.valueOf(configFile));
                return Optional.empty();
            }
        }
        catch (IOException ex) {
            plugin.getLogger().log(Level.SEVERE, "Failed to create file: %s".formatted(configFile.getName()), ex);
            return Optional.empty();
        }
        try (InputStream stream = plugin.getResource(resourceName);){
            if (stream == null) {
                plugin.getLogger().severe(() -> "Resource not found: " + resourceName);
                Optional<File> optional2 = Optional.empty();
                return optional2;
            }
            Files.copy(stream, configFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            Optional<File> optional = Optional.of(configFile);
            return optional;
        }
        catch (IOException ex) {
            plugin.getLogger().log(Level.SEVERE, "Failed to copy resource %s to %s".formatted(resourceName, configFile), ex);
            return Optional.empty();
        }
    }

    public static Optional<File> loadFileOrResource(@NotNull File directory, @NotNull String fileName, @NotNull String resourceName, @NotNull Plugin plugin) {
        return FileUtil.loadFileOrResource(directory, fileName, resourceName, plugin, false);
    }

    public static List<File> getFilesInDirectoryWithExtension(@NotNull File directory, @Nullable String extension, boolean ignoreUnderscoreFiles, boolean recursive) {
        ArrayList<File> finalList = new ArrayList<File>();
        if (!directory.exists() || !directory.isDirectory()) {
            return finalList;
        }
        try {
            FilenameFilter filter = extension == null ? null : (dir, name) -> name.endsWith(extension);
            File[] fileArray = directory.listFiles(filter);
            if (fileArray == null) {
                return finalList;
            }
            for (File file : fileArray) {
                if (ignoreUnderscoreFiles && file.getName().startsWith("_")) continue;
                if (file.isDirectory() && recursive) {
                    finalList.addAll(FileUtil.getFilesInDirectory(file, ignoreUnderscoreFiles, true));
                    continue;
                }
                if (!file.isFile()) continue;
                finalList.add(file);
            }
        }
        catch (SecurityException exception) {
            EMFPlugin.getInstance().getLogger().log(Level.WARNING, "Failed to retrieve files in %s: Access Denied.".formatted(directory.getAbsolutePath()), exception);
        }
        return finalList;
    }

    public static List<File> getFilesInDirectory(@NotNull File directory, boolean ignoreUnderscoreFiles, boolean recursive) {
        return FileUtil.getFilesInDirectoryWithExtension(directory, null, ignoreUnderscoreFiles, recursive);
    }

    public static boolean doesDirectoryContainFile(@NotNull File directory, @NotNull String fileName, boolean recursive) {
        for (File file : FileUtil.getFilesInDirectory(directory, false, recursive)) {
            if (!file.getName().equals(fileName)) continue;
            return true;
        }
        return false;
    }

    public static Set<String> getAddonFilenames(@NotNull Class<?> clazz, String jarPath) throws IOException {
        URL jarLocation = clazz.getProtectionDomain().getCodeSource().getLocation();
        Optional<File> jarFile = FileUtil.fromURL(jarLocation);
        if (jarFile.isEmpty()) {
            EMFPlugin.getInstance().debug("Empty Jar file");
            return Collections.emptySet();
        }
        String normalizedJarPath = jarPath.replace('\\', '/').replaceAll("^/+|/+$", "");
        try (JarFile jar = new JarFile(jarFile.get());){
            Set<String> set = jar.stream().map(ZipEntry::getName).filter(name -> name.startsWith(normalizedJarPath + "/")).filter(name -> name.endsWith(".addon")).map(name -> name.substring(name.lastIndexOf(47) + 1)).peek(result -> EMFPlugin.getInstance().debug("Found matching addon: " + result)).collect(Collectors.toSet());
            return set;
        }
    }

    private static Optional<File> fromURL(@NotNull URL jarLocation) {
        try {
            return Optional.of(new File(jarLocation.toURI()));
        }
        catch (URISyntaxException e) {
            EMFPlugin.getInstance().getLogger().log(Level.WARNING, "Failed to convert JAR URL to file path", e);
            return Optional.empty();
        }
    }

    public static void loadFilesFromJarDirectory(@NotNull String jarDirectory, @NotNull File targetDirectory, @NotNull Predicate<String> filter, boolean overwrite) {
        List<String> fileList = FileUtil.getFilesFromJarDirectory(jarDirectory);
        if (fileList.isEmpty()) {
            EMFPlugin.getInstance().getLogger().warning(() -> "No files in jar directory, either this impl is wrong, or the path is.");
            return;
        }
        fileList.stream().filter(filter).forEach(file -> {
            String fileName = file.substring(file.lastIndexOf(47) + 1);
            FileUtil.loadFileOrResource(targetDirectory, fileName, file, (Plugin)EMFPlugin.getInstance(), overwrite);
        });
    }

    public static List<String> getFilesFromJarDirectory(@NotNull String jarDirectory) {
        try {
            String dirPath = jarDirectory.endsWith("/") ? jarDirectory.substring(0, jarDirectory.length() - 1) : jarDirectory;
            URL dirURL = ((Object)((Object)EMFPlugin.getInstance())).getClass().getClassLoader().getResource(dirPath);
            if (dirURL == null) {
                EMFPlugin.getInstance().getLogger().warning(() -> "Directory not found: " + dirPath);
                return Collections.emptyList();
            }
            if ("jar".equals(dirURL.getProtocol())) {
                return FileUtil.getFilesFromJar(dirURL, dirPath);
            }
            if ("file".equals(dirURL.getProtocol())) {
                return FileUtil.getFilesFromFileSystem(dirURL, dirPath);
            }
            return Collections.emptyList();
        }
        catch (Exception e) {
            EMFPlugin.getInstance().getLogger().warning("Error reading jar directory '" + jarDirectory + "': " + e.getMessage());
            return Collections.emptyList();
        }
    }

    private static List<String> getFilesFromJar(URL dirURL, String dirPath) throws IOException {
        ArrayList<String> files = new ArrayList<String>();
        String jarPath = dirURL.getPath().substring(5, dirURL.getPath().indexOf("!"));
        try (JarFile jar = new JarFile(URLDecoder.decode(jarPath, StandardCharsets.UTF_8));){
            Enumeration<JarEntry> entries = jar.entries();
            while (entries.hasMoreElements()) {
                String entry = entries.nextElement().getName();
                if (!entry.startsWith(dirPath) || entry.equals(dirPath + "/")) continue;
                files.add(entry);
            }
        }
        return files;
    }

    private static List<String> getFilesFromFileSystem(URL dirURL, String dirPath) {
        return Arrays.stream(new File(dirURL.getPath()).listFiles()).map(File::getName).map(name -> dirPath + "/" + name).toList();
    }

    public static void regenExampleFiles(String jarDirectory, File targetDirectory) {
        FileUtil.loadFilesFromJarDirectory(jarDirectory, targetDirectory, file -> file.startsWith("_example") && file.endsWith(".yml"), true);
    }

    public static void loadDefaultFiles(String jarDirectory, File targetDirectory) {
        EMFPlugin.getInstance().getLogger().info(() -> "Loading default %s configs from jar".formatted(jarDirectory));
        FileUtil.loadFilesFromJarDirectory(jarDirectory, targetDirectory, file -> !file.startsWith("_") && file.endsWith(".yml"), false);
    }
}

