package yesman.epicfight.world.capabilities.skill;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import com.google.common.collect.HashMultimap;
import com.mojang.datafixers.util.Pair;

import net.minecraft.nbt.CompoundTag;
import net.minecraft.resources.ResourceLocation;
import net.neoforged.bus.api.Event;
import yesman.epicfight.registry.EpicFightRegistries;
import yesman.epicfight.skill.Skill;
import yesman.epicfight.skill.Skill.SkillEventSubscriber;
import yesman.epicfight.skill.SkillCategory;
import yesman.epicfight.skill.SkillContainer;
import yesman.epicfight.skill.SkillEvent;
import yesman.epicfight.skill.SkillSlot;
import yesman.epicfight.world.capabilities.entitypatch.player.PlayerPatch;

public class PlayerSkills {
	public static final PlayerSkills EMPTY = new PlayerSkills(null);
	public final SkillContainer[] skillContainers;
	private final Map<Skill, SkillContainer> containersBySkill = new HashMap<> ();
	private final HashMultimap<SkillCategory, SkillContainer> containersByCategory = HashMultimap.create();
	private final HashMultimap<SkillCategory, Skill> learnedSkills = HashMultimap.create();
	
	public PlayerSkills(PlayerPatch<?> playerpatch) {
		Collection<SkillSlot> slots = SkillSlot.ENUM_MANAGER.universalValues();
		this.skillContainers = new SkillContainer[slots.size()];
		
		for (SkillSlot slot : slots) {
			SkillContainer skillContainer = new SkillContainer(playerpatch, slot);
			this.skillContainers[slot.universalOrdinal()] = skillContainer;
			this.containersByCategory.put(slot.category(), skillContainer);
		}
	}
	
	public void addLearnedSkill(Skill skill) {
		SkillCategory category = skill.getCategory();
		
		if (!this.learnedSkills.containsKey(category) || !this.learnedSkills.get(category).contains(skill)) {
			this.learnedSkills.put(category, skill);
		}
	}
	
	public boolean removeLearnedSkill(Skill skill) {
		SkillCategory category = skill.getCategory();
		
		if (this.learnedSkills.containsKey(category)) {
			if (this.learnedSkills.remove(category, skill)) {
				if (this.learnedSkills.get(category).isEmpty()) {
					this.learnedSkills.removeAll(category);
				}
				
				return true;
			}
		}
		
		return false;
	}
	
	public boolean hasCategory(SkillCategory skillCategory) {
		return this.learnedSkills.containsKey(skillCategory);
	}
	
	public boolean hasEmptyContainer(SkillCategory skillCategory) {
		for (SkillContainer container : this.containersByCategory.get(skillCategory)) {
			if (container.isEmpty()) return true; 
		}
		
		return false;
	}
	
	/**
	 * @return null if there is not empty container
	 */
	@Nullable
	public SkillContainer getFirstEmptyContainer(SkillCategory skillCategory) {
		for (SkillContainer container : this.containersByCategory.get(skillCategory)) {
			if (container.isEmpty()) return container; 
		}
		
		return null;
	}
	
	public boolean isEquipping(Skill skill) {
		return this.containersBySkill.containsKey(skill);
	}
	
	public boolean hasLearned(Skill skill) {
		return this.learnedSkills.get(skill.getCategory()).contains(skill);
	}
	
	public Set<SkillContainer> getSkillContainersFor(SkillCategory skillCategory) {
		return this.containersByCategory.get(skillCategory);
	}
	
	public SkillContainer getSkillContainerFor(SkillSlot skillSlot) {
		return this.getSkillContainerFor(skillSlot.universalOrdinal());
	}
	
	public SkillContainer getSkillContainerFor(int slotIndex) {
		return this.skillContainers[slotIndex];
	}
	
	@ApiStatus.Internal
	public void setSkillToContainer(Skill skill, SkillContainer container) {
		this.containersBySkill.put(skill, container);
	}
	
	@ApiStatus.Internal
	public void removeSkillFromContainer(Skill skill) {
		this.containersBySkill.remove(skill);
	}
	
	public SkillContainer getSkillContainer(Skill skill) {
		return this.containersBySkill.get(skill);
	}
	
	public Stream<SkillContainer> listSkillContainers() {
		return Stream.of(this.skillContainers);
	}
	
	public Stream<Skill> listAcquiredSkills() {
		return this.learnedSkills.values().stream();
	}
	
	public void clearContainersAndLearnedSkills(boolean isLocalOrServerPlayer) {
		for (SkillContainer container : this.skillContainers) {
			if (container.getSlot().category().learnable()) {
				if (isLocalOrServerPlayer) container.setSkill(null);
				else container.setSkillRemote(null);
			}
		}
		
		this.learnedSkills.clear();
	}
	
	public void copyFrom(PlayerSkills capabilitySkill) {
		int i = 0;
		
		for (SkillContainer container : this.skillContainers) {
			Skill oldone = capabilitySkill.skillContainers[i].getSkill();
			
			if (oldone != null && oldone.getCategory().shouldSynchronize()) {
				container.setSkill(capabilitySkill.skillContainers[i].getSkill());
			}
			
			i++;
		}
		
		this.learnedSkills.putAll(capabilitySkill.learnedSkills);
	}
	
	/**
	 * Find methods with {@link SkillEvent} annotation
	 * 
	 * @param <T>
	 * @param caller give "epicfight" for playerpatch events
	 *               for custom events called by add-on, give your modid
	 * @param event
	 */
	public <T extends Event> T fireSkillEvents(String caller, T event) {
		List<Pair<SkillContainer, SkillEventSubscriber>> subscribers = new ArrayList<> ();
		
		for (SkillContainer skillContainer : this.skillContainers) {
			Skill skill = skillContainer.getSkill();
			
			if (skill == null) {
				continue;
			}
			
			SkillEventSubscriber skillEventSubscriber = skill.getSkillEvent(caller, event.getClass(), skillContainer.getExecutor().isLogicalClient());
			
			if (skillEventSubscriber != null) {
				subscribers.add(Pair.of(skillContainer, skillEventSubscriber));
			}
		}
		
		subscribers.stream().sorted((p1, p2) -> p2.getSecond().compareTo(p1.getSecond())).forEach(subscriber -> {
			subscriber.getSecond().eventSubscriber().accept(event, subscriber.getFirst());
		});
		
		return event;
	}
	
	public void write(CompoundTag compound) {
		CompoundTag nbt = new CompoundTag();
		
		for (SkillContainer container : this.skillContainers) {
			if (container.getSkill() != null && container.getSkill().getCategory().shouldSave()) {
				nbt.putString(container.getSlot().toString().toLowerCase(Locale.ROOT), container.getSkill().toString());
			}
		}
		
		for (Map.Entry<SkillCategory, Collection<Skill>> entry : this.learnedSkills.asMap().entrySet()) {
			CompoundTag learnedNBT = new CompoundTag();
			int i = 0;
			
			for (Skill skill : entry.getValue()) {
				learnedNBT.putString(String.valueOf(i++), skill.toString());
			}
			
			nbt.put("learned:" + entry.getKey().toString().toLowerCase(Locale.ROOT), learnedNBT);
		}
		
		nbt.putString("playerMode", this.skillContainers[0].getExecutor().getPlayerMode().toString());
		
		compound.put("playerSkills", nbt);
	}
	
	public void read(CompoundTag compound) {
		CompoundTag skillCompound = compound.getCompound("playerSkills");
		
		for (SkillContainer container : this.skillContainers) {
			String key = container.getSlot().toString().toLowerCase(Locale.ROOT);
			
			if (skillCompound.contains(key)) {
				EpicFightRegistries.SKILL.getHolder(ResourceLocation.parse(skillCompound.getString(key))).ifPresent(skill -> {
					container.setSkill(skill.value());
					this.addLearnedSkill(skill.value());
				});
			}
		}
		
		for (SkillCategory category : SkillCategory.ENUM_MANAGER.universalValues()) {
			if (skillCompound.contains("learned:" + category.toString().toLowerCase(Locale.ROOT))) {
				CompoundTag learnedNBT = skillCompound.getCompound("learned:" + category.toString().toLowerCase(Locale.ROOT));
				
				for (String key : learnedNBT.getAllKeys()) {
					EpicFightRegistries.SKILL.getHolder(ResourceLocation.parse(learnedNBT.getString(key))).ifPresent(skill -> {
						this.addLearnedSkill(skill.value());
					});
				}
			}
		}
		
		if (skillCompound.contains("playerMode")) {
			this.skillContainers[0].getExecutor().toMode(PlayerPatch.PlayerMode.valueOf(skillCompound.getString("playerMode").toUpperCase(Locale.ROOT)), true);
		} else {
			this.skillContainers[0].getExecutor().toEpicFightMode(true);
		}
	}
}