package yesman.epicfight.api.animation;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.logging.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import net.minecraft.client.Minecraft;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.network.ServerGamePacketListenerImpl;
import net.minecraft.server.packs.resources.ResourceManager;
import net.minecraft.server.packs.resources.SimpleJsonResourceReloadListener;
import net.minecraft.util.profiling.ProfilerFiller;
import net.neoforged.api.distmarker.Dist;
import net.neoforged.api.distmarker.OnlyIn;
import net.neoforged.bus.api.Event;
import net.neoforged.fml.event.IModBusEvent;
import yesman.epicfight.api.animation.property.AnimationProperty;
import yesman.epicfight.api.animation.types.DynamicAnimation;
import yesman.epicfight.api.animation.types.StaticAnimation;
import yesman.epicfight.api.asset.AssetAccessor;
import yesman.epicfight.api.asset.JsonAssetLoader;
import yesman.epicfight.api.client.animation.AnimationSubFileReader;
import yesman.epicfight.api.data.reloader.SkillReloadListener;
import yesman.epicfight.api.exception.AssetLoadingException;
import yesman.epicfight.api.utils.InstantiateInvoker;
import yesman.epicfight.api.utils.MutableBoolean;
import yesman.epicfight.gameasset.Animations;
import yesman.epicfight.gameasset.Armatures;
import yesman.epicfight.main.EpicFightMod;
import yesman.epicfight.main.EpicFightSharedConstants;
import yesman.epicfight.network.EpicFightNetworkManager;
import yesman.epicfight.network.client.CPPairingAnimationRegistry;
import yesman.epicfight.network.server.SPDatapackSync;

@SuppressWarnings("unchecked")
public class AnimationManager extends SimpleJsonResourceReloadListener {
	private static final AnimationManager INSTANCE = new AnimationManager();
	private static ResourceManager serverResourceManager = null;
	
	public static AnimationManager getInstance() {
		return INSTANCE;
	}
	
	private final Map<Integer, AnimationAccessor<? extends StaticAnimation>> animationById = new HashMap<> ();
	private final Map<ResourceLocation, AnimationAccessor<? extends StaticAnimation>> animationByName = new HashMap<> ();
	private final Map<AnimationAccessor<? extends StaticAnimation>, StaticAnimation> animations = new HashMap<> ();
	private final Map<AnimationAccessor<? extends StaticAnimation>, String> resourcepackAnimationCommands = new HashMap<> ();
	
	public AnimationManager() {
		super(new GsonBuilder().create(), "animmodels/animations");
	}
	
	public static boolean checkNonNull(AssetAccessor<? extends StaticAnimation> animation) {
		if (animation == null || animation.isEmpty()) {
			if (animation != null) {
				EpicFightMod.stacktraceIfDevSide("Empty animation accessor: " + animation.registryName(), NoSuchElementException::new);
			} else {
				EpicFightMod.stacktraceIfDevSide("Null animation accessor", NoSuchElementException::new);
			}
			
			return false;
		}
		
		return true;
	}
	
	public static <T extends StaticAnimation> AnimationAccessor<T> byKey(String registryName) {
		return byKey(ResourceLocation.parse(registryName));
	}
	
	public static <T extends StaticAnimation> AnimationAccessor<T> byKey(ResourceLocation registryName) {
		return (AnimationAccessor<T>)getInstance().animationByName.get(registryName);
	}
	
	public static <T extends StaticAnimation> AnimationAccessor<T> byId(int animationId) {
		return (AnimationAccessor<T>)getInstance().animationById.get(animationId);
	}
	
	public Map<ResourceLocation, AnimationAccessor<? extends StaticAnimation>> getAnimations(Predicate<AssetAccessor<? extends StaticAnimation>> filter) {
		Map<ResourceLocation, AnimationAccessor<? extends StaticAnimation>> filteredItems =
			this.animationByName.entrySet().stream()
				.filter(entry -> {
					return filter.test(entry.getValue());
				})
				.collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
		
		return ImmutableMap.copyOf(filteredItems);
	}
	
	public AnimationClip loadAnimationClip(StaticAnimation animation, BiFunction<JsonAssetLoader, StaticAnimation, AnimationClip> clipLoader) {
		try
		{
			if (getAnimationResourceManager() == null) {
				return null;
			}

			JsonAssetLoader modelLoader = new JsonAssetLoader(getAnimationResourceManager(), animation.getLocation());
			AnimationClip loadedClip = clipLoader.apply(modelLoader, animation);

			return loadedClip;
		}
		catch (AssetLoadingException e)
		{
			throw new AssetLoadingException("Failed to load animation clip from: " + animation, e);
		}

	}
	
	public static void readAnimationProperties(StaticAnimation animation) {
		ResourceLocation dataLocation = getSubAnimationFileLocation(animation.getLocation(), AnimationSubFileReader.SUBFILE_CLIENT_PROPERTY);
		ResourceLocation povLocation = getSubAnimationFileLocation(animation.getLocation(), AnimationSubFileReader.SUBFILE_POV_ANIMATION);
		
		getAnimationResourceManager().getResource(dataLocation).ifPresent((rs) -> {
			AnimationSubFileReader.readAndApply(animation, rs, AnimationSubFileReader.SUBFILE_CLIENT_PROPERTY);
		});
		
		getAnimationResourceManager().getResource(povLocation).ifPresent((rs) -> {
			AnimationSubFileReader.readAndApply(animation, rs, AnimationSubFileReader.SUBFILE_POV_ANIMATION);
		});
	}
	
	@Override
	protected Map<ResourceLocation, JsonElement> prepare(ResourceManager resourceManager, ProfilerFiller profilerIn) {
		if (!EpicFightSharedConstants.isPhysicalClient() && serverResourceManager == null) {
			serverResourceManager = resourceManager;
		}
		
		this.animations.clear();
		this.animationById.entrySet().removeIf((entry) -> !entry.getValue().inRegistry());
		this.animationByName.entrySet().removeIf((entry) -> !entry.getValue().inRegistry());
		this.resourcepackAnimationCommands.clear();
		
		return super.prepare(resourceManager, profilerIn);
	}
	
	@Override
	protected void apply(Map<ResourceLocation, JsonElement> objectIn, ResourceManager resourceManager, ProfilerFiller profilerIn) {
		Armatures.reload(resourceManager);
		
		Set<ResourceLocation> registeredAnimation = this.animationById.values().stream().reduce(Sets.newHashSet(), (set, accessor) -> {
			set.add(accessor.registryName());
			
			for (AssetAccessor<? extends StaticAnimation> subAnimAccessor : accessor.get().getSubAnimations()) {
				set.add(subAnimAccessor.registryName());
			}
			
			return set;
		}, (set1, set2) -> {
			set1.addAll(set2);
			return set1;
		});
		
		/**
		 * Load animations that are not registered by {@link AnimationRegistryEvent}
		 * Reads from /assets folder in physical client, /datapack in physical server.
		 */
		objectIn.entrySet().stream().filter((entry) -> !registeredAnimation.contains(entry.getKey()) && !entry.getKey().getPath().contains("/data/") && !entry.getKey().getPath().contains("/pov/"))
									.sorted((e1, e2) -> e1.getKey().toString().compareTo(e2.getKey().toString()))
									.forEach((entry) -> {
										try {
											this.readResourcepackAnimation(entry.getKey(), entry.getValue().getAsJsonObject());
										} catch (Exception e) {
											EpicFightMod.LOGGER.error("Failed to load User animation " + entry.getKey() + " because of " + e + ". Skipped.");
											e.printStackTrace();
										}
									});
		
		SkillReloadListener.reloadAllSkillsAnimations();
		
		this.animations.entrySet().stream().reduce(Lists.<AssetAccessor<? extends StaticAnimation>>newArrayList(), (list, entry) -> {
			MutableBoolean init = new MutableBoolean(true);
			
			if (entry.getValue() == null || entry.getValue().getAccessor() == null) {
				EpicFightMod.logAndStacktraceIfDevSide(Logger::error, "Invalid animation implementation: " + entry.getKey(), AssetLoadingException::new);
				init.set(false);
			}
			
			entry.getValue().getSubAnimations().forEach((subAnimation) -> {
				if (subAnimation == null || subAnimation.get() == null) {
					EpicFightMod.logAndStacktraceIfDevSide(Logger::error, "Invalid sub animation implementation: " + entry.getKey(), AssetLoadingException::new);
					init.set(false);
				}
			});
			
			if (init.value()) {
				list.add(entry.getValue().getAccessor());
				list.addAll(entry.getValue().getSubAnimations());
			}
			
			return list;
		}, (list1, list2) -> {
			list1.addAll(list2);
			return list1;
		}).forEach((accessor) -> {
			accessor.doOrThrow(StaticAnimation::postInit);
			
			if (EpicFightSharedConstants.isPhysicalClient()) {
				AnimationManager.readAnimationProperties(accessor.get());
			}
		});
	}
	
	public static ResourceLocation getSubAnimationFileLocation(ResourceLocation location, AnimationSubFileReader.SubFileType<?> subFileType) {
		int splitIdx = location.getPath().lastIndexOf('/');
		
		if (splitIdx < 0) {
			splitIdx = 0;
		}
		
		return ResourceLocation.fromNamespaceAndPath(location.getNamespace(), String.format("%s/" + subFileType.getDirectory() + "%s", location.getPath().substring(0, splitIdx), location.getPath().substring(splitIdx)));
	}
	
	public static void setServerResourceManager(ResourceManager pResourceManager) {
		serverResourceManager = pResourceManager;
	}
	
	public static ResourceManager getAnimationResourceManager() {
		return EpicFightSharedConstants.isPhysicalClient() ? Minecraft.getInstance().getResourceManager() : serverResourceManager;
	}
	
	public Stream<CompoundTag> getResourcepackAnimationStream() {
		return this.resourcepackAnimationCommands.entrySet().stream().map((entry) -> {
			CompoundTag compTag = new CompoundTag();
			compTag.putString("registry_name", entry.getKey().registryName().toString());
			compTag.putInt("id", entry.getKey().id());
			compTag.putString("invoke_command", entry.getValue());
			
			return compTag;
		});
	}
	
	/**
	 * @param mandatoryPack : creates dummy animations for animations from the server without animation clips when the server has mandatory resource pack.
	 *                        custom weapon types & mob capabilities won't be created because they won't be able to find the animations from the server
	 *                        dummy animations will be automatically removed right after reloading resourced as the server forces using resource pack
	 */
	@OnlyIn(Dist.CLIENT)
	public void processServerPacket(SPDatapackSync packet, boolean mandatoryPack) {
		if (mandatoryPack) {
			for (CompoundTag tag : packet.tags()) {
				String invocationCommand = tag.getString("invoke_command");
				ResourceLocation registryName = ResourceLocation.parse(tag.getString("registry_name"));
				int id = tag.getInt("id");
						
				if (this.animationByName.containsKey(registryName)) {
					continue;
				}
				
				AnimationAccessor<? extends StaticAnimation> accessor = AnimationAccessorImpl.create(registryName, this.resourcepackAnimationCommands.size(), false, (accessor$2) -> {
					try {
						return InstantiateInvoker.invoke(invocationCommand, StaticAnimation.class).getResult();
					} catch (Exception e) {
						EpicFightMod.LOGGER.warn("Failed at creating animation from server resource pack");
						e.printStackTrace();
						return Animations.EMPTY_ANIMATION;
					}
				});
				
				this.animationById.put(id, accessor);
				this.animationByName.put(registryName, accessor);
			}
		}
		
		int animationCount = this.animations.size();
		List<String> registryNames = Lists.newArrayList();
		
		for (int i = 0; i < animationCount; i++) {
			registryNames.add(this.animationById.get(i + 1).registryName().toString());
		}
		
		CPPairingAnimationRegistry registrySyncPacket = new CPPairingAnimationRegistry(registryNames);
		EpicFightNetworkManager.sendToServer(registrySyncPacket);
	}
	
	public void validateClientAnimationRegistry(CPPairingAnimationRegistry msg, ServerGamePacketListenerImpl connection) {
		StringBuilder messageBuilder = new StringBuilder();
		int count = 0;
		
		Set<String> clientAnimationRegistry = new HashSet<> (Set.copyOf(msg.registryNames()));
		
		for (String registryName : this.animations.keySet().stream().map((rl) -> rl.toString()).toList()) {
			if (!clientAnimationRegistry.contains(registryName)) {
				// Animations that don't exist in client
				if (count < 10) {
					messageBuilder.append(registryName);
					messageBuilder.append("\n");
				}
				
				count++;
			} else {
				clientAnimationRegistry.remove(registryName);
			}
		}
		
		// Animations that don't exist in server
		for (String registryName : clientAnimationRegistry) {
			if (registryName.equals("empty")) {
				continue;
			}
			
			if (count < 10) {
				messageBuilder.append(registryName);
				messageBuilder.append("\n");
			}
			
			count++;
		}
		
		if (count >= 10) {
			messageBuilder.append(Component.translatable("gui.epicfight.warn.animation_unsync.etc", (count - 9)).getString());
			messageBuilder.append("\n");
		}
		
		if (!messageBuilder.isEmpty()) {
			connection.disconnect(Component.translatable("gui.epicfight.warn.animation_unsync", messageBuilder.toString()));
		}
	}
	
	private static final Set<String> NO_WARNING_MODID = Sets.newHashSet();
	
	public static void addNoWarningModId(String modid) {
		NO_WARNING_MODID.add(modid);
	}
	
	/**************************************************
	 * User-animation loader
	 **************************************************/
	@SuppressWarnings({ "deprecation" })
	private void readResourcepackAnimation(ResourceLocation rl, JsonObject json) throws Exception {
		JsonElement constructorElement = json.get("constructor");
		
		if (constructorElement == null) {
			if (NO_WARNING_MODID.contains(rl.getNamespace())) {
				return;
			} else {
				EpicFightMod.logAndStacktraceIfDevSide(
					  Logger::error
					, "Datapack animation reading failed: No constructor information has provided: " + rl
					, IllegalStateException::new
					, "No constructor information has provided in User animation, " + rl + "\nPlease remove this resource if it's unnecessary to optimize your project."
				);
				return;
			}
		}
		
		JsonObject constructorObject = constructorElement.getAsJsonObject();
		String invocationCommand = constructorObject.get("invocation_command").getAsString();
		StaticAnimation animation = InstantiateInvoker.invoke(invocationCommand, StaticAnimation.class).getResult();
		JsonElement propertiesElement = json.getAsJsonObject().get("properties");
		
		if (propertiesElement != null) {
			JsonObject propertiesObject = propertiesElement.getAsJsonObject();
			
			for (Map.Entry<String, JsonElement> entry : propertiesObject.entrySet()) {
				AnimationProperty<?> propertyKey = AnimationProperty.getSerializableProperty(entry.getKey());
				Object value = propertyKey.parseFrom(entry.getValue());
				animation.addPropertyUnsafe(propertyKey, value);
			}
		}
		
		AnimationAccessor<StaticAnimation> accessor = AnimationAccessorImpl.create(rl, this.animations.size() + 1, false, null);
		animation.setAccessor(accessor);
		
		this.resourcepackAnimationCommands.put(accessor, invocationCommand);
		this.animationById.put(accessor.id(), accessor);
		this.animationByName.put(accessor.registryName(), accessor);
		this.animations.put(accessor, animation);
	}



	public interface AnimationAccessor<A extends DynamicAnimation> extends AssetAccessor<A> {
		int id();
		
		default boolean idBetween(AnimationAccessor<? extends StaticAnimation> a1, AnimationAccessor<? extends StaticAnimation> a2) {
			return a1.id() <= this.id() && a2.id() >= this.id();
		}
	}
	
	public static record AnimationAccessorImpl<A extends StaticAnimation> (ResourceLocation registryName, int id, boolean inRegistry, Function<AnimationAccessor<A>, A> onLoad) implements AnimationAccessor<A> {
		private static <A extends StaticAnimation> AnimationAccessor<A> create(ResourceLocation registryName, int id, boolean inRegistry, Function<AnimationAccessor<A>, A> onLoad) {
			return new AnimationAccessorImpl<A> (registryName, id, inRegistry, onLoad);
		}
		
		@Override
		public A get() {
			if (!INSTANCE.animations.containsKey(this)) {
				INSTANCE.animations.put(this, this.onLoad.apply(this));
			}
			
			return (A)INSTANCE.animations.get(this);
		}
		
		public String toString() {
			return this.registryName.toString();
		}
		
		public int hashCode() {
			return this.registryName.hashCode();
		}
		
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			} else if (obj instanceof AnimationAccessor armatureAccessor) {
				return this.registryName.equals(armatureAccessor.registryName());
			} else if (obj instanceof ResourceLocation rl) {
				return this.registryName.equals(rl);
			} else if (obj instanceof String name) {
				return this.registryName.toString().equals(name);
			} else {
				return false;
			}
		}
	}
	
	public static class AnimationRegistryEvent extends Event implements IModBusEvent {
		private List<AnimationBuilder> builders = Lists.newArrayList();
		private Set<String> namespaces = Sets.newHashSet();
		
		public void newBuilder(String namespace, Consumer<AnimationBuilder> build) {
			if (this.namespaces.contains(namespace)) {
				throw new IllegalArgumentException("Animation builder namespace '" + namespace + "' already exists!");
			}
			
			this.namespaces.add(namespace);
			this.builders.add(new AnimationBuilder(namespace, build));
		}
		
		public List<AnimationBuilder> getBuilders() {
			return this.builders;
		}
	}
	
	public static record AnimationBuilder(String namespace, Consumer<AnimationBuilder> task) {
		public <T extends StaticAnimation> AnimationManager.AnimationAccessor<T> nextAccessor(String id, Function<AnimationManager.AnimationAccessor<T>, T> onLoad) {
			AnimationAccessor<T> accessor = AnimationAccessorImpl.create(ResourceLocation.fromNamespaceAndPath(this.namespace, id), INSTANCE.animations.size() + 1, true, onLoad);
			
			INSTANCE.animationById.put(accessor.id(), accessor);
			INSTANCE.animationByName.put(accessor.registryName(), accessor);
			INSTANCE.animations.put(accessor, null);
			
			return accessor; 
		}
	}
}
