package yesman.epicfight.data.loot;

import java.util.HashMap;
import java.util.Map;

import org.jetbrains.annotations.NotNull;

import com.google.common.collect.Maps;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;

import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.storage.loot.LootContext;
import net.minecraft.world.level.storage.loot.LootTable;
import net.minecraft.world.level.storage.loot.parameters.LootContextParams;
import net.minecraft.world.level.storage.loot.predicates.LootItemCondition;
import net.neoforged.neoforge.common.NeoForge;
import net.neoforged.neoforge.common.loot.IGlobalLootModifier;
import net.neoforged.neoforge.common.loot.LootModifier;
import yesman.epicfight.api.neoevent.SkillLootTableRegistryEvent;

public class OnSkillBookDroppedByEntity extends LootModifier {
	
	public static final MapCodec<OnSkillBookDroppedByEntity> SKILL_CODEC = RecordCodecBuilder.mapCodec(
		instance ->
			codecStart(instance)
			.apply(instance, OnSkillBookDroppedByEntity::new)
	);
	
	private static final Map<EntityType<?>, LootTable> SKILL_LOOT_TABLE = Maps.newHashMap();
	
	public static void registerEntitySkillLootTable() {
		Map<EntityType<?>, LootTable.Builder> builders = new HashMap<> ();
		SkillLootTableRegistryEvent entitySkillLootTable = new SkillLootTableRegistryEvent(builders);
		NeoForge.EVENT_BUS.post(entitySkillLootTable);
		
		SKILL_LOOT_TABLE.clear();
		
		builders.forEach((k, v) -> {
			SKILL_LOOT_TABLE.put(k, v.build());
		});
	}
	
	public OnSkillBookDroppedByEntity(LootItemCondition[] lootItemConditions) {
		super(lootItemConditions);
	}
	
	/**
	 * Applies the modifier to the generated loot (all loot conditions have already been checked
	 * and have returned true).
	 *
	 * @param generatedLoot the list of ItemStacks that will be dropped, generated by loot tables
	 * @param context       the LootContext, identical to what is passed to loot tables
	 * @return modified loot drops
	 */
	@SuppressWarnings("deprecation")
	@Override
	protected @NotNull ObjectArrayList<ItemStack> doApply(ObjectArrayList<ItemStack> generatedLoot, LootContext context) {
		Entity entity = context.getParamOrNull(LootContextParams.THIS_ENTITY);
		
		if (entity != null && SKILL_LOOT_TABLE.containsKey(entity.getType())) {
			
			// Call getRandomItemsRaw to avoid recursive call of global loot modifier
			SKILL_LOOT_TABLE.get(entity.getType()).getRandomItemsRaw(context, generatedLoot::add);
		}
		
		return generatedLoot;
	}

	@Override
	public MapCodec<? extends IGlobalLootModifier> codec() {
		return SKILL_CODEC;
	}
}