package yesman.epicfight.skill.passive;

import java.util.List;

import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.resources.sounds.SimpleSoundInstance;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.item.component.ItemAttributeModifiers;
import net.neoforged.api.distmarker.Dist;
import net.neoforged.api.distmarker.OnlyIn;
import yesman.epicfight.api.neoevent.playerpatch.DealDamageEvent;
import yesman.epicfight.api.neoevent.playerpatch.PlayerKilledEvent;
import yesman.epicfight.api.neoevent.playerpatch.TakeDamageEvent;
import yesman.epicfight.api.utils.math.ValueModifier;
import yesman.epicfight.client.gui.BattleModeGui;
import yesman.epicfight.main.EpicFightMod;
import yesman.epicfight.network.EntityPairingPacketTypes;
import yesman.epicfight.network.EpicFightNetworkManager;
import yesman.epicfight.network.server.SPEntityPairingPacket;
import yesman.epicfight.network.server.SPPlayUISound;
import yesman.epicfight.registry.entries.EpicFightSkillDataKeys;
import yesman.epicfight.registry.entries.EpicFightSounds;
import yesman.epicfight.skill.SkillBuilder;
import yesman.epicfight.skill.SkillContainer;
import yesman.epicfight.skill.SkillEvent;
import yesman.epicfight.skill.SkillEvent.Side;

public class VengeanceSkill extends PassiveSkill {
	private float damageBonus;
	
	public VengeanceSkill(SkillBuilder<?> builder) {
		super(builder);
	}
	
	@Override
	public void loadDatapackParameters(CompoundTag parameters) {
		super.loadDatapackParameters(parameters);
		this.damageBonus = parameters.getFloat("damage_bonus");
	}
	
	@SkillEvent(caller = EpicFightMod.MODID, side = Side.SERVER)
	public void takeDamagePre(TakeDamageEvent.Income event, SkillContainer container) {
		if (event.getDamageSource().getEntity() == null) {
			return;
		}
		
		int currentTargetId = container.getDataManager().getDataValue(EpicFightSkillDataKeys.ENTITY_ID);
		
		if (currentTargetId == -1 && event.getDamageSource().getEntity() instanceof LivingEntity livingentity) {
			setNewTarget(container, livingentity);
		} else if (currentTargetId == event.getDamageSource().getEntity().getId()) {
			container.getDataManager().setDataSync(EpicFightSkillDataKeys.TICK_RECORD, event.getPlayerPatch().getOriginal().tickCount);
		} else if (currentTargetId != event.getDamageSource().getEntity().getId() && event.getDamageSource().getEntity() instanceof LivingEntity livingentity) {
			if (canResetTarget(container)) {
				setNewTarget(container, livingentity);
			}
		}
	}
	
	@SkillEvent(caller = EpicFightMod.MODID, side = Side.SERVER)
	public void dealDamagePre(DealDamageEvent.Pre event, SkillContainer container) {
		int currentTargetId = container.getDataManager().getDataValue(EpicFightSkillDataKeys.ENTITY_ID);
		
		if (currentTargetId == event.getTarget().getId()) {
			event.getDamageSource().attachDamageModifier(ValueModifier.multiplier(1.0F + this.damageBonus));
			container.getDataManager().setDataSync(EpicFightSkillDataKeys.TICK_RECORD, event.getPlayerPatch().getOriginal().tickCount);
		} else if (container.isActivated()) {
			float f = this.damageBonus * container.getDurationRatio(1.0F);
			event.getDamageSource().attachDamageModifier(ValueModifier.multiplier(1.0F + f));
		}
	}
	
	@SkillEvent(caller = EpicFightMod.MODID, side = Side.SERVER)
	public void playerKilled(PlayerKilledEvent event, SkillContainer container) {
		if (container.isActivated()) {
			return;
		}
		
		int currentTargetId = container.getDataManager().getDataValue(EpicFightSkillDataKeys.ENTITY_ID);
		
		if (currentTargetId == event.getKilledEntity().getId()) {
			this.executeOnServer(container, null);
			container.getDataManager().setDataSync(EpicFightSkillDataKeys.ENTITY_ID, -1);
		}
	}
	
	@Override
	public void onRemoved(SkillContainer container) {
		container.runOnServer(serverplayerpatch -> {
			int currentTargetId = container.getDataManager().getDataValue(EpicFightSkillDataKeys.ENTITY_ID);
			Entity entity = container.getExecutor().getLevel().getEntity(currentTargetId);
			
			if (entity != null) {
				EpicFightNetworkManager.sendToPlayer(new SPEntityPairingPacket(currentTargetId, EntityPairingPacketTypes.VENGEANCE_TARGET_CANCEL), serverplayerpatch.getOriginal());
			}
		});
		
		super.onRemoved(container);
	}
	
	@Override
	public void updateContainer(SkillContainer container) {
		super.updateContainer(container);
		
		container.runOnServer(serverplayerpatch -> {
			int currentTargetId = container.getDataManager().getDataValue(EpicFightSkillDataKeys.ENTITY_ID);
			
			if (currentTargetId > -1) {
				Entity entity = container.getExecutor().getLevel().getEntity(currentTargetId);
				
				if (container.getExecutor().getOriginal().tickCount - container.getDataManager().getDataValue(EpicFightSkillDataKeys.TICK_RECORD) >= 160) {
					cancelTarget(container);
				} else if (entity == null || !entity.isAlive()) {
					cancelTarget(container);
				}
			}
		});
	}
	
	@Override
	public void executeOnServer(SkillContainer container, CompoundTag args) {
		super.executeOnServer(container, args);
		container.getExecutor().getOriginal().playSound(EpicFightSounds.VENGEANCE.get(), 1.0F, 1.0F);
		EpicFightNetworkManager.sendToAllPlayerTrackingThisEntityWithSelf(new SPEntityPairingPacket(container.getExecutor().getOriginal().getId(), EntityPairingPacketTypes.VENGEANCE_OVERLAY), container.getServerExecutor().getOriginal());
	}
	
	@Override
	public void cancelOnServer(SkillContainer container, CompoundTag args) {
		super.cancelOnServer(container, args);
		EpicFightNetworkManager.sendToAllPlayerTrackingThisEntityWithSelf(new SPEntityPairingPacket(container.getExecutor().getOriginal().getId(), EntityPairingPacketTypes.VENGEANCE_TARGET_CANCEL), container.getServerExecutor().getOriginal());
	}
	
	@Override
	public void onTracked(SkillContainer container, EpicFightNetworkManager.PayloadBundleBuilder payloadBuilder) {
		if (container.isActivated()) {
			payloadBuilder.and(new SPEntityPairingPacket(container.getExecutor().getOriginal().getId(), EntityPairingPacketTypes.VENGEANCE_OVERLAY));
		}
	}
	
	@OnlyIn(Dist.CLIENT)
	public void executeOnClient(SkillContainer container, FriendlyByteBuf args) {
		container.activate();
		// Playing sound twice fixes volume issue...
		Minecraft.getInstance().getSoundManager().play(SimpleSoundInstance.forUI(EpicFightSounds.VENGEANCE.get(), 1.0F, 1.0F));
		Minecraft.getInstance().getSoundManager().play(SimpleSoundInstance.forUI(EpicFightSounds.VENGEANCE.get(), 1.0F, 1.0F));
	}
	
	public static boolean tickExceeded(SkillContainer container) {
		return container.getExecutor().getOriginal().tickCount - container.getDataManager().getDataValue(EpicFightSkillDataKeys.TICK_RECORD) >= 160;
	}
	
	public static boolean canResetTarget(SkillContainer container) {
		return container.getExecutor().getOriginal().tickCount - container.getDataManager().getDataValue(EpicFightSkillDataKeys.TICK_RECORD) >= 80;
	}
	
	public static void setNewTarget(SkillContainer container, LivingEntity target) {
		cancelTarget(container);
		
		container.getDataManager().setDataSync(EpicFightSkillDataKeys.ENTITY_ID, target.getId());
		container.getDataManager().setDataSync(EpicFightSkillDataKeys.TICK_RECORD, container.getExecutor().getOriginal().tickCount);
		
		container.runOnServer(serverplayerpatch -> {
			EpicFightNetworkManager.sendToPlayer(new SPPlayUISound(EpicFightSounds.VENGEANCE), serverplayerpatch.getOriginal());
			EpicFightNetworkManager.sendToPlayer(new SPEntityPairingPacket(target.getId(), EntityPairingPacketTypes.VENGEANCE_OVERLAY), serverplayerpatch.getOriginal());
		});
	}
	
	public static void cancelTarget(SkillContainer container) {
		int currentTargetId = container.getDataManager().getDataValue(EpicFightSkillDataKeys.ENTITY_ID);
		Entity entity = container.getExecutor().getLevel().getEntity(currentTargetId);
		container.getDataManager().setDataSync(EpicFightSkillDataKeys.ENTITY_ID, -1);
		
		container.runOnServer(serverplayerpatch -> {
			if (entity != null) {
				EpicFightNetworkManager.sendToPlayer(new SPEntityPairingPacket(currentTargetId, EntityPairingPacketTypes.VENGEANCE_TARGET_CANCEL), serverplayerpatch.getOriginal());
			}
		});
	}
	
	@OnlyIn(Dist.CLIENT)
	@Override
	public boolean shouldDraw(SkillContainer container) {
		return container.isActivated() || (container.getDataManager().getDataValue(EpicFightSkillDataKeys.ENTITY_ID) > -1 && !tickExceeded(container));
	}
	
	@OnlyIn(Dist.CLIENT)
	@Override
	public void drawOnGui(BattleModeGui gui, SkillContainer container, GuiGraphics guiGraphics, float x, float y, float partialTick) {
		guiGraphics.blit(this.getSkillTexture(), (int)x, (int)y, 24, 24, 0, 0, 1, 1, 1, 1);
		
		if (container.isActivated()) {
			float f = Math.round(this.damageBonus * 100.0F * container.getDurationRatio(1.0F));
			guiGraphics.drawString(gui.getFont(), ItemAttributeModifiers.ATTRIBUTE_MODIFIER_FORMAT.format(f) + "%", x + 6, y + 8, 16777215, true);
		} else if (canResetTarget(container)) {
			int seconds = 4 - ((container.getExecutor().getOriginal().tickCount - container.getDataManager().getDataValue(EpicFightSkillDataKeys.TICK_RECORD)) - 80) / 20;
			guiGraphics.drawString(gui.getFont(), String.valueOf(seconds), x + 6, y + 8, 16777215, true);
		}
	}
	
	@OnlyIn(Dist.CLIENT)
	@Override
	public List<Object> getTooltipArgsOfScreen(List<Object> list) {
		list.add(ItemAttributeModifiers.ATTRIBUTE_MODIFIER_FORMAT.format(this.damageBonus * 100.0F));
		list.add(ItemAttributeModifiers.ATTRIBUTE_MODIFIER_FORMAT.format(this.damageBonus * 100.0F));
		list.add(String.valueOf(this.maxDuration / 20));
		
		return list;
	}
}
