package yesman.epicfight.client.events.engine;

import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang3.mutable.MutableBoolean;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.glfw.GLFW;

import com.mojang.blaze3d.platform.InputConstants;

import net.minecraft.ChatFormatting;
import net.minecraft.client.KeyMapping;
import net.minecraft.client.Minecraft;
import net.minecraft.client.Options;
import net.minecraft.client.multiplayer.ClientPacketListener;
import net.minecraft.client.player.Input;
import net.minecraft.client.player.LocalPlayer;
import net.minecraft.core.BlockPos;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.chat.Component;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.player.Inventory;
import net.minecraft.world.item.ShieldItem;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.EntityHitResult;
import net.minecraft.world.phys.HitResult;
import net.neoforged.bus.api.IEventBus;
import net.neoforged.neoforge.client.ClientHooks;
import net.neoforged.neoforge.client.event.ClientTickEvent;
import net.neoforged.neoforge.client.event.InputEvent;
import net.neoforged.neoforge.client.event.MovementInputUpdateEvent;
import net.neoforged.neoforge.event.entity.living.LivingEvent.LivingJumpEvent;
import yesman.epicfight.api.animation.types.EntityState;
import yesman.epicfight.api.client.input.PlayerInputState;
import yesman.epicfight.api.client.input.action.EpicFightInputActions;
import yesman.epicfight.api.client.input.handlers.InputManager;
import yesman.epicfight.api.client.neoevent.MappedMovementInputUpdateEvent;
import yesman.epicfight.api.neoevent.playerpatch.SkillCastEvent;
import yesman.epicfight.api.utils.FakeLevel;
import yesman.epicfight.client.ClientEngine;
import yesman.epicfight.client.gui.screen.SkillEditScreen;
import yesman.epicfight.client.gui.screen.config.IngameConfigurationScreen;
import yesman.epicfight.client.world.capabilites.entitypatch.player.LocalPlayerPatch;
import yesman.epicfight.config.ClientConfig;
import yesman.epicfight.main.EpicFightMod;
import yesman.epicfight.network.EpicFightNetworkManager;
import yesman.epicfight.skill.SkillContainer;
import yesman.epicfight.skill.SkillSlot;
import yesman.epicfight.skill.SkillSlots;
import yesman.epicfight.skill.modules.ChargeableSkill;
import yesman.epicfight.skill.modules.HoldableSkill;
import yesman.epicfight.world.capabilities.EpicFightCapabilities;
import yesman.epicfight.world.capabilities.skill.PlayerSkills;
import yesman.epicfight.world.gamerule.EpicFightGameRules;

public class ControlEngine implements IEventBasedEngine {
	private static final ControlEngine INSTANCE = new ControlEngine();
	
	public static ControlEngine getInstance() {
		return INSTANCE;
	}
	
	private final Set<CustomPacketPayload> packetsToSend = new HashSet<> ();
	private final Minecraft minecraft;
	private LocalPlayer player;
	private LocalPlayerPatch playerpatch;
	private int weaponInnatePressCounter = 0;
	private int sneakPressCounter = 0;
	private int moverPressCounter = 0;
	private int tickSinceLastJump = 0;
	private int lastHotbarLockedTime;
	private boolean weaponInnatePressToggle = false;
	private boolean sneakPressToggle = false;
	private boolean moverPressToggle = false;
	private boolean attackLightPressToggle = false;
	private boolean hotbarLocked;
	private boolean holdingFinished;
	private int reserveCounter;
    /**
     * <b>DEPRECATED:</b> This field is retained for backward compatibility and should not be used
     * for comparisons or method calls on this instance. In future updates, {@link EpicFightInputActions}
     * will be stored directly instead of a vanilla {@link KeyMapping}.
     *
     * <p>Do not rely on this field for new functionality. For mapping a {@link KeyMapping} to an
     * action, use {@link ControlEngine#mapKeyMappingToAction} instead (temporary solution).</p>
     *
     * @see ControlEngine#mapKeyMappingToAction
     */
    @SuppressWarnings("DeprecatedIsStillUsed")
    @Deprecated
	private KeyMapping reservedKey;
	private SkillSlot reservedOrHoldingSkillSlot;
    /**
     * <b>DEPRECATED:</b> Consider using {@link ControlEngine#isCurrentHoldingAction} or 
     * {@link ControlEngine#isCurrentHoldingActionActive} instead of directly 
     * accessing or comparing this field. This field is retained for backward 
     * compatibility; in future updates, {@link EpicFightInputActions} will be 
     * stored directly instead of a vanilla {@link KeyMapping}.
     *
     * @see ControlEngine#mapKeyMappingToAction
     */
    @SuppressWarnings("DeprecatedIsStillUsed")
    @Deprecated
    private KeyMapping currentHoldingKey;
	public Options options;
	
	private ControlEngine() {
		this.minecraft = Minecraft.getInstance();
		
		if (this.minecraft != null) {
			this.options = this.minecraft.options;
		}
	}
	
	public void reloadPlayerPatch(LocalPlayerPatch playerpatch) {
		this.weaponInnatePressCounter = 0;
		this.weaponInnatePressToggle = false;
		this.sneakPressCounter = 0;
		this.sneakPressToggle = false;
		this.attackLightPressToggle = false;
		this.player = playerpatch.getOriginal();
		this.playerpatch = playerpatch;
	}
	
	public LocalPlayerPatch getPlayerPatch() {
		return this.playerpatch;
	}
	
	public boolean canPlayerMove(EntityState playerState) {
		return !playerState.movementLocked() || this.player.jumpableVehicle() != null;
	}
	
	public boolean canPlayerRotate(EntityState playerState) {
		return !playerState.turningLocked() || this.player.jumpableVehicle() != null;
	}
	
	public void handleEpicFightKeyMappings() {
		// Pause here if playerpatch is null
		if (this.playerpatch == null) {
			return;
		}
		
        InputManager.triggerOnPress(EpicFightInputActions.OPEN_SKILL_SCREEN, false, this::openSkillEditor);
        
        InputManager.triggerOnPress(EpicFightInputActions.OPEN_CONFIG_SCREEN, false, this::openConfig);
        
        InputManager.triggerOnPress(EpicFightInputActions.SWITCH_VANILLA_MODEL_DEBUGGING, false, this::switchVanillaModelDebugging);
        
        InputManager.triggerOnPress(EpicFightInputActions.ATTACK, true, this::maybeAttack);
        
        InputManager.triggerOnPress(EpicFightInputActions.DODGE, true, this::maybeDodge);
        
        if (InputManager.isActionActive(EpicFightInputActions.GUARD)) this.maybeGuard();
        
        InputManager.triggerOnPress(EpicFightInputActions.WEAPON_INNATE_SKILL, true, this::handleSeparateWeaponInnateSkill);
        
        InputManager.triggerOnPress(EpicFightInputActions.MOBILITY, true, this::maybePerformMoverSkill);
        
        InputManager.triggerOnPress(EpicFightInputActions.SWITCH_MODE, false, this::switchMode);
		
        InputManager.triggerOnPress(EpicFightInputActions.LOCK_ON, false, this.playerpatch::toggleLockOn);
        
        if (shouldDisableSwapHandItems()) consumeSwapOffhandKeyClicks();
		
		// Pause here if player is not in battle mode
		if (!this.playerpatch.isEpicFightMode() || Minecraft.getInstance().isPaused()) {
			return;
		}
		
		if (this.player.tickCount - this.lastHotbarLockedTime > 20 && this.hotbarLocked) {
			this.unlockHotkeys();
		}
		
		if (this.weaponInnatePressToggle) {
			if (!InputManager.isActionActive(EpicFightInputActions.WEAPON_INNATE_SKILL)) {
				this.attackLightPressToggle = true;
				this.weaponInnatePressToggle = false;
				this.weaponInnatePressCounter = 0;
			} else {
				if (InputManager.isBoundToSamePhysicalInput(EpicFightInputActions.WEAPON_INNATE_SKILL, EpicFightInputActions.ATTACK)) {
					if (this.weaponInnatePressCounter > ClientConfig.longPressCounter) {
						if (this.playerpatch.getSkill(SkillSlots.WEAPON_INNATE).sendCastRequest(this.playerpatch, this).shouldReserveKey()) {
							if (!this.player.isSpectator()) {
								this.reserveKey(SkillSlots.WEAPON_INNATE, EpicFightInputActions.WEAPON_INNATE_SKILL);
							}
						} else {
							this.lockHotkeys();
						}
						
						this.weaponInnatePressToggle = false;
						this.weaponInnatePressCounter = 0;
					} else {
						this.weaponInnatePressCounter++;
					}
				}
			}
		}
		
		if (this.attackLightPressToggle) {
			SkillContainer airSlash = this.playerpatch.getSkill(SkillSlots.AIR_SLASH);
			SkillSlot slot = (this.tickSinceLastJump > 0 && airSlash.getSkill() != null && airSlash.getSkill().canExecute(airSlash)) ? SkillSlots.AIR_SLASH : SkillSlots.COMBO_ATTACKS;
			SkillCastEvent skillCastEvent = this.playerpatch.getSkill(slot).sendCastRequest(this.playerpatch, this);
			
			if (skillCastEvent.isExecutable()) {
				this.player.resetAttackStrengthTicker();
				this.releaseAllServedKeys();
			} else {
				if (!this.player.isSpectator() && slot == SkillSlots.COMBO_ATTACKS) {
					this.reserveKey(slot, EpicFightInputActions.ATTACK);
				}
			}
			
			this.lockHotkeys();
			
			this.attackLightPressToggle = false;
			this.weaponInnatePressToggle = false;
			this.weaponInnatePressCounter = 0;
		}
		
		if (this.sneakPressToggle) {
			if (!InputManager.isActionActive(EpicFightInputActions.SNEAK)) {
				SkillSlot skillSlot = (this.playerpatch.getEntityState().knockDown()) ? SkillSlots.KNOCKDOWN_WAKEUP : SkillSlots.DODGE;
				SkillContainer skill = this.playerpatch.getSkill(skillSlot);

				if (skill.sendCastRequest(this.playerpatch, this).shouldReserveKey()) {
					this.reserveKey(skillSlot, EpicFightInputActions.SNEAK);
				}
				
				this.sneakPressToggle = false;
				this.sneakPressCounter = 0;
			} else {
				if (this.sneakPressCounter > ClientConfig.longPressCounter) {
					this.sneakPressToggle = false;
					this.sneakPressCounter = 0;
				} else {
					this.sneakPressCounter++;
				}
			}
		}
		
		if (this.currentHoldingKey != null) {
			SkillContainer container = this.playerpatch.getSkill(this.reservedOrHoldingSkillSlot);
			
			if (!container.isEmpty()) {
				if (container.getSkill() instanceof HoldableSkill) {
					if (!this.isCurrentHoldingActionActive()) {
						this.holdingFinished = true;
					}
					
					if (container.getSkill() instanceof ChargeableSkill chargingSkill) {
						if (this.holdingFinished) {
							if (this.playerpatch.getSkillChargingTicks() > chargingSkill.getMinChargingTicks()) {
								container.sendCastRequest(this.playerpatch, this);
								this.releaseAllServedKeys();
							}
						} else if (this.playerpatch.getSkillChargingTicks() >= chargingSkill.getAllowedMaxChargingTicks()) {
							this.releaseAllServedKeys();
						}
					} else {
						if (this.holdingFinished) {
							// Note: Holdable skills are canceled in client first
							this.playerpatch.resetHolding();
							CompoundTag arguments = new CompoundTag();
							container.getSkill().gatherArguments(container, this, arguments);
							container.getSkill().cancelOnClient(container, arguments);
							container.sendCancelRequest(this.playerpatch, this);
							this.releaseAllServedKeys();
						}
					}
				} else {
					this.releaseAllServedKeys();
				}
			}
		}
		
		if (this.reservedKey != null) {
			if (this.reserveCounter > 0) {
				SkillContainer skill = this.playerpatch.getSkill(this.reservedOrHoldingSkillSlot);
				this.reserveCounter--;
				
				if (skill.getSkill() != null) {
					if (skill.sendCastRequest(this.playerpatch, this).isExecutable()) {
						this.releaseAllServedKeys();
						this.lockHotkeys();
					}
				}
			} else {
				this.releaseAllServedKeys();
			}
		}

		if (isSwitchOrDropBlocked()) {
			disableHotbarSlotPresses();
			consumeDropKeyClicks();
		}
	}

    private void openSkillEditor() {
        final PlayerSkills playerSkills = this.playerpatch.getPlayerSkills();
        if (playerSkills == null) {
            return;
        }
        
        this.minecraft.setScreen(new SkillEditScreen(this.player, playerSkills));
    }

    private void openConfig() {
    	this.minecraft.setScreen(new IngameConfigurationScreen(null, null));
    }

    private void switchVanillaModelDebugging() {
        boolean flag = ClientEngine.getInstance().switchVanillaModelDebuggingMode();
        this.minecraft.keyboardHandler.debugFeedbackTranslated(flag ? "debug.vanilla_model_debugging.on" : "debug.vanilla_model_debugging.off");
    }

    private void maybeAttack() {
        if (!this.playerpatch.isEpicFightMode() || isCurrentHoldingAction(EpicFightInputActions.ATTACK)) {
            return;
        }
        final EpicFightInputActions vanillaAttack = EpicFightInputActions.VANILLA_ATTACK_DESTROY;
        final EpicFightInputActions epicFightAttack = EpicFightInputActions.ATTACK;

        boolean shouldPlayAttackAnimation = this.playerpatch.canPlayAttackAnimation();
        if (vanillaAttack.keyMapping().getKey() == epicFightAttack.keyMapping().getKey() &&
                Minecraft.getInstance().hitResult != null && shouldPlayAttackAnimation) {
            consumeVanillaAttackKeyClicks();
        }

        if (shouldPlayAttackAnimation) {
            if (!InputManager.isBoundToSamePhysicalInput(epicFightAttack, EpicFightInputActions.WEAPON_INNATE_SKILL)) {
                SkillContainer airSlash = this.playerpatch.getSkill(SkillSlots.AIR_SLASH);
                SkillSlot slot = (this.tickSinceLastJump > 0 && airSlash.getSkill() != null && airSlash.getSkill().canExecute(airSlash)) ? SkillSlots.AIR_SLASH : SkillSlots.COMBO_ATTACKS;
                SkillCastEvent skillCastEvent = this.playerpatch.getSkill(slot).sendCastRequest(this.playerpatch, this);

                if (skillCastEvent.isExecutable()) {
                    this.player.resetAttackStrengthTicker();
                    this.attackLightPressToggle = false;
                    this.releaseAllServedKeys();
                } else {
                    if (!this.player.isSpectator() && slot == SkillSlots.COMBO_ATTACKS) {
                        this.reserveKey(slot, epicFightAttack);
                    }
                }

                this.lockHotkeys();
                this.attackLightPressToggle = false;
                this.weaponInnatePressToggle = false;
                this.weaponInnatePressCounter = 0;
            } else {
                if (!this.weaponInnatePressToggle) {
                    this.weaponInnatePressToggle = true;
                }
            }
        }
    }

    private void maybeDodge() {
        if (!this.playerpatch.isEpicFightMode() || isCurrentHoldingAction(EpicFightInputActions.DODGE)) {
            return;
        }
        if (InputManager.isBoundToSamePhysicalInput(EpicFightInputActions.DODGE, EpicFightInputActions.SNEAK)) {
            if (this.player.getVehicle() == null) {
                if (!this.sneakPressToggle) {
                    this.sneakPressToggle = true;
                }
            }
        } else {
            SkillSlot skillCategory = (this.playerpatch.getEntityState().knockDown()) ? SkillSlots.KNOCKDOWN_WAKEUP : SkillSlots.DODGE;
            SkillContainer skill = this.playerpatch.getSkill(skillCategory);

            if (!skill.isEmpty() && skill.sendCastRequest(this.playerpatch, this).shouldReserveKey()) {
                this.reserveKey(SkillSlots.DODGE, EpicFightInputActions.DODGE);
            }
        }
    }

    private void maybeGuard() {
        if (!this.playerpatch.isEpicFightMode() || isCurrentHoldingAction(EpicFightInputActions.GUARD)) {
            return;
        }
        boolean shouldCancelGuard = false;
        
        if (this.playerpatch.isHoldingAny()) {
            shouldCancelGuard = true;
        } else if (ShieldItem.class.isAssignableFrom(this.player.getMainHandItem().getItem().getClass()) || ShieldItem.class.isAssignableFrom(this.player.getOffhandItem().getItem().getClass())) {
            shouldCancelGuard = true;
        }
        
        if (!shouldCancelGuard) {
            SkillCastEvent skillCastEvent = this.playerpatch.getSkill(SkillSlots.GUARD).sendCastRequest(this.playerpatch, this);

            if (skillCastEvent.shouldReserveKey()) {
                if (!this.player.isSpectator()) {
                    this.reserveKey(SkillSlots.GUARD, EpicFightInputActions.GUARD);
                }
            } else {
                this.lockHotkeys();
            }
        }
    }

    private void handleSeparateWeaponInnateSkill() {
        if (!this.playerpatch.isEpicFightMode() || isCurrentHoldingAction(EpicFightInputActions.WEAPON_INNATE_SKILL)) {
            return;
        }
        if (!InputManager.isBoundToSamePhysicalInput(EpicFightInputActions.ATTACK, EpicFightInputActions.WEAPON_INNATE_SKILL)) {
            if (this.playerpatch.getSkill(SkillSlots.WEAPON_INNATE).sendCastRequest(this.playerpatch, this).shouldReserveKey()) {
                if (!this.player.isSpectator()) {
                    this.reserveKey(SkillSlots.WEAPON_INNATE, EpicFightInputActions.WEAPON_INNATE_SKILL);
                }
            } else {
                this.lockHotkeys();
            }
        }
    }

    private void maybePerformMoverSkill() {
        if (!this.playerpatch.isEpicFightMode() || this.playerpatch.isHoldingAny()) {
            return;
        }
        
        if (InputManager.isBoundToSamePhysicalInput(EpicFightInputActions.MOBILITY, EpicFightInputActions.JUMP)) {
            SkillContainer skillContainer = this.playerpatch.getSkill(SkillSlots.MOVER);

            if (!skillContainer.isEmpty()) {
                SkillCastEvent event = new SkillCastEvent(this.playerpatch, skillContainer, null);

                if (skillContainer.canUse(this.playerpatch, event) && this.player.getVehicle() == null) {
                    if (!this.moverPressToggle) {
                        this.moverPressToggle = true;
                    }
                }
            }
        } else {
            // Immediately trigger the skill cast if Mover and Jump are bound to different keys/buttons.
            SkillContainer skill = this.playerpatch.getSkill(SkillSlots.MOVER);
            skill.sendCastRequest(this.playerpatch, this);
        }
    }

    private void switchMode() {
        final boolean canSwitch = EpicFightGameRules.CAN_SWITCH_PLAYER_MODE.getRuleValue(this.playerpatch.getOriginal().level());
        if (!canSwitch) {
            this.minecraft.gui.getChat().addMessage(Component.translatable("epicfight.messages.mode_switching_disabled").withStyle(ChatFormatting.RED));
            return;
        }
        this.playerpatch.toggleMode();
    }
	
	private PlayerInputState inputTick(Input input) {
		if (this.tickSinceLastJump > 0) this.tickSinceLastJump--;
		PlayerInputState inputState = InputManager.getInputState(input);
		
		if (this.moverPressToggle) {
			if (!InputManager.isActionActive(EpicFightInputActions.JUMP)) {
				this.moverPressToggle = false;
				this.moverPressCounter = 0;
				
				if (this.player.onGround()) {
					this.player.noJumpDelay = 0;
                    inputState = inputState.withJumping(true);
                    InputManager.setInputState(inputState);
				}
			} else {
				if (this.moverPressCounter > ClientConfig.longPressCounter) {
					SkillContainer skill = this.playerpatch.getSkill(SkillSlots.MOVER);
					skill.sendCastRequest(this.playerpatch, this);
					
					this.moverPressToggle = false;
					this.moverPressCounter = 0;
				} else {
					this.player.noJumpDelay = 2;
					this.moverPressCounter++;
				}
			}
		}
		
		if (!this.canPlayerMove(this.playerpatch.getEntityState())) {
            inputState = inputState.copyWith(0F, 0F, false, false, false, false, false, false);
            InputManager.setInputState(inputState);
			this.player.sprintTriggerTime = -1;
			this.player.setSprinting(false);
		}
		
		return inputState;
	}

    /**
     * <b>DEPRECATED:</b> This method is retained for backward compatibility and will 
     * be removed in a future release. Do not use it for new code.
     * <p>Instead of using this method, use
     * {@link #reserveKey(SkillSlot, EpicFightInputActions)}, which works directly
     * with {@link EpicFightInputActions}.</p>
     */
    @SuppressWarnings("DeprecatedIsStillUsed")
    @Deprecated(forRemoval = true)
	private void reserveKey(SkillSlot slot, KeyMapping keyMapping) {
		this.reservedKey = keyMapping;
		this.reservedOrHoldingSkillSlot = slot;
		this.reserveCounter = 8;
	}

    private void reserveKey(SkillSlot slot, EpicFightInputActions action) {
        reserveKey(slot, action.keyMapping());
    }
	
	public void releaseAllServedKeys() {
		this.holdingFinished = true;
		this.currentHoldingKey = null;
		this.reservedOrHoldingSkillSlot = null;
		this.reserveCounter = -1;
		this.reservedKey = null;
	}
	
	public void setHoldingKey(SkillSlot chargingSkillSlot, KeyMapping keyMapping) {
		this.holdingFinished = false;
		this.currentHoldingKey = keyMapping;
		this.reservedOrHoldingSkillSlot = chargingSkillSlot;
		this.reserveCounter = -1;
		this.reservedKey = null;
	}
	
	public void lockHotkeys() {
		this.hotbarLocked = true;
		this.lastHotbarLockedTime = this.player.tickCount;
        disableHotbarSlotPresses();
	}
	
	public void unlockHotkeys() {
		this.hotbarLocked = false;
	}
	
	public void addPacketToSend(CustomPacketPayload packet) {
		this.packetsToSend.add(packet);
	}

    /**
     * <b>DEPRECATED:</b> Use {@link InputManager#isActionActive} instead for controller support,
     * though it only handles Epic Fight supported actions. For checking a custom mod keybind,
     * use the vanilla {@link KeyMapping#isDown}.
     * <p>
     * Please note that there is a difference between vanilla {@link KeyMapping#isDown} and this method,
     * {@link KeyMapping#isDown} may be <code>false</code> in some cases even if the physical key
     * is actually down, for example, if a screen is open.
     * <p>
     * Even though this is a private method, it is retained in case an Epic Fight addon
     * accesses it by bypassing Java private access modifier restriction.
     */
	@SuppressWarnings({"JavadocReference", "DeprecatedIsStillUsed"})
    @Deprecated(forRemoval = true)
    public static boolean isKeyDown(KeyMapping key) {
		if (key.getKey().getType() == InputConstants.Type.KEYSYM) {
			return key.isDown() || GLFW.glfwGetKey(Minecraft.getInstance().getWindow().getWindow(), key.getKey().getValue()) > 0;
		} else if(key.getKey().getType() == InputConstants.Type.MOUSE) {
			return key.isDown() || GLFW.glfwGetMouseButton(Minecraft.getInstance().getWindow().getWindow(), key.getKey().getValue()) > 0;
		} else {
			return false;
		}
	}

    /**
     * <b>DEPRECATED:</b> Use {@link InputManager#triggerOnPress} instead for controller support,
     * though it only handles Epic Fight supported actions. For checking a custom mod keybind,
     * use the vanilla {@link KeyMapping#consumeClick} with a {@code while} statement.
     * <p>
     * Even though this is a private method, it is retained in case an Epic Fight addon
     * accesses it by bypassing Java private access modifier restriction.
     * @see InputManager#isPhysicalKeyDownInternalWorkaround(KeyMapping)
     * @see InputManager#isKeyDown(KeyMapping)
     * @deprecated Consider adapting the Minecraft's {@link KeyMapping#isDown()}, keep in mind that
     *  this may return false in some cases, such as when a screen or chat is open.
     */
    @SuppressWarnings({"JavadocReference", "removal"})
    @Deprecated(forRemoval = true)
	private static boolean isKeyPressed(KeyMapping key, boolean eventCheck) {
		boolean consumes = key.consumeClick();
		
		if (consumes && eventCheck) {
			int mouseButton = InputConstants.Type.MOUSE == key.getKey().getType() ? key.getKey().getValue() : -1;
			InputEvent.InteractionKeyMappingTriggered inputEvent = ClientHooks.onClickInput(mouseButton, key, InteractionHand.MAIN_HAND);
			
	        if (inputEvent.isCanceled()) {
	        	return false;
	        }
		}
        
    	return consumes;
	}

    /**
     * <b>DISCOURAGED:</b> Does not support controller mods or other input systems.
     * <p>
     * This method was previously called before {@link Minecraft#handleKeybinds} to disable some vanilla
     * input actions.
     * <p>
     * Rather than relying on this method, consider fixing the problem at the core.
     * For example, to disable vanilla attack:
     * <pre>
     * {@code
     * @Mixin(value = Minecraft.class)
     * public class MixinMinecraft {
     *     @Inject(method = "startAttack", at = @At("HEAD"), cancellable = true)
     *     private void onVanillaAttack(CallbackInfoReturnable<Boolean> cir) {
     *         cir.cancel();
     *     }
     * }
     * }
     * </pre>
     * <b>NOTE:</b> You may still want to call this method to reset the internal {@link KeyMapping#clickCount} state,
     * as a safety guard to prevent potential issues with other mods.
     * Refer to {@link ControlEngine#consumeVanillaAttackKeyClicks} and {@link ControlEngine#shouldDisableVanillaAttack} as an example.
     *
     * @see ControlEngine#shouldDisableVanillaAttack
     * @see ControlEngine#shouldDisableSwapHandItems
     */
    @SuppressWarnings({"JavadocReference", "DeprecatedIsStillUsed", "StatementWithEmptyBody", "DefaultAnnotationParam"})
    @Deprecated(forRemoval = false)
    public static void makeUnpressed(KeyMapping keyMapping) {
        while (keyMapping.consumeClick()) {
        }
        KeyMapping.set(keyMapping.getKey(), false);
    }

    /**
     * <b>DISCOURAGED:</b> Does not allow Epic Fight to perform additional
     * logic that is independent of the Minecraft vanilla {@link KeyMapping}.
     * Previously used to set the {@link KeyMapping#isDown} state for sprint keybind,
     * but that is now handled in {@link ControlEngine#setSprintingKeyStateNotDown}.
     * Alternatively, you could use the vanilla @{@link KeyMapping#set} directly.
     *
     * @see ControlEngine#makeUnpressed
     * @see ControlEngine#setSprintingKeyStateNotDown
     */
    @SuppressWarnings("JavadocReference")
    @Deprecated(forRemoval = true)
    public static void setKeyBind(KeyMapping key, boolean setter) {
        KeyMapping.set(key.getKey(), setter);
    }

    /**
     * Sets the state of the sprint keybind ({@link KeyMapping#isDown}) to `false`.
     * <p>
     * This only changes the internal {@link KeyMapping} state and does not actually disable sprinting.
     * To actually stop sprinting, this should usually be called alongside
     * {@link LocalPlayer#setSprinting}.
     * <p>
     * This is intended to support mods and other systems that rely on {@link KeyMapping#isDown()}
     * to check whether the player is sprinting, rather than {@link LocalPlayer#isSprinting()}.
     * This is not needed for controller mods.
     */
    @SuppressWarnings("JavadocReference")
    public static void setSprintingKeyStateNotDown() {
        KeyMapping.set(EpicFightInputActions.SPRINT.keyMapping().getKey(), false);
    }

    /**
     * Disables the player's vanilla attacks while in Epic Fight mode.
     * <p>
     * Previously, we injected into the vanilla {@link Minecraft#handleKeybinds} method
     * and used this workaround to reset the internal counter for the vanilla attack keybind:
     * <pre>
     * {@code
     * // Called before Minecraft#handleKeybinds is called.
     * KeyMapping attack = Minecraft.getInstance().options.keyAttack;
     * while (attack.consumeClick()) {}
     * KeyMapping.set(attack.getKey(), false);
     * }
     * </pre>
     * <p>
     * However, that approach relied on assumptions and did not support other mods, inputs, or systems.
     * The problem is now solved by injecting into {@link Minecraft#startAttack} and
     * canceling the call when the player is in Epic Fight mode.
     * This also means, the player must be always in vanilla mode to perform vanilla attacks, which is as intended.
     *
     * @see ControlEngine#consumeVanillaAttackKeyClicks
     */
    @SuppressWarnings("JavadocReference")
    @ApiStatus.Internal
    public static boolean shouldDisableVanillaAttack() {
        final LocalPlayerPatch playerpatch = ClientEngine.getInstance().getPlayerPatch();
        if (playerpatch == null) {
            return false;
        }
        return playerpatch.isEpicFightMode() && playerpatch.canPlayAttackAnimation();
    }

    /**
     * Disables the swap offhand items while the player is in action.
     * <p>
     * Previously, we injected into the vanilla {@link Minecraft#handleKeybinds} method
     * and used this workaround to reset the internal counter for the vanilla swap offhand keybind:
     * <pre>
     * {@code
     * // Called before Minecraft#handleKeybinds is called.
     * KeyMapping swapOffhand = Minecraft.getInstance().options.keySwapOffhand;
     * while (swapOffhand.consumeClick()) {}
     * KeyMapping.set(swapOffhand.getKey(), false);
     * }
     * </pre>
     * <p>
     * However, that approach relied on assumptions and did not support other mods, inputs, or systems.
     * The problem is now solved by injecting into {@link ClientPacketListener#send} and
     * canceling the call when the player is in action and trying to swap offhand items.
     *
     * @see ControlEngine#consumeSwapOffhandKeyClicks
     */
    @SuppressWarnings("JavadocReference")
    @ApiStatus.Internal
    public static boolean shouldDisableSwapHandItems() {
        final LocalPlayerPatch playerpatch = ClientEngine.getInstance().getPlayerPatch();
        if (playerpatch == null) {
            return false;
        }
        return playerpatch.getEntityState().inaction() || (!playerpatch.getHoldingItemCapability(InteractionHand.MAIN_HAND).canBePlacedOffhand());
    }


    /**
     * Previously used to disable the vanilla attack key so the player
     * can't attack entities or break grass when in Epic Fight mode, but that is now handled by
     * {@link yesman.epicfight.mixin.client.MixinMinecraft#onStartVanillaAttack} and
     * {@link yesman.epicfight.mixin.client.MixinMinecraft#onContinueVanillaAttack}.
     * <p>
     * This method now only decrements the internal counter of the vanilla {@link KeyMapping#clickCount}
     * to prevent potential conflicts with other mods. It acts as a safety measure;
     * removing it should no longer cause issues.
     * <p>
     * This method does not rely on {@link InputManager} because it operates solely on
     * the vanilla {@link KeyMapping} behavior.
     * @see ControlEngine#shouldDisableVanillaAttack
     */
    @SuppressWarnings("JavadocReference")
    private static void consumeVanillaAttackKeyClicks() {
        makeUnpressed(EpicFightInputActions.VANILLA_ATTACK_DESTROY.keyMapping());
    }

    /**
     * Previously used to temporarily disable the vanilla swap-offhand key while the player
     * was performing an action or attacking, but this is now handled by
     * {@link yesman.epicfight.mixin.client.MixinClientCommonPacketListenerImpl#onBeforeSendPacket}.
     * <p>
     * This method now only decrements the internal counter of the vanilla {@link KeyMapping#clickCount}
     * to prevent potential conflicts with other mods. It serves as a safety measure;
     * removing it should no longer cause any issues.
     * <p>
     * This method does not rely on {@link InputManager} because it operates solely on
     * the vanilla {@link KeyMapping} behavior.
     * @see ControlEngine#shouldDisableSwapHandItems
     */
    @SuppressWarnings("JavadocReference")
    private static void consumeSwapOffhandKeyClicks() {
        makeUnpressed(EpicFightInputActions.SWAP_OFF_HAND.keyMapping());
    }

    /**
     * Disables hotbar slot key presses (keyboard only).
     * <p>
     * This feature is strictly for keyboards and will not support controllers,
     * as controllers have limited buttons. Keyboard users can switch slots via
     * number keys or the mouse wheel. Controller users can only switch using
     * forward/backward buttons.
     *
     * @see ControlEngine#isHotbarCyclingDisabled
     */
    private static void disableHotbarSlotPresses() {
        final KeyMapping[] hotbarSlots = Minecraft.getInstance().options.keyHotbarSlots;
        for (int i = 0; i < 9; ++i) {
            final KeyMapping hotbarSlot = hotbarSlots[i];
            makeUnpressed(hotbarSlot);
        }
    }

    /**
     * Previously used to temporarily disable the vanilla item drop key while the player
     * was performing an action or attacking, but this is now handled by
     * {@link yesman.epicfight.mixin.client.MixinLocalPlayer#onDrop}.
     * <p>
     * This method now only decrements the internal counter of the vanilla {@link KeyMapping#clickCount}
     * to prevent potential conflicts with other mods. It serves as a safety measure;
     * removing it should no longer cause any issues.
     * <p>
     */
    @SuppressWarnings("JavadocReference")
    private static void consumeDropKeyClicks() {
        makeUnpressed(EpicFightInputActions.DROP.keyMapping());
    }

    /**
     * Maps a {@link KeyMapping} to its corresponding input action, if defined.
     * <p>
     * Each {@link EpicFightInputActions} enum constant has an associated {@link KeyMapping},
     * but not every {@link KeyMapping} corresponds to an {@link EpicFightInputActions}, so this may return {@code null}.
     * Using {@link KeyMapping} directly does not support controllers.
     * <p>
     * Ideally, this workaround should not exist. The code should depend on {@link EpicFightInputActions} directly
     * instead of storing {@link KeyMapping} instances. However, since some classes and Epic Fight addons still rely on:
     * <ul>
     *   <li>{@link HoldableSkill#getKeyMapping}</li>
     *   <li>{@link ControlEngine#currentHoldingKey}</li>
     *   <li>{@link ControlEngine#reservedKey}</li>
     * </ul>
     * this method remains temporarily for backward compatibility. Future updates should refactor these dependencies
     * to remove reliance on {@link KeyMapping}, allowing this method to be fully removed.
     * <p>
     * Sometimes, it makes sense to use this method, for example, if you're using an event such as {@link InputEvent.InteractionKeyMappingTriggered},
     * which provides only a {@link KeyMapping}.
     */
    private static @Nullable EpicFightInputActions mapKeyMappingToAction(@NotNull KeyMapping keyMapping) {
        return EpicFightInputActions.fromKeyMapping(keyMapping);
    }

    /**
     * Checks if the specified input action is currently being held.
     *
     * @param other the input action to check
     * @return {@code true} if the given action is currently held; otherwise {@code false}
     * @see ControlEngine#mapKeyMappingToAction
     */
    private boolean isCurrentHoldingAction(@NotNull EpicFightInputActions other) {
        if (currentHoldingKey == null) {
            return false;
        }
        final EpicFightInputActions currentHoldingAction = mapKeyMappingToAction(currentHoldingKey);
        if (currentHoldingAction == null) {
            // Fallback for legacy or custom key mappings.
            // This is IMPORTANT to prevent addon breakage; this allows custom keybinds from other mods,
            // but controller inputs will not support those custom keybinds.
            return currentHoldingKey == other.keyMapping();
        }
        return other == currentHoldingAction;
    }
    
    private boolean isCurrentHoldingActionActive() {
        if (currentHoldingKey == null) {
            return false;
        }
        final EpicFightInputActions currentHoldingAction = mapKeyMappingToAction(currentHoldingKey);
        if (currentHoldingAction == null) {
            // Fallback for legacy or custom key mappings.
            // This is IMPORTANT to prevent addon breakage; this allows custom keybinds from other mods,
            // but controller inputs will not support those custom keybinds.
            return isKeyDown(currentHoldingKey);
        }
        return InputManager.isActionActive(currentHoldingAction);
    }

    /**
     * Determines whether hotbar cycling should be disabled.
     * <p>
     * Used internally in {@link InputEvent.MouseScrollingEvent} and
     * {@link yesman.epicfight.mixin.client.MixinInventory}. Cancelling the mouse
     * scroll event disables cycling for vanilla mouse input, but other input
     * systems (e.g., controllers) still call {@link Inventory#swapPaint}, so we
     * cancel those calls as well. This ensures universal behavior while
     * maximizing compatibility.
     *
     * @return {@code true} if hotbar item cycling should be disabled; {@code false} otherwise.
     * */
    @ApiStatus.Internal
    public static boolean isHotbarCyclingDisabled() {
        final Minecraft minecraft = Minecraft.getInstance();
        final LocalPlayerPatch localPlayerPatch = ClientEngine.getInstance().getPlayerPatch();
        return minecraft.player != null && localPlayerPatch != null && !localPlayerPatch.getEntityState().canSwitchHoldingItem() && minecraft.screen == null;
    }

    /**
     * Checks whether the player is blocked from switching or dropping the held item.
     *
     * @return true if switching or dropping is blocked, false otherwise
     */
    public boolean isSwitchOrDropBlocked() {
        return !this.playerpatch.getEntityState().canSwitchHoldingItem() || this.hotbarLocked;
    }

    public boolean moverToggling() {
		return this.moverPressToggle;
	}
	
	public boolean sneakToggling() {
		return this.sneakPressToggle;
	}
	
	public boolean attackToggling() {
		return this.attackLightPressToggle;
	}
	
	public boolean weaponInnateToggling() {
		return this.weaponInnatePressToggle;
	}
	
	/******************
	 * Event listeners
	 ******************/
	private void epicfight$mouseScrollEvent(InputEvent.MouseScrollingEvent event) {
        // Disables item switching for the vanilla mouse input
        if (isHotbarCyclingDisabled()) {
            event.setCanceled(true);
        }
	}
	
	private void epicfight$moveInputEvent(MovementInputUpdateEvent event) {
		if (this.playerpatch == null) {
			return;
		}
		
		PlayerInputState playerinputstate = this.inputTick(event.getInput());
		MappedMovementInputUpdateEvent wrappedEvent = new MappedMovementInputUpdateEvent(this.playerpatch, playerinputstate);
		this.playerpatch.getPlayerSkills().fireSkillEvents(EpicFightMod.MODID, wrappedEvent);
	}
	
	private void epicfight$clientTickEndEvent(ClientTickEvent.Post event) {
		if (this.player == null) {
			return;
		}
		
		this.packetsToSend.forEach(EpicFightNetworkManager::sendToServer);
		this.packetsToSend.clear();
	}

	private void epicfight$interactionKeyMappingTriggered(InputEvent.InteractionKeyMappingTriggered event) {
        if (this.minecraft.player == null || this.minecraft.hitResult == null) return;

        final EpicFightInputActions triggeredAction = mapKeyMappingToAction(event.getKeyMapping());

        if (triggeredAction == null) {
            // The key mapping corresponds to a fixed vanilla action (attack, pick block, or use item).
            // These are predictable, so it's safe to map the key mapping to an input action.
            return;
        }

		if (
            triggeredAction == EpicFightInputActions.VANILLA_ATTACK_DESTROY &&
            InputManager.isBoundToSamePhysicalInput(EpicFightInputActions.ATTACK, EpicFightInputActions.VANILLA_ATTACK_DESTROY) &&
			this.minecraft.hitResult.getType() == HitResult.Type.BLOCK &&
			ClientConfig.combatPreferredItems.contains(this.player.getMainHandItem().getItem())
		) {
			BlockPos bp = ((BlockHitResult)this.minecraft.hitResult).getBlockPos();
			BlockState bs = this.minecraft.level.getBlockState(bp);
			
			if (!this.player.getMainHandItem().getItem().canAttackBlock(bs, this.player.level(), bp, this.player) || this.player.getMainHandItem().getDestroySpeed(bs) <= 1.0F) {
				event.setSwingHand(false);
				event.setCanceled(true);
			}
		}
		
		if (
			triggeredAction == EpicFightInputActions.USE &&
			InputManager.isBoundToSamePhysicalInput(EpicFightInputActions.USE, EpicFightInputActions.GUARD) &&
			(
				ClientConfig.keyConflictResolveScope.cancelInteraction() ||
				ClientConfig.keyConflictResolveScope.cancelItemUse()
			)
		) {
			MutableBoolean canGuard = new MutableBoolean();
			
			EpicFightCapabilities.getUnparameterizedEntityPatch(this.minecraft.player, LocalPlayerPatch.class).ifPresent(playerpatch -> {
				SkillContainer skillcontainer = playerpatch.getSkill(SkillSlots.GUARD);
				
				if (skillcontainer.getSkill() != null && skillcontainer.getSkill().canExecute(skillcontainer)) {
					canGuard.setValue(true);
				}
			});
			
			if (this.minecraft.hitResult.getType() == HitResult.Type.MISS) {
				if (canGuard.booleanValue() && ClientConfig.keyConflictResolveScope.cancelItemUse()) {
					event.setSwingHand(false);
					event.setCanceled(true);
				}
			} else {
				if (canGuard.booleanValue()) {
					InteractionResult interactionResult = switch (this.minecraft.hitResult.getType()) {
						case ENTITY -> {
							yield ((EntityHitResult)this.minecraft.hitResult).getEntity().interact(this.minecraft.player, event.getHand());
						}
						case BLOCK -> {
							BlockHitResult blockHitResult = ((BlockHitResult)this.minecraft.hitResult);
							BlockPos blockpos = blockHitResult.getBlockPos();
							BlockState blockstate = this.minecraft.level.getBlockState(blockpos);
							FakeLevel fakeLevelForSimulation = FakeLevel.getFakeLevel(this.minecraft.level);
							FakeLevel.FakeClientPlayer fakePlayerForSimulation = FakeLevel.getFakePlayer(this.minecraft.player.getGameProfile());
							
							InteractionResult useItemInteractionResult = blockstate.useItemOn(this.player.getItemInHand(event.getHand()), fakeLevelForSimulation, fakePlayerForSimulation, event.getHand(), blockHitResult).result();
							if (useItemInteractionResult != InteractionResult.PASS) yield useItemInteractionResult;
							yield blockstate.useWithoutItem(fakeLevelForSimulation, fakePlayerForSimulation, blockHitResult);
						}
						default -> throw new IllegalArgumentException();
					};
					
					if (interactionResult != InteractionResult.PASS && ClientConfig.keyConflictResolveScope.cancelInteraction()) {
						event.setSwingHand(false);
						event.setCanceled(true);
					} else if (interactionResult == InteractionResult.PASS && ClientConfig.keyConflictResolveScope.cancelItemUse()) {
						event.setSwingHand(false);
						event.setCanceled(true);
					}
				}
			}
		}
	}
	
	private void epicfight$livingJumpEvent(LivingJumpEvent event) {
		if (event.getEntity() == this.player) {
			this.tickSinceLastJump = 5;
		}
	}
	
	/**********************
	 * Event listeners end
	 **********************/
	@Override
	public void gameEventBus(IEventBus gameEventBus) {
		gameEventBus.addListener(this::epicfight$mouseScrollEvent);
		gameEventBus.addListener(this::epicfight$moveInputEvent);
		gameEventBus.addListener(this::epicfight$clientTickEndEvent);
		gameEventBus.addListener(this::epicfight$interactionKeyMappingTriggered);
		gameEventBus.addListener(this::epicfight$livingJumpEvent);
	}
	
	@Override
	public void modEventBus(IEventBus modEventBus) {
	}
}