package dev.dubhe.anvilcraft.entity;

import com.google.common.collect.ImmutableSet;
import dev.dubhe.anvilcraft.item.HeavyHalberdItem;
import net.minecraft.core.HolderSet;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.syncher.EntityDataAccessor;
import net.minecraft.network.syncher.EntityDataSerializers;
import net.minecraft.network.syncher.SynchedEntityData;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.sounds.SoundEvent;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.util.Mth;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityDimensions;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.MobCategory;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.entity.projectile.AbstractArrow;
import net.minecraft.world.flag.FeatureFlagSet;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.enchantment.EnchantmentHelper;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.EntityHitResult;
import net.minecraft.world.phys.Vec3;
import org.jetbrains.annotations.Nullable;

import java.io.Serializable;
import java.util.function.Predicate;
import java.util.function.ToIntFunction;

public abstract class ThrownHeavyHalberdEntity extends AbstractArrow {
    private static final EntityDataAccessor<Byte> ID_LOYALTY = SynchedEntityData.defineId(
        ThrownHeavyHalberdEntity.class, EntityDataSerializers.BYTE);
    private static final EntityDataAccessor<Boolean> ID_FOIL = SynchedEntityData.defineId(
        ThrownHeavyHalberdEntity.class, EntityDataSerializers.BOOLEAN);
    private boolean dealtDamage;
    public int clientSideReturnHeavyHalberdTickCount;

    @SuppressWarnings("unchecked")
    public ThrownHeavyHalberdEntity(EntityType<? extends Entity> type, Level level) {
        super((EntityType<? extends AbstractArrow>) type, level);
    }

    public ThrownHeavyHalberdEntity(
        EntityType<? extends ThrownHeavyHalberdEntity> type, Level level, LivingEntity shooter, ItemStack pickupItemStack
    ) {
        super(type, shooter, level, pickupItemStack, null);
        this.setBaseDamage(HeavyHalberdItem.getThrownBaseDamage(pickupItemStack));
        this.entityData.set(ID_LOYALTY, this.getLoyaltyFromItem(pickupItemStack));
        this.entityData.set(ID_FOIL, pickupItemStack.hasFoil());
    }

    public ThrownHeavyHalberdEntity(
        EntityType<? extends ThrownHeavyHalberdEntity> type, Level level, double x, double y, double z, ItemStack pickupItemStack
    ) {
        super(type, x, y, z, level, pickupItemStack, pickupItemStack);
        this.setBaseDamage(HeavyHalberdItem.getThrownBaseDamage(pickupItemStack));
        this.entityData.set(ID_LOYALTY, this.getLoyaltyFromItem(pickupItemStack));
        this.entityData.set(ID_FOIL, pickupItemStack.hasFoil());
    }

    public abstract ResourceLocation getTextureBase();

    @Override
    protected void defineSynchedData(SynchedEntityData.Builder builder) {
        super.defineSynchedData(builder);
        builder.define(ID_LOYALTY, (byte) 0);
        builder.define(ID_FOIL, false);
    }

    @Override
    public void tick() {
        if (this.inGroundTime > 4) {
            this.dealtDamage = true;
        }

        Entity entity = this.getOwner();
        int i = this.entityData.get(ID_LOYALTY);
        if (i > 0 && (this.dealtDamage || this.isNoPhysics() || this.getY() <= this.level().getMinBuildHeight()) && entity != null) {
            if (!this.isAcceptableReturnOwner()) {
                if (!this.level().isClientSide && this.pickup == AbstractArrow.Pickup.ALLOWED) {
                    this.spawnAtLocation(this.getPickupItem(), 0.1F);
                }

                this.discard();
            } else {
                this.setNoPhysics(true);
                Vec3 vec3 = entity.getEyePosition().subtract(this.position());
                this.setPosRaw(this.getX(), this.getY() + vec3.y * 0.015 * (double) i, this.getZ());
                if (this.level().isClientSide) {
                    this.yOld = this.getY();
                }

                double d0 = 0.05 * (double) i;
                this.setDeltaMovement(this.getDeltaMovement().scale(0.95).add(vec3.normalize().scale(d0)));
                if (this.clientSideReturnHeavyHalberdTickCount == 0) {
                    this.playSound(SoundEvents.TRIDENT_RETURN, 10.0F, 1.0F);
                }

                this.clientSideReturnHeavyHalberdTickCount++;
            }
        }

        super.tick();
    }

    private boolean isAcceptableReturnOwner() {
        Entity entity = this.getOwner();
        return entity != null && entity.isAlive()
               && (!(entity instanceof ServerPlayer) || !entity.isSpectator());
    }

    public boolean isFoil() {
        return this.entityData.get(ID_FOIL);
    }

    /**
     * Gets the EntityHitResult representing the entity hit
     */
    @Nullable
    @Override
    protected EntityHitResult findHitEntity(Vec3 startVec, Vec3 endVec) {
        return this.dealtDamage ? null : super.findHitEntity(startVec, endVec);
    }

    /**
     * Called when the arrow hits an entity
     */
    @Override
    protected void onHitEntity(EntityHitResult result) {
        Entity victim = result.getEntity();
        double speed = this.getDeltaMovement().length();
        double baseDamage = this.getBaseDamage();
        Entity owner = this.getOwner();
        DamageSource source = this.damageSources().arrow(this, owner != null ? owner : this);
        if (!this.getWeaponItem().isEmpty() && this.level() instanceof ServerLevel serverlevel) {
            baseDamage = EnchantmentHelper.modifyDamage(serverlevel, this.getWeaponItem(), victim, source, (float) baseDamage);
        }

        float damage = Mth.ceil(Mth.clamp(speed * baseDamage, 0.0, 2.147483647E9));

        this.dealtDamage = true;
        if (victim.hurt(source, damage)) {
            if (victim.getType() == EntityType.ENDERMAN) {
                return;
            }

            if (this.level() instanceof ServerLevel level) {
                EnchantmentHelper.doPostAttackEffectsWithItemSource(level, victim, source, this.getWeaponItem());
            }

            if (victim instanceof LivingEntity livingentity) {
                this.doKnockback(livingentity, source);
                this.doPostHurtEffects(livingentity);
            }
        }

        this.setDeltaMovement(this.getDeltaMovement().multiply(-0.01, -0.1, -0.01));
        this.playSound(SoundEvents.TRIDENT_HIT, 1.0F, 1.0F);
    }

    @Override
    protected void hitBlockEnchantmentEffects(ServerLevel level, BlockHitResult hitResult, ItemStack stack) {
        Vec3 vec3 = hitResult.getBlockPos().clampLocationWithin(hitResult.getLocation());
        EnchantmentHelper.onHitBlock(
            level,
            stack,
            this.getOwner() instanceof LivingEntity livingentity ? livingentity : null,
            this,
            null,
            vec3,
            level.getBlockState(hitResult.getBlockPos()),
            item -> this.kill()
        );
    }

    @Override
    public ItemStack getWeaponItem() {
        return this.getPickupItemStackOrigin();
    }

    @Override
    protected boolean tryPickup(Player player) {
        return super.tryPickup(player) || this.isNoPhysics() && this.ownedBy(player) && player.getInventory().add(this.getPickupItem());
    }

    @Override
    protected SoundEvent getDefaultHitGroundSoundEvent() {
        return SoundEvents.TRIDENT_HIT_GROUND;
    }

    /**
     * Called by a player entity when they collide with an entity
     */
    @Override
    public void playerTouch(Player entity) {
        if (this.ownedBy(entity) || this.getOwner() == null) {
            super.playerTouch(entity);
        }
    }

    /**
     * (abstract) Protected helper method to read subclass entity data from NBT.
     */
    @Override
    public void readAdditionalSaveData(CompoundTag compound) {
        super.readAdditionalSaveData(compound);
        this.dealtDamage = compound.getBoolean("DealtDamage");
        this.entityData.set(ID_LOYALTY, this.getLoyaltyFromItem(this.getPickupItemStackOrigin()));
    }

    @Override
    public void addAdditionalSaveData(CompoundTag compound) {
        super.addAdditionalSaveData(compound);
        compound.putBoolean("DealtDamage", this.dealtDamage);
    }

    private byte getLoyaltyFromItem(ItemStack stack) {
        return this.level() instanceof ServerLevel serverlevel
               ? (byte) Mth.clamp(EnchantmentHelper.getTridentReturnToOwnerAcceleration(serverlevel, stack, this), 0, 127)
               : 0;
    }

    @Override
    public void tickDespawn() {
        int i = this.entityData.get(ID_LOYALTY);
        if (this.pickup != AbstractArrow.Pickup.ALLOWED || i <= 0) {
            super.tickDespawn();
        }
    }

    @Override
    protected float getWaterInertia() {
        return 0.99F;
    }

    @Override
    public boolean shouldRender(double x, double y, double z) {
        return true;
    }

    public interface Factory<T extends ThrownHeavyHalberdEntity> extends EntityType.EntityFactory<T>, Serializable {
    }

    public static class HeavyHalberdType<T extends ThrownHeavyHalberdEntity> extends EntityType<T> {
        public HeavyHalberdType(
            EntityFactory<T> factory,
            MobCategory category,
            boolean serialize,
            boolean summon,
            boolean fireImmune,
            boolean canSpawnFarFromPlayer,
            ImmutableSet<Block> immuneTo,
            EntityDimensions dimensions,
            float spawnDimensionsScale,
            int clientTrackingRange,
            int updateInterval,
            FeatureFlagSet requiredFeatures,
            Predicate<EntityType<?>> trackDeltasSupplier,
            ToIntFunction<EntityType<?>> trackingRangeSupplier,
            ToIntFunction<EntityType<?>> updateIntervalSupplier
        ) {
            super(
                factory,
                category,
                serialize,
                summon,
                fireImmune,
                canSpawnFarFromPlayer,
                immuneTo,
                dimensions,
                spawnDimensionsScale,
                clientTrackingRange,
                updateInterval,
                requiredFeatures,
                trackDeltasSupplier,
                trackingRangeSupplier,
                updateIntervalSupplier
            );
        }

        @Override
        public boolean is(HolderSet<EntityType<?>> entityType) {
            return super.is(entityType)
                   || (entityType instanceof HolderSet.Direct<EntityType<?>> direct
                       && direct.size() == 1
                       && direct.get(0).is(ResourceLocation.withDefaultNamespace("trident")));
        }
    }
}
