package dev.dubhe.anvilcraft.recipe.anvil.wrap;

import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.anvilcraft.lib.recipe.component.ChanceItemStack;
import dev.anvilcraft.lib.recipe.component.ItemIngredientPredicate;
import dev.dubhe.anvilcraft.init.reicpe.ModRecipeTypes;
import dev.dubhe.anvilcraft.recipe.anvil.predicate.block.HasCauldron;
import dev.dubhe.anvilcraft.recipe.anvil.util.WrapUtils;
import dev.dubhe.anvilcraft.recipe.component.HasCauldronSimple;
import net.minecraft.core.Vec3i;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.codec.ByteBufCodecs;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.item.crafting.RecipeSerializer;
import net.minecraft.world.item.crafting.RecipeType;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.material.Fluids;
import net.minecraft.world.phys.Vec3;
import java.util.List;

/**
 * 膨发配方类
 *
 * <p>该配方用于在铁砧下落时使物品在炼药锅中膨发，需要在铁砧下方放置炼药锅作为触发条件</p>
 */
public class BulgingRecipe extends AbstractProcessRecipe<BulgingRecipe> {
    /**
     * 构造一个膨发配方
     *
     * @param itemIngredients 物品原料列表
     * @param results         结果物品列表
     * @param hasCauldron     炼药锅条件
     */
    public BulgingRecipe(List<ItemIngredientPredicate> itemIngredients, List<ChanceItemStack> results, HasCauldronSimple hasCauldron) {
        super(new Property().setItemInputOffset(new Vec3(0.0, -0.375, 0.0)).setItemInputRange(new Vec3(0.75, 0.75, 0.75)).setInputItems(itemIngredients).setItemOutputOffset(new Vec3(0.0, -0.75, 0.0)).setResultItems(results).setCauldronOffset(new Vec3i(0, -1, 0)).setHasCauldron(hasCauldron));
    }

    @Override
    public RecipeSerializer<BulgingRecipe> getSerializer() {
        return ModRecipeTypes.BULGING_SERIALIZER.get();
    }

    @Override
    public RecipeType<BulgingRecipe> getType() {
        return ModRecipeTypes.BULGING_TYPE.get();
    }

    /**
     * 创建一个构建器实例
     *
     * @return 构建器实例
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * 是否消耗流体
     *
     * @return 如果消耗流体返回true，否则返回false
     */
    public boolean isConsumeFluid() {
        HasCauldronSimple hasCauldron = this.getHasCauldron();
        return HasCauldron.isNotEmpty(hasCauldron.fluid()) && this.getHasCauldron().consume() > 0;
    }

    /**
     * 是否产生流体
     *
     * @return 如果产生流体返回true，否则返回false
     */
    public boolean isProduceFluid() {
        HasCauldronSimple hasCauldron = this.getHasCauldron();
        return HasCauldron.isNotEmpty(hasCauldron.transform()) && this.getHasCauldron().consume() < 0;
    }

    /**
     * 是否使用水作为流体
     *
     * @return 如果使用水返回true，否则返回false
     */
    public boolean isFromWater() {
        return this.getHasCauldron().fluid().equals(BuiltInRegistries.FLUID.getKey(Fluids.WATER));
    }


    /**
     * 膨发配方序列化器
     */
    public static class Serializer implements RecipeSerializer<BulgingRecipe> {
        /**
         * 编解码器
         */
        public static final MapCodec<BulgingRecipe> CODEC = RecordCodecBuilder.mapCodec(instance -> instance.group(ItemIngredientPredicate.CODEC.listOf().fieldOf("ingredients").forGetter(BulgingRecipe::getInputItems), ChanceItemStack.CODEC.listOf().fieldOf("results").forGetter(BulgingRecipe::getResultItems), HasCauldronSimple.CODEC.forGetter(BulgingRecipe::getHasCauldron)).apply(instance, BulgingRecipe::new));
        /**
         * 流编解码器
         */
        public static final StreamCodec<RegistryFriendlyByteBuf, BulgingRecipe> STREAM_CODEC = StreamCodec.composite(ItemIngredientPredicate.STREAM_CODEC.apply(ByteBufCodecs.list()), BulgingRecipe::getInputItems, ChanceItemStack.STREAM_CODEC.apply(ByteBufCodecs.list()), BulgingRecipe::getResultItems, HasCauldronSimple.STREAM_CODEC, BulgingRecipe::getHasCauldron, BulgingRecipe::new);

        @Override
        public MapCodec<BulgingRecipe> codec() {
            return Serializer.CODEC;
        }

        @Override
        public StreamCodec<RegistryFriendlyByteBuf, BulgingRecipe> streamCodec() {
            return Serializer.STREAM_CODEC;
        }
    }


    /**
     * 膨发配方构建器
     */
    public static class Builder extends SimpleAbstractBuilder<BulgingRecipe, Builder> {
        /**
         * 炼药锅条件构建器
         */
        private final HasCauldronSimple.Builder hasCauldron = HasCauldronSimple.empty();

        /**
         * 设置炼药锅流体
         *
         * @param fluid 流体ID
         * @return 构建器实例
         */
        public Builder cauldron(ResourceLocation fluid) {
            this.hasCauldron.fluid(fluid);
            return this;
        }

        /**
         * 设置炼药锅方块
         *
         * @param cauldron 炼药锅方块
         * @return 构建器实例
         */
        public Builder cauldron(Block cauldron) {
            this.cauldron(WrapUtils.cauldron2Fluid(cauldron));
            return this;
        }

        /**
         * 设置转换后的流体
         *
         * @param transform 转换后的流体ID
         * @return 构建器实例
         */
        public Builder transform(ResourceLocation transform) {
            this.hasCauldron.transform(transform);
            return this;
        }

        /**
         * 设置转换后的炼药锅方块
         *
         * @param transform 转换后的炼药锅方块
         * @return 构建器实例
         */
        public Builder transform(Block transform) {
            this.hasCauldron.transform(WrapUtils.cauldron2Fluid(transform));
            return this;
        }

        /**
         * 设置是否产生流体
         *
         * @param produce 是否产生流体
         * @return 构建器实例
         */
        public Builder produce(int produce) {
            if (produce <= 0) return this;
            this.hasCauldron.consume(-produce);
            return this;
        }

        /**
         * 设置是否消耗流体
         *
         * @param consume 消耗流体
         * @return 构建器实例
         */
        public Builder consume(int consume) {
            if (consume <= 0) return this;
            this.hasCauldron.consume(consume);
            return this;
        }

        @Override
        protected BulgingRecipe of(List<ItemIngredientPredicate> itemIngredients, List<ChanceItemStack> results) {
            return new BulgingRecipe(itemIngredients, results, this.hasCauldron.build());
        }

        @Override
        public void validate(ResourceLocation id) {
        }

        @Override
        public String getType() {
            return "bulging";
        }

        @Override
        protected Builder getThis() {
            return this;
        }
    }
}
