package dev.dubhe.anvilcraft.recipe.transform;

import com.google.common.collect.Lists;
import com.mojang.brigadier.StringReader;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.anvilcraft.lib.util.CodecUtil;
import dev.dubhe.anvilcraft.AnvilCraft;
import net.minecraft.commands.arguments.NbtPathArgument;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.nbt.SnbtPrinterTagVisitor;
import net.minecraft.nbt.Tag;
import net.minecraft.nbt.TagParser;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.codec.ByteBufCodecs;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.util.StringRepresentable;

import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

/**
 * 对生成出来的生物进行nbt修改
 */
public record TagModification(String path, ModifyOperation op, int index, Tag tag) implements Consumer<Tag> {
    public static final Codec<TagModification> CODEC = RecordCodecBuilder.create(ins -> ins.group(
            Codec.STRING.fieldOf("path").forGetter(o -> o.path),
            ModifyOperation.CODEC.fieldOf("op").forGetter(o -> o.op),
            Codec.INT
                .optionalFieldOf("index")
                .forGetter(o -> o.index < 0 ? Optional.empty() : Optional.of(o.index)),
            Codec.STRING.fieldOf("tag").forGetter(o -> {
                SnbtPrinterTagVisitor visitor = new SnbtPrinterTagVisitor("", 0, Lists.newArrayList());
                return visitor.visit(o.tag);
            })
    ).apply(ins, TagModification::create));
    public static final StreamCodec<RegistryFriendlyByteBuf, TagModification> STREAM_CODEC = StreamCodec.composite(
        ByteBufCodecs.STRING_UTF8,
        TagModification::path,
        ModifyOperation.STREAM_CODEC,
        TagModification::op,
        ByteBufCodecs.INT,
        TagModification::index,
        ByteBufCodecs.TAG,
        TagModification::tag,
        TagModification::new
    );

    /**
     * 初始化 TagModification
     *
     * @param tag snbt表示的nbt标签
     */
    @SuppressWarnings("OptionalUsedAsFieldOrParameterType")
    private static TagModification create(String path, ModifyOperation op, Optional<Integer> index, String tag) {
        try {
            StringReader reader = new StringReader(tag);
            TagParser parser = new TagParser(reader);
            Tag parseTag = parser.readValue();
            return new TagModification(path, op, index.orElse(0), parseTag);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 初始化 TagModification
     */
    public TagModification {
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public void accept(Tag input) {
        if (op == ModifyOperation.SET || op == ModifyOperation.ROOT_SET) {
            int index = this.path.lastIndexOf('.');
            String path = this.path.substring(0, index == -1 ? this.path.length() : index);
            String key = this.path.substring(index + 1);
            this.readAndAcceptTag(path, 0, key);
            return;
        }
        this.readAndAcceptTag(this.path, this.index, path);
    }

    public void readAndAcceptTag(String path, int index, String key) {
        try {
            StringReader reader = new StringReader(path);
            NbtPathArgument argument = new NbtPathArgument();
            NbtPathArgument.NbtPath nbtPath = argument.parse(reader);
            List<Tag> contract = nbtPath.get(this.tag);
            if (contract.size() >= 2) {
                throw new IllegalArgumentException("TagModification does not allow multiple tag at path: " + path);
            }
            if (contract.isEmpty()) return;
            Tag value = contract.getFirst();
            this.op.accept(value, this.tag, index, key);
        } catch (CommandSyntaxException e) {
            AnvilCraft.LOGGER.error(e.getMessage(), e);
        }
    }

    public enum ModifyOperation implements StringRepresentable {
        SET {
            @Override
            public void accept(Tag inputSrc, Tag tag, int index, String key) {
                if (inputSrc instanceof CompoundTag compoundTag) {
                    compoundTag.put(key, tag);
                } else {
                    throw new RuntimeException("Expected CompoundTag, got " + inputSrc.getAsString());
                }
            }
        },
        APPEND {
            @Override
            public void accept(Tag inputSrc, Tag tag, int index, String key) {
                if (inputSrc instanceof ListTag listTag) {
                    listTag.add(tag);
                } else {
                    throw new RuntimeException("Expected list, got " + inputSrc.getAsString());
                }
            }
        },
        INSERT {
            @Override
            public void accept(Tag inputSrc, Tag tag, int index, String key) {
                if (inputSrc instanceof ListTag listTag) {
                    listTag.add(index, tag);
                } else {
                    throw new RuntimeException("Expected list, got " + inputSrc.getAsString());
                }
            }
        },
        MERGE {
            @Override
            public void accept(Tag inputSrc, Tag tag, int index, String key) {
                if (inputSrc instanceof ListTag listTag && tag instanceof ListTag tag2) {
                    listTag.addAll(tag2);
                } else {
                    throw new RuntimeException(
                        "Expected list, got " + inputSrc.getAsString() + ", " + tag.getAsString());
                }
            }
        },
        PREPEND {
            @Override
            public void accept(Tag inputSrc, Tag tag, int index, String key) {
                if (inputSrc instanceof ListTag listTag) {
                    listTag.add(0, tag);
                } else {
                    throw new RuntimeException("Expected list, got " + inputSrc.getAsString());
                }
            }
        },
        ROOT_SET {
            @Override
            public void accept(Tag inputSrc, Tag tag, int index, String key) {
                if (inputSrc instanceof CompoundTag src && tag instanceof CompoundTag target) {
                    src.merge(target);
                } else {
                    throw new RuntimeException("Expected Compound Tag, got " + inputSrc.getAsString() + " and " + tag.getAsString()
                    );
                }
            }
        };

        public static final Codec<ModifyOperation> CODEC = StringRepresentable.fromEnum(ModifyOperation::values);
        public static final StreamCodec<RegistryFriendlyByteBuf, ModifyOperation> STREAM_CODEC = CodecUtil.enumStreamCodec(
            ModifyOperation.class
        );

        public abstract void accept(Tag inputSrc, Tag tag, int index, String key);

        @Override
        public String getSerializedName() {
            return name();
        }
    }

    public static class Builder {
        private String path = "";
        private ModifyOperation op;
        private int index = -1;
        private Tag tag;

        Builder() {
        }

        public Builder path(String tagKeyPath) {
            this.path = tagKeyPath;
            return this;
        }

        public Builder operation(ModifyOperation fn) {
            this.op = fn;
            return this;
        }

        public Builder index(int value) {
            this.index = value;
            return this;
        }

        public Builder value(Tag tag) {
            this.tag = tag;
            return this;
        }

        public TagModification build() {
            return new TagModification(path, op, index, tag);
        }
    }
}
