package dev.dubhe.anvilcraft.api.rendering;

import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.VertexBuffer;
import com.mojang.blaze3d.vertex.VertexFormat;
import net.minecraft.client.renderer.RenderType;
import org.lwjgl.opengl.GL15;
import org.lwjgl.opengl.GL15C;
import org.lwjgl.system.MemoryUtil;

final class CompileResult {
    private static final MemoryUtil.MemoryAllocator ALLOCATOR = MemoryUtil.getAllocator(false);
    private final RenderType renderType;
    private final int vertexCount;
    private final int vertexSize;
    private final long vertexBufferPtr;
    final int indexCount;
    private boolean freed = false;

    CompileResult(RenderType renderType, int vertexCount, int vertexSize, long vertexBufferPtr, int indexCount) {
        this.renderType = renderType;
        this.vertexCount = vertexCount;
        this.vertexSize = vertexSize;
        this.vertexBufferPtr = vertexBufferPtr;
        this.indexCount = indexCount;
    }

    void upload(VertexBuffer vertexBuffer) {
        if (freed) return;
        VertexFormat.Mode mode = renderType.mode;
        vertexBuffer.bind();
        if (vertexBuffer.format != null) {
            vertexBuffer.format.clearBufferState();
        }
        GL15.glBindBuffer(GL15.GL_ARRAY_BUFFER, vertexBuffer.vertexBufferId);
        renderType.format.setupBufferState();
        vertexBuffer.format = renderType.format;
        GL15C.nglBufferData(GL15.GL_ARRAY_BUFFER, (long) vertexCount * vertexSize, vertexBufferPtr, GL15.GL_STATIC_DRAW);
        RenderSystem.AutoStorageIndexBuffer indexBuffer = RenderSystem.getSequentialBuffer(mode);
        if (indexBuffer != vertexBuffer.sequentialIndices || !indexBuffer.hasStorage(indexCount)) {
            indexBuffer.bind(indexCount);
        }
        vertexBuffer.sequentialIndices = indexBuffer;
        VertexBuffer.unbind();
    }

    void free() {
        if (freed) return;
        ALLOCATOR.free(vertexBufferPtr);
        freed = true;
    }

    @Override
    public boolean equals(final Object o) {
        if (o == this) return true;
        if (!(o instanceof CompileResult)) return false;
        final CompileResult other = (CompileResult) o;
        if (this.vertexCount != other.vertexCount) return false;
        if (this.vertexSize != other.vertexSize) return false;
        if (this.vertexBufferPtr != other.vertexBufferPtr) return false;
        if (this.indexCount != other.indexCount) return false;
        if (this.freed != other.freed) return false;
        final Object this$renderType = this.renderType;
        final Object other$renderType = other.renderType;
        if (this$renderType == null ? other$renderType != null : !this$renderType.equals(other$renderType)) return false;
        return true;
    }

    @Override
    public int hashCode() {
        final int PRIME = 59;
        int result = 1;
        result = result * PRIME + this.vertexCount;
        result = result * PRIME + this.vertexSize;
        final long $vertexBufferPtr = this.vertexBufferPtr;
        result = result * PRIME + (int) ($vertexBufferPtr >>> 32 ^ $vertexBufferPtr);
        result = result * PRIME + this.indexCount;
        result = result * PRIME + (this.freed ? 79 : 97);
        final Object $renderType = this.renderType;
        result = result * PRIME + ($renderType == null ? 43 : $renderType.hashCode());
        return result;
    }
}
