package dev.dubhe.anvilcraft.recipe;

import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.anvilcraft.lib.util.CodecUtil;
import dev.dubhe.anvilcraft.init.reicpe.ModRecipeTypes;
import dev.dubhe.anvilcraft.recipe.anvil.builder.AbstractRecipeBuilder;
import dev.dubhe.anvilcraft.recipe.anvil.input.IItemsInput;
import dev.dubhe.anvilcraft.util.RecipeUtil;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import net.minecraft.core.HolderLookup;
import net.minecraft.core.NonNullList;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.tags.TagKey;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.crafting.Ingredient;
import net.minecraft.world.item.crafting.Recipe;
import net.minecraft.world.item.crafting.RecipeInput;
import net.minecraft.world.item.crafting.RecipeSerializer;
import net.minecraft.world.item.crafting.RecipeType;
import net.minecraft.world.level.ItemLike;
import net.minecraft.world.level.Level;
import net.neoforged.neoforge.common.conditions.ICondition;
import org.jetbrains.annotations.Contract;
import java.util.ArrayList;
import java.util.List;

public class JewelCraftingRecipe implements Recipe<JewelCraftingRecipe.Input> {
    public final List<ICondition> conditions;
    public final NonNullList<Ingredient> ingredients;
    public final ItemStack result;
    public final List<Object2IntMap.Entry<Ingredient>> mergedIngredients;
    public Input cache;
    public int cacheTimes;

    public JewelCraftingRecipe(List<ICondition> conditions, NonNullList<Ingredient> ingredients, ItemStack result) {
        this.conditions = conditions;
        this.ingredients = ingredients;
        this.result = result;
        this.mergedIngredients = RecipeUtil.mergeIngredient(ingredients);
        if (mergedIngredients.size() > 4) {
            throw new IllegalArgumentException("Too many different ingredients");
        }
    }

    @Contract(" -> new")
    public static Builder builder() {
        return new Builder();
    }

    @Override
    public RecipeType<?> getType() {
        return ModRecipeTypes.JEWEL_CRAFTING_TYPE.get();
    }

    @Override
    public RecipeSerializer<?> getSerializer() {
        return ModRecipeTypes.JEWEL_CRAFTING_SERIALIZER.get();
    }

    @Override
    public boolean canCraftInDimensions(int width, int height) {
        return true;
    }

    @Override
    public ItemStack getResultItem(HolderLookup.Provider registries) {
        return result;
    }

    @Override
    public ItemStack assemble(Input input, HolderLookup.Provider registries) {
        return result.copy();
    }

    @Override
    public boolean matches(Input input, Level level) {
        if (input == cache) {
            return cacheTimes >= 1;
        }
        int times = RecipeUtil.getMaxCraftTime(input, ingredients);
        cache = input;
        cacheTimes = times;
        return cacheTimes >= 1;
    }

    @Override
    public boolean isSpecial() {
        return true;
    }


    public record Input(ItemStack source, List<ItemStack> items) implements RecipeInput, IItemsInput {
        @Override
        public ItemStack getItem(int index) {
            return items.get(index);
        }

        @Override
        public int size() {
            return items.size();
        }
    }


    public static class Serializer implements RecipeSerializer<JewelCraftingRecipe> {
        private static final MapCodec<JewelCraftingRecipe> CODEC = RecordCodecBuilder.mapCodec(ins -> ins.group(ICondition.LIST_CODEC.optionalFieldOf("neoforge:conditions", new ArrayList<>()).forGetter(JewelCraftingRecipe::getConditions), CodecUtil.createIngredientListCodec("ingredients", 256, "jewel_crafting").forGetter(JewelCraftingRecipe::getIngredients), ItemStack.CODEC.fieldOf("result").forGetter(JewelCraftingRecipe::getResult)).apply(ins, JewelCraftingRecipe::new));
        public static final StreamCodec<RegistryFriendlyByteBuf, JewelCraftingRecipe> STREAM_CODEC = StreamCodec.of(Serializer::encode, Serializer::decode);

        @Override
        public MapCodec<JewelCraftingRecipe> codec() {
            return CODEC;
        }

        @Override
        public StreamCodec<RegistryFriendlyByteBuf, JewelCraftingRecipe> streamCodec() {
            return STREAM_CODEC;
        }

        private static void encode(RegistryFriendlyByteBuf buf, JewelCraftingRecipe recipe) {
            buf.writeVarInt(recipe.ingredients.size());
            for (Ingredient ingredient : recipe.ingredients) {
                Ingredient.CONTENTS_STREAM_CODEC.encode(buf, ingredient);
            }
            ItemStack.STREAM_CODEC.encode(buf, recipe.result);
        }

        private static JewelCraftingRecipe decode(RegistryFriendlyByteBuf buf) {
            int size = buf.readVarInt();
            NonNullList<Ingredient> ingredients = NonNullList.withSize(size, Ingredient.EMPTY);
            ingredients.replaceAll(i -> Ingredient.CONTENTS_STREAM_CODEC.decode(buf));
            ItemStack result = ItemStack.STREAM_CODEC.decode(buf);
            return new JewelCraftingRecipe(new ArrayList<>(), ingredients, result);
        }
    }


    public static class Builder extends AbstractRecipeBuilder<JewelCraftingRecipe> {
        private List<ICondition> conditions = new ArrayList<>();
        private NonNullList<Ingredient> ingredients = NonNullList.create();
        private ItemStack result = ItemStack.EMPTY;

        public Builder withCondition(ICondition condition) {
            this.conditions.add(condition);
            return this;
        }

        public Builder requires(Ingredient ingredient, int count) {
            for (int i = 0; i < count; i++) {
                this.ingredients.add(ingredient);
            }
            return this;
        }

        public Builder requires(Ingredient ingredient) {
            return requires(ingredient, 1);
        }

        public Builder requires(ItemLike item, int count) {
            return requires(Ingredient.of(item), count);
        }

        public Builder requires(ItemLike item) {
            return requires(item, 1);
        }

        public Builder requires(TagKey<Item> tag, int count) {
            return requires(Ingredient.of(tag), count);
        }

        public Builder requires(TagKey<Item> tag) {
            return requires(tag, 1);
        }

        @Override
        public JewelCraftingRecipe buildRecipe() {
            return new JewelCraftingRecipe(conditions, ingredients, result);
        }

        @Override
        public void validate(ResourceLocation id) {
            if (ingredients.isEmpty() || ingredients.size() > 256) {
                throw new IllegalArgumentException("Recipe ingredients size must in 0-256, RecipeId: " + id);
            }
            if (result.isEmpty()) {
                throw new IllegalArgumentException("Recipe result must not be empty, RecipeId: " + id);
            }
        }

        @Override
        public String getType() {
            return "jewel_crafting";
        }

        @Override
        public Item getResult() {
            return result.getItem();
        }

        /**
         * @return {@code this}.
         */
        public JewelCraftingRecipe.Builder conditions(final List<ICondition> conditions) {
            this.conditions = conditions;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public JewelCraftingRecipe.Builder ingredients(final NonNullList<Ingredient> ingredients) {
            this.ingredients = ingredients;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public JewelCraftingRecipe.Builder result(final ItemStack result) {
            this.result = result;
            return this;
        }
    }

    public List<ICondition> getConditions() {
        return this.conditions;
    }

    public NonNullList<Ingredient> getIngredients() {
        return this.ingredients;
    }

    public ItemStack getResult() {
        return this.result;
    }

    public List<Object2IntMap.Entry<Ingredient>> getMergedIngredients() {
        return this.mergedIngredients;
    }

    public Input getCache() {
        return this.cache;
    }

    public int getCacheTimes() {
        return this.cacheTimes;
    }
}
