package dev.dubhe.anvilcraft.util;

import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.ItemInteractionResult;
import net.neoforged.fml.ModList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collector;
import java.util.stream.Collectors;

public class Util {
    public static final Lazy<Boolean> jadePresent = new Lazy<>(() -> isLoaded("jade") || isLoaded("wthit"));
    private static final StackWalker STACK_WALKER = StackWalker.getInstance(StackWalker.Option.RETAIN_CLASS_REFERENCE);
    public static final Direction[] HORIZONTAL_DIRECTIONS = new Direction[] {Direction.SOUTH, Direction.WEST, Direction.EAST, Direction.NORTH};
    public static final Direction[] VERTICAL_DIRECTIONS = new Direction[] {Direction.UP, Direction.DOWN};
    public static final Direction[][] CORNER_DIRECTIONS = new Direction[][] {{Direction.EAST, Direction.NORTH}, {Direction.EAST, Direction.SOUTH}, {Direction.WEST, Direction.NORTH}, {Direction.WEST, Direction.SOUTH}};

    /**
     * 判断给定的 {@code modId} 对应的模组是否加载
     *
     * @return 模组是否加载
     */
    public static boolean isLoaded(String modId) {
        return ModList.get().isLoaded(modId);
    }

    public static Function<InteractionResult, ItemInteractionResult> interactionResultConverter() {
        return it -> switch (it) {
            case SUCCESS, SUCCESS_NO_ITEM_USED -> ItemInteractionResult.SUCCESS;
            case CONSUME -> ItemInteractionResult.CONSUME;
            case CONSUME_PARTIAL -> ItemInteractionResult.CONSUME_PARTIAL;
            case PASS -> ItemInteractionResult.PASS_TO_DEFAULT_BLOCK_INTERACTION;
            case FAIL -> ItemInteractionResult.FAIL;
        };
    }

    public static <E> Optional<List<E>> intoOptional(List<E> collection) {
        if (collection.isEmpty()) return Optional.empty();
        return Optional.of(collection);
    }

    @NotNull
    public static String generateUniqueRecipeSuffix() {
        return "_generated_" + generateRandomString(8, true, false);
    }

    @NotNull
    public static String generateRandomString(int len) {
        return generateRandomString(len, true, true);
    }

    @NotNull
    public static String generateRandomString(int len, boolean hasInteger, boolean hasUpperLetter) {
        String ch = "abcdefghijklmnopqrstuvwxyz" + (hasUpperLetter ? "ABCDEFGHIGKLMNOPQRSTUVWXYZ" : "") + (hasInteger ? "0123456789" : "");
        StringBuilder stringBuffer = new StringBuilder();
        for (int i = 0; i < len; i++) {
            Random random = new Random(System.nanoTime());
            int num = random.nextInt(ch.length() - 1);
            stringBuffer.append(ch.charAt(num));
        }
        return stringBuffer.toString();
    }

    public static int comparingIntReversed(int x, int y) {
        return Integer.compare(y, x);
    }

    public static boolean findCaller(String caller) {
        return STACK_WALKER.walk(it -> it.anyMatch(frame -> frame.getMethodName().equals(caller)));
    }

    public static <K, V> Collector<Map.Entry<K, V>, ?, Map<K, V>> toMapCollector() {
        return Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue);
    }

    public static void acceptDirections(BlockPos blockPos, Consumer<BlockPos> blockPosConsumer) {
        for (Direction direction : Direction.values()) {
            blockPosConsumer.accept(blockPos.relative(direction));
        }
        for (Direction horizontal : HORIZONTAL_DIRECTIONS) {
            for (Direction vertical : VERTICAL_DIRECTIONS) {
                blockPosConsumer.accept(blockPos.relative(horizontal).relative(vertical));
            }
        }
        for (Direction[] corner : CORNER_DIRECTIONS) {
            BlockPos pos1 = blockPos;
            for (Direction direction : corner) {
                pos1 = pos1.relative(direction);
            }
            for (Direction verticalDirection : VERTICAL_DIRECTIONS) {
                pos1 = pos1.relative(verticalDirection);
                blockPosConsumer.accept(pos1);
            }
        }
    }

    public static void acceptHorizontalDirections(BlockPos blockPos, Consumer<BlockPos> blockPosConsumer) {
        for (Direction direction : HORIZONTAL_DIRECTIONS) {
            blockPosConsumer.accept(blockPos.relative(direction));
        }
    }

    public static <T> boolean isEqualCollection(Collection<T> list1, Collection<T> list2) {
        if (list1.size() != list2.size()) {
            return false;
        }
        Map<T, Integer> countMap = new HashMap<>();
        for (T obj : list1) {
            countMap.put(obj, countMap.getOrDefault(obj, 0) + 1);
        }
        for (T obj : list2) {
            Integer count = countMap.get(obj);
            if (count == null) {
                return false;
            }
            int newCount = count - 1;
            if (newCount < 0) {
                return false;
            }
            countMap.put(obj, newCount);
        }
        return true;
    }

    public static <T> boolean anyMatch(Collection<T> collection, Predicate<T> matcher) {
        for (T t : collection) {
            if (matcher.test(t)) return true;
        }
        return false;
    }

    /**
     * 将传入的值强转为{@code T}类型
     *
     * @param <T> 想要转为的类型
     * @param o   一个值
     * @return 传入的值，但是类型为{@code T}
     * @throws ClassCastException 当无法将传入的值强转时抛出
     */
    @SuppressWarnings("unchecked")
    public static <T> T cast(@NotNull Object o) {
        return (T) o;
    }

    /**
     * 若传入的值可被强转为{@code T}类型，则返回包含传入的值的{@link Optional}
     *
     * @param <T> 想要转为的类型
     * @param o   一个值，可为null
     * @return 一个可能包含传入的值的{@link Optional}
     */
    public static <T> Optional<T> castSafely(@Nullable Object o, Class<T> clazz) {
        return Optional.ofNullable(o).filter(clazz::isInstance).map(Util::cast);
    }

    /**
     * 若传入的值可被强转为传入的任意类型，则返回true
     *
     * @param o 一个值，可为null
     * @return 传入的值，但是类型为{@code T}
     */
    @SuppressWarnings("TypeParameterExplicitlyExtendsObject")
    @SafeVarargs
    public static boolean instanceOfAny(@Nullable Object o, Class<? extends Object>... classes) {
        Optional<Object> op = Optional.empty();
        for (Class<?> clazz : classes) {
            op = op.or(() -> Util.castSafely(o, clazz));
        }
        return op.isPresent();
    }

    /**
     * 若传入的值可被强转为{@code T}类型，则使用传入的值执行传入的方法<br>
     * 等效于{@code Util.castSafely(o, clazz).ifPresent(action);}
     *
     * @param <T>    想要转为的类型
     * @param o      一个值，可为null
     * @param action 将要执行的操作
     */
    public static <T> void ifCastable(@Nullable Object o, Class<T> clazz, Consumer<T> action) {
        Optional.ofNullable(o).filter(clazz::isInstance).<T>map(Util::cast).ifPresent(action);
    }

    private Util() {
    }
}
